(* Badguy.ml: implementation of badguy behaviour and rendering. *)

open Collision

type kind = Guard    (* so far just one type of badguy *)

type t = {
  kind: kind;
  hull: Collision.hull;
  mutable gait: float;
  mutable vx: float;
  mutable vy: float;
  mutable vz: float;
  mutable heading: float;
  mutable triedjump: bool;

  (* initial position in level *)
  initx: float;
  inity: float;
  initz: float;
  initheading:float;
}


(* create a new badguy with the given fields *)
let newbadguy kind initx inity initz initheading =
  { kind = kind;
    hull = {x = initx; y = inity; z = initz; lx = 0.4;ly = 0.499;lz = 0.4};
    gait = 0.0;
    vx = 0.0;
    vy = 0.0;
    vz = 0.0;
    heading = initheading;
    triedjump = false;
    initx = initx;
    inity = inity;
    initz = initz;
    initheading = initheading
  }


(* initialize stuff before rendering *)

type displaylists = NotInitialized | DisplayLists of GlList.base

(* we use display lists for speed *)
let plainlists = ref NotInitialized
and texlists = ref NotInitialized
and plaingreylists = ref NotInitialized
and texgreylists = ref NotInitialized


(* some helper functions for the modelling *)

(* return normalized difference between two vectors *)
let normdiff (ax,ay,az) (bx,by,bz) =
  let dx = ax-.bx and dy = ay-.by and dz = az-.bz in
  let norm = sqrt(dx*.dx +. dy*.dy +. dz*.dz) in
  (dx/.norm, dy/.norm, dz/.norm)

(* Given two triples of floats, return two o.n. vectors (also triples) that
 * are orthogonal to the difference. *)
let orthodiff a b =
  let (dx,dy,dz) = normdiff a b in
  (* find first o.n. vector based on whether or not dz is zero *)
  let (qx1,qy1,qz1) =
    if dz = 0.0 then
      (0.0,0.0,1.0)
    else begin
      let nxy = sqrt(dx*.dx +. dy*.dy) in
      (dy/.nxy, -.dx/.nxy, 0.0)
    end in
  (* find other o.n. vector by crossing the first two *)
  let (qx2,qy2,qz2) = (dy*.qz1-.dz*.qy1, dz*.qx1-.dx*.qz1, dx*.qy1-.dy*.qx1) in
  ((qx1,qy1,qz1), (qx2,qy2,qz2))

(* Return an array packed with vertices for given tube (centres and radii).
 * nr is how many faces in the radial direction (more means smoother).
 * Also return smooth normals, and texture coordinates for the rectangle
 * defined by texl--texr, text--texb. *)
let maketube ctr r nr texl texr text texb =
  let n = Array.length ctr in
  let vtx = Array.make ((n-2)*(nr+1)+2) (0.0,0.0,0.0)
  and nrm = Array.make ((n-2)*(nr+1)+2) (0.0,0.0,0.0)
  and texc = Array.make ((n-2)*(nr+1)+2) (0.0,0.0) in
  (* first the two ends *)
  vtx.(0) <- ctr.(0);
  nrm.(0) <- normdiff ctr.(0) ctr.(1);
  texc.(0) <- (0.5*.(texl+.texr),texb);
  vtx.((n-2)*(nr+1)+1) <- ctr.(n-1);
  nrm.((n-2)*(nr+1)+1) <- normdiff ctr.(n-1) ctr.(n-2);
  texc.((n-2)*(nr+1)+1) <- (0.5*.(texl+.texr),text);
  (* then the intermediate vertices *)
  for i = 1 to n-2 do
    let (cx,cy,cz) = ctr.(i) in
    let (qx1,qy1,qz1),(qx2,qy2,qz2) = orthodiff ctr.(i-1) ctr.(i) in
    for j = 0 to nr-1 do
      let t = 3.1415926535897 *. (float (2*j))/.(float nr) in
      let c = cos t and s = sin t in
      vtx.((i-1)*(nr+1)+j+1) <- (cx +. r.(i)*.c*.qx1 +. r.(i)*.s*.qx2,
                                 cy +. r.(i)*.c*.qy1 +. r.(i)*.s*.qy2,
                                 cz +. r.(i)*.c*.qz1 +. r.(i)*.s*.qz2);
      nrm.((i-1)*(nr+1)+j+1) <- (c*.qx1 +. s*.qx2,
                                 c*.qy1 +. s*.qy2,
                                 c*.qz1 +. s*.qz2);
      texc.((i-1)*(nr+1)+j+1) <- (texl+.(texr-.texl)*.(float j)/.(float nr),
                                  texb+.(text-.texb)*.(float i)/.(float (n-1)))
    done;
    (* complete the circle *)
    vtx.(i*(nr+1)) <- vtx.((i-1)*(nr+1)+1);
    nrm.(i*(nr+1)) <- nrm.((i-1)*(nr+1)+1);
    texc.(i*(nr+1)) <- (texr, texb+.(text-.texb)*.(float i)/.(float (n-1)))
  done;
  (vtx,nrm,texc)

(* draw the intermediate part of a tube *)
let drawopentube vtx nrm nr =
  let n = 2+((Array.length vtx)-2)/(nr+1) in
  for i = 1 to n-3 do
    GlDraw.begins `triangle_strip;
    for j = 0 to nr do
      GlDraw.normal3 nrm.((i-1)*(nr+1)+j+1);
      GlDraw.vertex3 vtx.((i-1)*(nr+1)+j+1);
      GlDraw.normal3 nrm.(i*(nr+1)+j+1);
      GlDraw.vertex3 vtx.(i*(nr+1)+j+1);
    done;
    GlDraw.ends ();
  done

(* draw the intermediate part of a tube with textures *)
let drawopentubetex vtx nrm texc nr =
  let n = 2+((Array.length vtx)-2)/(nr+1) in
  for i = 1 to n-3 do
    GlDraw.begins `triangle_strip;
    for j = 0 to nr do
      GlDraw.normal3 nrm.((i-1)*(nr+1)+j+1);
      GlTex.coord2 texc.((i-1)*(nr+1)+j+1);
      GlDraw.vertex3 vtx.((i-1)*(nr+1)+j+1);
      GlDraw.normal3 nrm.(i*(nr+1)+j+1);
      GlTex.coord2 texc.(i*(nr+1)+j+1);
      GlDraw.vertex3 vtx.(i*(nr+1)+j+1);
    done;
    GlDraw.ends ();
  done

(* draw cap for start of tube *)
let drawfirstcap vtx nrm nr =
  GlDraw.begins `triangle_fan;
  GlDraw.normal3 nrm.(0);
  GlDraw.vertex3 vtx.(0);
  for j = 1 to nr+1 do
    GlDraw.normal3 nrm.(j);
    GlDraw.vertex3 vtx.(j);
  done;
  GlDraw.ends ()

(* draw cap for start of tube with textures *)
let drawfirstcaptex vtx nrm texc nr =
  GlDraw.begins `triangle_fan;
  GlDraw.normal3 nrm.(0);
  GlTex.coord2 texc.(0);
  GlDraw.vertex3 vtx.(0);
  for j = 1 to nr+1 do
    GlDraw.normal3 nrm.(j);
    GlTex.coord2 texc.(j);
    GlDraw.vertex3 vtx.(j);
  done;
  GlDraw.ends ()

(* draw cap for end of tube *)
let drawlastcap vtx nrm nr =
  let nv = Array.length vtx in
  GlDraw.begins `triangle_fan;
  GlDraw.normal3 nrm.(nv-1);
  GlDraw.vertex3 vtx.(nv-1);
  for j = 1 to nr+1 do
    GlDraw.normal3 nrm.(nv-j-1);
    GlDraw.vertex3 vtx.(nv-j-1);
  done;
  GlDraw.ends ()

(* draw cap for end of tube with textures *)
let drawlastcaptex vtx nrm texc nr =
  let nv = Array.length vtx in
  GlDraw.begins `triangle_fan;
  GlDraw.normal3 nrm.(nv-1);
  GlTex.coord2 texc.(nv-1);
  GlDraw.vertex3 vtx.(nv-1);
  for j = 1 to nr+1 do
    GlDraw.normal3 nrm.(nv-j-1);
    GlTex.coord2 texc.(nv-j-1);
    GlDraw.vertex3 vtx.(nv-j-1);
  done;
  GlDraw.ends ()


(* Draw badguy's torso, with or without textures or in greyscale according
 * to the Boolean flags passed in, and with the given texture id if textures
 * are on. *)
let drawtorso textures greyscale texid =
  if textures then begin
    Gl.enable `texture_2d;
    GlTex.env (`mode `modulate);
    Tex.bind texid;
    GlDraw.color (1.0,1.0,1.0);
  end else if greyscale then
    GlDraw.color (0.4,0.4,0.4)
  else
    GlDraw.color (0.4,0.45,0.2);

  if greyscale then begin
    GlLight.material ~face:`front (`ambient_and_diffuse(0.4,0.4,0.4,1.0));
    GlLight.material ~face:`front (`specular(0.1,0.1,0.1,1.0));
  end else begin
    GlLight.material ~face:`front (`ambient_and_diffuse(0.4,0.45,0.2,1.0));
    GlLight.material ~face:`front (`specular(0.1,0.1,0.0,1.0));
  end;
  GlLight.material ~face:`front (`shininess 1.0);

  let (vtx,nrm,texc) = maketube
    [| -0.03,  -0.4,   0.0;
       -0.012, -0.385, 0.0;
        0.02,   -0.34, 0.0;
        0.075,  -0.23, 0.0;
        0.10,   -0.15, 0.0;
        0.11,   -0.08, 0.0;
        0.105,  -0.02, 0.0;
        0.09,    0.04, 0.0;
        0.065,   0.10, 0.0;
       -0.005,   0.20, 0.0;
       -0.017,   0.23, 0.0;
       -0.025,   0.235,0.0; |]
    [| 0.0;
       0.06; 0.1; 0.13; 0.16; 0.175; 0.175; 0.185; 0.185; 0.19; 0.195; 0.195;
       0.0 |]
    15 0.0 1.0 0.6 0.0 in

  if textures then begin
    drawopentubetex vtx nrm texc 15;
    drawfirstcaptex vtx nrm texc 15;
    Gl.disable `texture_2d;
  end else begin
    drawopentube vtx nrm 15;
    drawfirstcap vtx nrm 15;
  end;

  (* draw inside the mouth *)
  if greyscale then begin
    GlDraw.color (0.2,0.2,0.2);
    GlLight.material ~face:`front (`ambient_and_diffuse(0.2,0.2,0.2,1.0));
    GlLight.material ~face:`front (`specular(0.8,0.8,0.8,1.0));
  end else begin
    GlDraw.color (0.8,0.1,0.2);
    GlLight.material ~face:`front (`ambient_and_diffuse(0.8,0.1,0.2,1.0));
    GlLight.material ~face:`front (`specular(1.0,0.0,0.0,1.0));
  end;
  GlLight.material ~face:`front (`shininess 10.0);
  drawlastcap vtx nrm 15;

  (* add teeth *)
  GlDraw.color (0.8,0.8,0.8);
  GlLight.material ~face:`front (`ambient_and_diffuse(0.8,0.8,0.8,1.0));
  GlLight.material ~face:`front (`specular(1.0,1.0,1.0,1.0));
  GlLight.material ~face:`front (`shininess 20.0);
  let teeth = 6 in
  for i = 0 to teeth-1 do
    let t = 1.5707963 *. (float i)/.(float (teeth-1)) +. 2.3561945 in
    let s = sin t and c = cos t in
    let (vtx,nrm,_) = maketube
      [| -0.017 +. 0.18  *. c, 0.23 +. 0.1*.c,  0.18 *.s;
         -0.017 +. 0.185 *. c, 0.24 +. 0.1*.c,  0.185*.s;
         -0.017 +. 0.19  *. c, 0.265 +. 0.1*.c, 0.19 *.s;
         -0.017 +. 0.195 *. c, 0.27 +. 0.1*.c,  0.195*.s;
         -0.017 +. 0.2   *. c, 0.272 +. 0.1*.c, 0.2  *.s |]
      [| 0.0; 0.013; 0.01; 0.005; 0.0 |]
      7 0.0 0.0 0.0 0.0 in
    drawopentube vtx nrm 7;
    drawfirstcap vtx nrm 7;
    drawlastcap vtx nrm 7;
  done


(* draw badguy's upper head *)
let drawupperhead textures greyscale texid =
  if textures then begin
    Gl.enable `texture_2d;
    GlTex.env (`mode `modulate);
    Tex.bind texid;
    GlDraw.color (1.0,1.0,1.0);
  end else if greyscale then
    GlDraw.color (0.4,0.4,0.4)
  else
    GlDraw.color (0.4,0.45,0.2);

  if greyscale then begin
    GlLight.material ~face:`front (`ambient_and_diffuse(0.4,0.4,0.4,1.0));
    GlLight.material ~face:`front (`specular(0.1,0.1,0.1,1.0));
  end else begin
    GlLight.material ~face:`front (`ambient_and_diffuse(0.4,0.45,0.2,1.0));
    GlLight.material ~face:`front (`specular(0.1,0.1,0.0,1.0));
  end;
  GlLight.material ~face:`front (`shininess 1.0);

  let (vtx,nrm,texc) = maketube
    [| -0.001, -0.01, 0.0;
       0.0,     0.0,  0.0;
       0.01,    0.14, 0.0;
       0.03,    0.2,  0.0;
       0.05,    0.22, 0.0 |]
    [| 0.0; 0.2; 0.19; 0.1; 0.0 |]
    15 0.0 1.0 1.0 0.6 in

  if textures then begin
    drawopentubetex vtx nrm texc 15;
    drawlastcaptex vtx nrm texc 15;
    Gl.disable `texture_2d;
  end else begin
    drawopentube vtx nrm 15;
    drawlastcap vtx nrm 15;
  end;

  (* draw inside the mouth *)
  if greyscale then begin
    GlDraw.color (0.2,0.2,0.2);
    GlLight.material ~face:`front (`ambient_and_diffuse(0.2,0.2,0.2,1.0));
    GlLight.material ~face:`front (`specular(0.8,0.8,0.8,1.0));
  end else begin
    GlDraw.color (0.8,0.1,0.2);
    GlLight.material ~face:`front (`ambient_and_diffuse(0.8,0.1,0.2,1.0));
    GlLight.material ~face:`front (`specular(1.0,0.0,0.0,1.0));
  end;
  GlLight.material ~face:`front (`shininess 10.0);
  drawfirstcap vtx nrm 15;

  (* add teeth *)
  GlDraw.color (0.8,0.8,0.8);
  GlLight.material ~face:`front (`ambient_and_diffuse(0.8,0.8,0.8,1.0));
  GlLight.material ~face:`front (`specular(1.0,1.0,1.0,1.0));
  GlLight.material ~face:`front (`shininess 20.0);
  let teeth = 5 in
  for i = 0 to teeth-1 do
    let t = 1.5707963 *. (0.5 +. float i)/.(float teeth) +. 2.3561945 in
    let s = sin t and c = cos t in
    let (vtx,nrm,_) = maketube
      [| -0.017 +. 0.18  *. c,  0.03,  0.18 *.s;
         -0.017 +. 0.185 *. c,  0.02,  0.185*.s;
         -0.017 +. 0.19  *. c,  0.015, 0.19 *.s;
         -0.017 +. 0.195 *. c, -0.005, 0.195*.s;
         -0.017 +. 0.2   *. c, -0.008, 0.2  *.s |]
      [| 0.0; 0.013; 0.01; 0.005; 0.0 |]
      7 0.0 0.0 0.0 0.0 in
    drawopentube vtx nrm 7;
    drawfirstcap vtx nrm 7;
    drawlastcap vtx nrm 7;
  done


(* draw badguy's arms, hands, and spear *)
let drawarms textures greyscale texid =
  if greyscale then begin
    GlDraw.color (0.25,0.25,0.25);
    GlLight.material ~face:`front (`ambient_and_diffuse(0.25,0.25,0.25,1.0));
    GlLight.material ~face:`front (`specular(0.1,0.1,0.1,1.0));
  end else begin
    GlDraw.color (0.25,0.2,0.0);
    GlLight.material ~face:`front (`ambient_and_diffuse(0.25,0.2,0.0,1.0));
    GlLight.material ~face:`front (`specular(0.1,0.1,0.0,1.0));
  end;
  GlLight.material ~face:`front (`shininess 1.0);

  (* arms *)
  let (vtx,nrm,_) = maketube
    [|-0.16,   0.165, -0.203;
      -0.155,  0.16,  -0.202;   (* wrist *)
       0.1051, 0.15,  -0.201;   (* elbow *)
       0.13,   0.151, -0.2;
       0.131,  0.152,  0.0;     (* centre *)
       0.13,   0.151,  0.203;
       0.1051, 0.125,  0.202;   (* elbow *)
      -0.12,  -0.049,  0.201;   (* wrist *)
      -0.125, -0.05,   0.2; |]
    [| 0.0; 0.02; 0.023; 0.02; 0.025; 0.02; 0.023; 0.02; 0.0 |]
    7 0.0 0.2 1.0 0.6 in
  drawopentube vtx nrm 7;
  drawfirstcap vtx nrm 7;
  drawlastcap vtx nrm 7;

  (* left first finger *)
  let segs = 10 in
  let ctr = Array.make segs (0.0, 0.0, 0.0)
  and r = Array.make segs 0.0 in
  for i = 0 to segs-1 do
    let t = 5.2 *. (float i)/.(float segs) in
    ctr.(i) <- (-0.16 +. 0.04 *. cos t,
                -0.049 +. 0.04*.sin t,
                 0.2 -. 0.01*.t);
    r.(i) <- 0.01 +. 0.02 *. abs_float (sin (1.2 *. t));
  done;
  let (vtx,nrm,_) = maketube ctr r 7 0.0 0.5 1.0 0.6 in
  drawopentube vtx nrm 7;
  drawfirstcap vtx nrm 7;
  drawlastcap vtx nrm 7;

  (* left second finger *)
  for i = 0 to segs-1 do
    let t = 5.2 *. (float i)/.(float segs) in
    ctr.(i) <- (-0.16 +. 0.04 *. cos t,
                -0.0495 +. 0.04*.sin t,
                 0.2 +. 0.01*.t);
  done;
  let (vtx,nrm,_) = maketube ctr r 7 0.0 0.5 1.0 0.6 in
  drawopentube vtx nrm 7;
  drawfirstcap vtx nrm 7;
  drawlastcap vtx nrm 7;

  (* right first finger *)
  for i = 0 to segs-1 do
    let t = 5.2 *. (float i)/.(float segs) in
    ctr.(i) <- (-0.195 +. 0.04 *. cos t,
                0.165 -. 0.04*.sin t,
                -0.2 -. 0.01*.t);
  done;
  let (vtx,nrm,_) = maketube ctr r 7 0.0 0.5 1.0 0.6 in
  drawopentube vtx nrm 7;
  drawfirstcap vtx nrm 7;
  drawlastcap vtx nrm 7;

  (* right second finger *)
  for i = 0 to segs-1 do
    let t = 5.2 *. (float i)/.(float segs) in
    ctr.(i) <- (-0.195 +. 0.04 *. cos t,
                0.16 -. 0.04*.sin t,
                -0.2 +. 0.01*.t);
  done;
  let (vtx,nrm,_) = maketube ctr r 7 0.0 0.5 1.0 0.6 in
  drawopentube vtx nrm 7;
  drawfirstcap vtx nrm 7;
  drawlastcap vtx nrm 7;

  (* spear shaft *)
  if greyscale then begin
    GlDraw.color (0.1,0.1,0.1);
    GlLight.material ~face:`front (`ambient_and_diffuse(0.1,0.1,0.1,1.0));
    GlLight.material ~face:`front (`specular(0.5,0.5,0.5,1.0));
  end else begin
    GlDraw.color (0.15,0.1,0.0);
    GlLight.material ~face:`front (`ambient_and_diffuse(0.15,0.1,0.0,1.0));
    GlLight.material ~face:`front (`specular(0.5,0.5,0.0,1.0));
  end;
  GlLight.material ~face:`front (`shininess 10.0);

  let (vtx,nrm,_) = maketube
    [| -0.1513,   -0.10251, 0.331;
       -0.15125, -0.1025, 0.33;
       -0.199375, 0.18625, -0.275;
       -0.20375, 0.2125, -0.33 |]
    [| -0.0; 0.015; 0.015; 0.0 |] 6 0.0 0.0 0.0 0.0 in
  drawopentube vtx nrm 6;
  drawfirstcap vtx nrm 6;
  drawlastcap vtx nrm 6


(* draw badguy's legs *)
let drawlegs textures greyscale texid =
  if textures then begin
    Gl.enable `texture_2d;
    GlTex.env (`mode `modulate);
    Tex.bind texid;
    GlDraw.color (1.0,1.0,1.0);
  end else if greyscale then
    GlDraw.color (0.4,0.4,0.4)
  else
    GlDraw.color (0.4,0.45,0.2);

  if greyscale then begin
    GlLight.material ~face:`front (`ambient_and_diffuse(0.4,0.4,0.4,1.0));
    GlLight.material ~face:`front (`specular(0.1,0.1,0.1,1.0));
  end else begin
    GlLight.material ~face:`front (`ambient_and_diffuse(0.4,0.45,0.2,1.0));
    GlLight.material ~face:`front (`specular(0.1,0.1,0.0,1.0));
  end;
  GlLight.material ~face:`front (`shininess 1.0);

  let (vtx,nrm,texc) = maketube 
    [| 0.035,  -0.49,  -0.158;
       0.035,  -0.47,  -0.16;   (* ankle *)
       0.01,   -0.4,   -0.17;   (* knee *)
       0.0351, -0.35,  -0.161;  (* hip *)
       0.041,  -0.32,  -0.101;  (* centre *)
       0.04,   -0.321,  0.1;    (* centre *)
       0.035,  -0.351,  0.16;   (* hip *)
       0.01,   -0.4,    0.17;   (* knee *)
       0.035,  -0.47,   0.16;   (* ankle *)
       0.035,  -0.49,   0.158; |]
    [| 0.0; 0.035; 0.04; 0.04; 0.05; 0.05; 0.04; 0.04; 0.035; 0.0 |]
    9 0.0 1.0 1.0 0.6 in

  if textures then begin
    drawopentubetex vtx nrm texc 9;
    drawfirstcaptex vtx nrm texc 9;
    drawlastcaptex vtx nrm texc 9;
  end else begin
    drawopentube vtx nrm 9;
    drawfirstcap vtx nrm 9;
    drawlastcap vtx nrm 9;
  end;

  (* right foot *)
  let (vtx,nrm,texc) = maketube
    [| -0.15, -0.499,   0.23;
       -0.1,  -0.474, 0.2;
       -0.05, -0.459, 0.17;
        0.0,  -0.449, 0.15;
        0.1,  -0.449, 0.15 |]
    [| 0.0; 0.025; 0.04; 0.05; 0.0 |]
    9 0.0 0.5 1.0 0.6 in
  if textures then begin
    drawopentubetex vtx nrm texc 9;
    drawfirstcaptex vtx nrm texc 9;
  end else begin
    drawopentube vtx nrm 9;
    drawfirstcap vtx nrm 9;
  end;
  let (vtx,nrm,texc) = maketube
    [| -0.15, -0.499, 0.07;
       -0.1,  -0.474, 0.1;
       -0.05, -0.459, 0.13;
        0.0,  -0.449, 0.15;
        0.1,  -0.449, 0.15 |]
    [| 0.0; 0.025; 0.04; 0.05; 0.0 |]
    9 0.0 0.5 1.0 0.6 in
  if textures then begin
    drawopentubetex vtx nrm texc 9;
    drawfirstcaptex vtx nrm texc 9;
  end else begin
    drawopentube vtx nrm 9;
    drawfirstcap vtx nrm 9;
  end;
  let (vtx,nrm,texc) = maketube
    [| 0.15, -0.499, 0.15;
       0.08, -0.459, 0.1501;
       0.0,  -0.449, 0.1502;
      -0.01, -0.449, 0.1503 |]
    [| 0.0; 0.04; 0.05; 0.05; 0.0 |]
    9 0.0 0.5 1.0 0.6 in
  if textures then begin
    drawopentubetex vtx nrm texc 9;
    drawfirstcaptex vtx nrm texc 9;
  end else begin
    drawopentube vtx nrm 9;
    drawfirstcap vtx nrm 9;
  end;

  (* left foot *)
  let (vtx,nrm,texc) = maketube
    [| -0.15, -0.499, -0.23;
       -0.1,  -0.474, -0.2;
       -0.05, -0.459, -0.17;
        0.0,  -0.449, -0.15;
        0.1,  -0.449, -0.15 |]
    [| 0.0; 0.025; 0.04; 0.05; 0.0 |]
    9 0.0 0.5 1.0 0.6 in
  if textures then begin
    drawopentubetex vtx nrm texc 9;
    drawfirstcaptex vtx nrm texc 9;
  end else begin
    drawopentube vtx nrm 9;
    drawfirstcap vtx nrm 9;
  end;
  let (vtx,nrm,texc) = maketube
    [| -0.15, -0.499, -0.07;
       -0.1,  -0.474, -0.1;
       -0.05, -0.459, -0.13;
        0.0,  -0.449, -0.15;
        0.1,  -0.449, -0.15 |]
    [| 0.0; 0.025; 0.04; 0.05; 0.0 |]
    9 0.0 0.5 1.0 0.6 in
  if textures then begin
    drawopentubetex vtx nrm texc 9;
    drawfirstcaptex vtx nrm texc 9;
  end else begin
    drawopentube vtx nrm 9;
    drawfirstcap vtx nrm 9;
  end;
  let (vtx,nrm,texc) = maketube
    [| 0.15, -0.499, -0.15;
       0.08, -0.459, -0.1501;
       0.0,  -0.449, -0.1502;
      -0.01, -0.449, -0.1503 |]
    [| 0.0; 0.04; 0.05; 0.05; 0.0 |]
    9 0.0 0.5 1.0 0.6 in
  if textures then begin
    drawopentubetex vtx nrm texc 9;
    drawfirstcaptex vtx nrm texc 9;
    Gl.disable `texture_2d;
  end else begin
    drawopentube vtx nrm 9;
    drawfirstcap vtx nrm 9;
  end




(********************* init **************************************************)
let init () =
  (* set up display lists for guard's various body parts *)
  let plains = GlList.gen_lists ~len:4 in
  plainlists := DisplayLists plains;
  let texs = GlList.gen_lists ~len:4 in
  texlists := DisplayLists texs;
  let plaingreys = GlList.gen_lists ~len:4 in
  plaingreylists := DisplayLists plaingreys;
  let texgreys = GlList.gen_lists ~len:4 in
  texgreylists := DisplayLists texgreys;

  (* get the guard texture *) 
  let texid = Tex.loadPPM "Data/guard.ppm" in
  if texid = Nativeint.zero then begin
    prerr_endline ("Problem loading image Data/guard.ppm");
    exit 0
  end else ();
  let texgreyid = Tex.loadPPM "Data/guard_grey.ppm" in
  if texgreyid = Nativeint.zero then begin
    prerr_endline ("Problem loading image Data/guard_grey.ppm");
    exit 0
  end else ();

  (* guard is looking along negative x axis *)
  (* torso *)
  GlList.begins (GlList.nth plains ~pos:0) ~mode:`compile;
  drawtorso false false Nativeint.zero;
  GlList.ends ();
  GlList.begins (GlList.nth texs ~pos:0) ~mode:`compile;
  drawtorso true false texid;
  GlList.ends ();
  GlList.begins (GlList.nth plaingreys ~pos:0) ~mode:`compile;
  drawtorso false true Nativeint.zero;
  GlList.ends ();
  GlList.begins (GlList.nth texgreys ~pos:0) ~mode:`compile;
  drawtorso true true texgreyid;
  GlList.ends ();
 
  (* upper head *)
  GlList.begins (GlList.nth plains ~pos:1) ~mode:`compile;
  drawupperhead false false Nativeint.zero;
  GlList.ends ();
  GlList.begins (GlList.nth texs ~pos:1) ~mode:`compile;
  drawupperhead true false texid;
  GlList.ends ();
  GlList.begins (GlList.nth plaingreys ~pos:1) ~mode:`compile;
  drawupperhead false true Nativeint.zero;
  GlList.ends ();
  GlList.begins (GlList.nth texgreys ~pos:1) ~mode:`compile;
  drawupperhead true true texgreyid;
  GlList.ends ();
 
  (* arms, hands, and spear *)
  GlList.begins (GlList.nth plains ~pos:2) ~mode:`compile;
  drawarms false false Nativeint.zero;
  GlList.ends ();
  GlList.begins (GlList.nth texs ~pos:2) ~mode:`compile;
  drawarms true false texid;
  GlList.ends ();
  GlList.begins (GlList.nth plaingreys ~pos:2) ~mode:`compile;
  drawarms false true Nativeint.zero;
  GlList.ends ();
  GlList.begins (GlList.nth texgreys ~pos:2) ~mode:`compile;
  drawarms true true texgreyid;
  GlList.ends ();
 
  (* legs *)
  GlList.begins (GlList.nth plains ~pos:3) ~mode:`compile;
  drawlegs false false Nativeint.zero;
  GlList.ends ();
  GlList.begins (GlList.nth texs ~pos:3) ~mode:`compile;
  drawlegs true false texid;
  GlList.ends ();
  GlList.begins (GlList.nth plaingreys ~pos:3) ~mode:`compile;
  drawlegs false true Nativeint.zero;
  GlList.ends ();
  GlList.begins (GlList.nth texgreys ~pos:3) ~mode:`compile;
  drawlegs true true texgreyid;
  GlList.ends ()


(* update the given bad guy by the time increment *)
let update bg dt avoidlist scents =
  (* first check if the badguy is in collision with a rematterized object, and
   * if so, send him back to his initial position. *)
  if List.fold_left ~f:(fun incol hull -> incol || incollision hull bg.hull)
                    ~init:false avoidlist then begin
    bg.vx <- 0.0;
    bg.vy <- 0.0;
    bg.vz <- 0.0;
    bg.hull.x <- bg.initx;
    bg.hull.y <- bg.inity;
    bg.hull.z <- bg.initz;
    bg.heading <- bg.initheading;

  end else begin
    let per_sec = 1.0/.dt in
    (* modify heading to point more towards most favourable scent *)
    let wx = (scents.(6) +. scents.(7) +. scents.(8))
             -. (scents.(0)+.scents.(1)+.scents.(2))
             +. (if bg.hull.x -. floor bg.hull.x < 0.5 then 0.5 else -0.5)
             *. (scents.(3) +. scents.(4) +. scents.(5))
    and wz = (scents.(2) +. scents.(5) +. scents.(8))
             -. (scents.(0)+.scents.(3)+.scents.(6))
             +. (if bg.hull.z -. floor bg.hull.z < 0.5 then 0.5 else -0.5)
             *. (scents.(1) +. scents.(4) +. scents.(7)) in
    let dir = 57.29578 *. atan2 wx (-.wz) in
    if abs_float (bg.heading -. dir) < 180.0 then
      bg.heading <- bg.heading +. dt *. (dir -. bg.heading)
    else
      bg.heading <- bg.heading +. dt *. (360.0 +. dir -. bg.heading);
    bg.heading <- mod_float bg.heading 360.0;

    (* Next decide if his legs are in contact with the ground *)
    let h_onfloor =
      List.fold_left
        ~f:(fun h ob -> min h (topcollision bg.hull ob 0.01))
        ~init:1.0 avoidlist in
    let decay = if h_onfloor < 0.0 then 0.008*.per_sec
                                   else 0.99 in
    let accel = if h_onfloor < 0.0 then 50.0 *. dt
                                   else dt in
    let radians = 0.017453293 *. bg.heading in
    let headx = sin radians and headz = -.cos radians in
    bg.vx <- decay *. (bg.vx +. accel*.headx);
    bg.vz <- decay *. (bg.vz +. accel*.headz);
    if h_onfloor >= 0.0 then
      bg.vy <- bg.vy -. 5.0*.dt   (* gravity *)
    else ();

    (* now clamp velocity to make sure bad things don't happen *)
    if bg.vx < -.0.5*.per_sec then bg.vx <- -0.5*.per_sec
    else if bg.vx > 0.5*.per_sec then bg.vx <- 0.5*.per_sec else ();
    if bg.vy < -.0.5*.per_sec then bg.vy <- -0.5*.per_sec
    else if bg.vy > 0.5*.per_sec then bg.vy <- 0.5*.per_sec else ();
    if bg.vz < -.0.5*.per_sec then bg.vz <- -0.5*.per_sec
    else if bg.vz > 0.5*.per_sec then bg.vz <- 0.5*.per_sec else ();

    (* check for collisions we're supposed to avoid *)
    let dx = dt*.bg.vx and dy = dt*.bg.vy and dz = dt*.bg.vz in
    let (newdx,newdy,newdz) = deflect bg.hull dx dy dz avoidlist in
    bg.vx <- 0.999 *. newdx *. per_sec;
    bg.vy <- 0.999 *. newdy *. per_sec;
    bg.vz <- 0.999 *. newdz *. per_sec;

    (* advance badguy's gait if not falling *)
    if h_onfloor < 0.0 then
      bg.gait <- bg.gait +. dt
    else ();

    (* if we're not making adequate progress, maybe jump *)
    if h_onfloor < 0.0
    && abs_float(newdx)+.abs_float(newdz) < 0.3*.(abs_float(dx)+.abs_float(dz))
    then begin
      if bg.triedjump || Random.float 1.0 > dt then
        (* just try turning to the side *)
        bg.heading <- mod_float (bg.heading +. 60.0 *. dt) 360.0
      else begin
        bg.vy <- 2.4;  (* jump *)
        bg.triedjump <- true
      end
    end else
      bg.triedjump <- false;

    (* finally, move the badguy *)
    bg.hull.x <- bg.hull.x +. 0.999 *. newdx;
    bg.hull.y <- bg.hull.y +. 0.999 *. newdy;
    bg.hull.z <- bg.hull.z +. 0.999 *. newdz
  end


(* return the hull-action pair for the given badguy (the action is what
 * happens when he touches Spiff) *)
let actionpair bg =
  (bg.hull, LoseChance)


(* render the bad guy *)
let render bg textureson greyscaleon =
  GlMat.push ();
  GlMat.translate ~x:bg.hull.x ~y:bg.hull.y ~z:bg.hull.z ();
  GlMat.rotate ~angle:(-.bg.heading-.90.0) ~y:1.0 ();

  let DisplayLists dls =
    if textureson then begin
      if greyscaleon then !texgreylists
      else !texlists
    end else begin
      if greyscaleon then !plaingreylists
      else !plainlists
   end in

  GlMat.push ();
  GlMat.translate ~x:0.0 ~y:(0.05*.abs_float (sin (6.5 *. bg.gait))) ~z:0.0 ();
  GlList.call (GlList.nth dls ~pos:0);  (* torso *)
  GlMat.translate ~x:(-0.025) ~y:0.235 ~z:0.0 ();
  GlMat.rotate ~angle:(11.0 +. 10.0 *. sin (2.78 *. bg.gait)) ~z:1.0 ();
  GlList.call (GlList.nth dls ~pos:1);  (* upper head *)
  GlMat.pop ();

  GlMat.push ();
  GlMat.translate ~x:0.0 ~y:(0.03*.abs_float (sin (6.5 *. bg.gait))) ~z:0.0 ();
  GlMat.rotate ~angle:(15.0 *. sin (6.5 *. bg.gait)) ~y:1.0 ();
  GlList.call (GlList.nth dls ~pos:2);  (* arms and spear *)
  GlMat.pop ();

  GlMat.push ();
  GlMat.rotate ~angle:(27.0 *. sin (6.5 *. (bg.gait+.0.5))) ~y:1.0 ();
  GlMat.rotate ~angle:(5.0  *. cos (6.5 *. (bg.gait+.0.5))) ~x:1.0 ();
  GlList.call (GlList.nth dls ~pos:3);  (* legs *)
  GlMat.pop ();
 
  GlMat.pop ()

