(* freeobj.ml: implementation of free objects behaviour and rendering *)

open Collision

type kind = Charge | Bribe | Teleporter

type t = {
  kind: kind;
  hull: Collision.hull;
  mutable vy: float;
  mutable dmtimer: float;
}

(* create a new free object of the given kind at the given position *)
let newfree kind x y z =
  { kind = kind;
    hull = (match kind with
              Charge -> {x=x; y=y; z=z; lx = 0.05; ly = 0.075; lz = 0.05}
            | Bribe  -> {x=x; y=y; z=z; lx = 0.1; ly = 0.1; lz = 0.07}
            | Teleporter -> {x=x; y=y; z=z; lx = 0.48; ly = 0.48; lz = 0.48});
    vy = 0.0;
    dmtimer = 0.0;
  }


(* we use display lists for rendering solid objects *)

type displaylists = NotInitialized | DisplayLists of GlList.base

let plainlists = ref NotInitialized
and texlists = ref NotInitialized
and plaingreylists = ref NotInitialized
and texgreylists = ref NotInitialized

let chargetexid = ref Nativeint.zero
and chargetexgreyid = ref Nativeint.zero
and bribetexid = ref Nativeint.zero
and bribetexgreyid = ref Nativeint.zero
and teleportertexid = ref Nativeint.zero
and teleportertexgreyid = ref Nativeint.zero


(* draw a teleporter charge with given alpha value, flags for texture and
 * greyscale modes, and the id of the texture to use if requested. *)
let drawcharge alpha textures greyscale texid =
  let nr = 15 in  (* how many faces around the cylinder *)

  (* the bottom *)
  GlDraw.color ~alpha (0.7,0.7,0.7);
  GlLight.material ~face:`front (`ambient_and_diffuse(0.7,0.7,0.7,alpha));
  GlLight.material ~face:`front (`specular(1.0,1.0,1.0,alpha));
  GlLight.material ~face:`front (`shininess 30.0);
  GlDraw.begins `triangle_fan;
    GlDraw.normal ~x:0.0 ~y:(-1.0) ~z:0.0 ();
    GlDraw.vertex ~x:0.0 ~y:(-0.075) ~z:0.0 ();
    for i = 0 to nr do
      let theta = 2.0 *. 3.1415926535897 *. (float_of_int (i mod nr))
                                         /. (float_of_int nr) in
      let c = cos theta and s = sin theta in
      GlDraw.vertex ~x:(0.05*.c) ~y:(-0.075) ~z:(0.05*.s) ();
    done;
  GlDraw.ends ();

  (* the main outside *)
  if textures then begin
    Gl.enable `texture_2d;
    GlTex.env (`mode `modulate);
    Tex.bind texid;
    GlDraw.color ~alpha (1.0,1.0,1.0);
  end else if greyscale then
    GlDraw.color ~alpha (0.7,0.7,0.7)
  else
    GlDraw.color ~alpha (0.2,0.2,1.0);

  if greyscale then begin
    GlLight.material ~face:`front (`ambient_and_diffuse(0.7,0.7,0.7,alpha));
    GlLight.material ~face:`front (`specular(1.0,1.0,1.0,alpha));
  end else begin
    GlLight.material ~face:`front (`ambient_and_diffuse(0.2,0.2,1.0,alpha));
    GlLight.material ~face:`front (`specular(1.0,1.0,1.0,alpha));
  end;
  GlLight.material ~face:`front (`shininess 20.0);

  GlDraw.begins `quad_strip;
    for i = 0 to nr do
      let theta = 2.0 *. 3.1415926535897 *. (float_of_int (i mod nr))
                                         /. (float_of_int nr) in
      let c = cos theta and s = sin theta in
      GlDraw.normal ~x:c ~y:0.0 ~z:s ();
      if textures then GlTex.coord2 ((float i)/.(float nr),0.0) else ();
      GlDraw.vertex ~x:(0.05 *. c) ~y:(-0.075) ~z:(0.05*.s) ();
      if textures then GlTex.coord2 ((float i)/.(float nr),0.43) else ();
      GlDraw.vertex ~x:(0.05 *. c) ~y:0.06 ~z:(0.05 *. s) ();
    done; 
  GlDraw.ends ();

  if textures then
    Gl.disable `texture_2d
  else ();

  (* the top, not including the copper contact *)
  GlDraw.color ~alpha (0.7,0.7,0.7);
  GlLight.material ~face:`front (`ambient_and_diffuse(0.7,0.7,0.7,alpha));
  GlLight.material ~face:`front (`specular(1.0,1.0,1.0,alpha));
  GlLight.material ~face:`front (`shininess 30.0);
  GlDraw.begins `quad_strip;
    GlDraw.normal ~x:0.0 ~y:1.0 ~z:0.0 ();
    for i = 0 to nr do
      let theta = 2.0 *. 3.1415926535897 *. (float_of_int (i mod nr))
                                         /. (float_of_int nr) in
      let c = cos theta and s = sin theta in
      GlDraw.vertex ~x:(0.05 *. c) ~y:0.06 ~z:(0.05 *. s) ();
      GlDraw.vertex ~x:(0.02 *. c) ~y:0.06 ~z:(0.02 *. s) ();
    done; 
  GlDraw.ends ();

  (* the copper contact on top *)
  if greyscale then begin
    GlDraw.color ~alpha (0.3,0.3,0.3)
  end else begin
    GlDraw.color ~alpha (0.4,0.3,0.0);
    GlLight.material ~face:`front (`ambient_and_diffuse(0.4,0.3,0.0,alpha));
    GlLight.material ~face:`front (`specular(0.8,0.7,0.0,alpha));
  end;
  GlLight.material ~face:`front (`shininess 30.0);

  GlDraw.begins `quad_strip;
    for i = 0 to nr do
      let theta = 2.0 *. 3.1415926535897 *. (float_of_int (i mod nr))
                                         /. (float_of_int nr) in
      let c = cos theta and s = sin theta in
      GlDraw.normal ~x:c ~y:0.0 ~z:s ();
      GlDraw.vertex ~x:(0.02 *. c) ~y:0.06 ~z:(0.02 *. s) ();
      GlDraw.normal ~x:(0.7071 *. cos theta) ~y:0.7071 ~z:(0.7071 *. sin theta)
                    ();
      GlDraw.vertex ~x:(0.02 *. c) ~y:0.075 ~z:(0.02 *. s) ();
    done; 
  GlDraw.ends ();
  GlDraw.begins `triangle_fan;
    GlDraw.normal ~x:0.0 ~y:1.0 ~z:0.0 ();
    GlDraw.vertex ~x:0.0 ~y:0.075 ~z:0.0 ();
    for i = 0 to nr do
      let theta = 2.0 *. 3.1415926535897 *. (float_of_int ((nr-i) mod nr))
                                         /. (float_of_int nr) in
      GlDraw.normal ~x:(0.7071 *. cos theta) ~y:0.7071 ~z:(0.7071 *. sin theta)
                    ();
      GlDraw.vertex ~x:(0.02 *. cos theta) ~y:0.075 ~z:(0.02 *. sin theta) ();
    done;
  GlDraw.ends ()


(* draw a bribe *)
let drawbribe alpha textures greyscale texid =
  if textures then begin
    Gl.enable `texture_2d;
    GlDraw.color ~alpha (1.0,1.0,1.0);
    GlTex.env (`mode `modulate);
    Tex.bind texid;
  end else if greyscale then
    GlDraw.color ~alpha (0.3,0.3,0.3)
  else
    GlDraw.color ~alpha (0.5,0.2,0.0);

  if greyscale then begin
    GlLight.material ~face:`front (`ambient_and_diffuse(0.3,0.3,0.3,alpha));
    GlLight.material ~face:`front (`specular(0.1,0.1,0.1,alpha));
  end else begin
    GlLight.material ~face:`front (`ambient_and_diffuse(0.5,0.2,0.0,alpha));
    GlLight.material ~face:`front (`specular(0.1,0.1,0.0,alpha));
  end;
  GlLight.material ~face:`front (`shininess 1.0);

  (* the side faces, apart from the rounded section *)
  GlDraw.begins `quads;
    GlDraw.normal ~x:(-1.0) ();
    if textures then GlTex.coord2 (0.4,0.0) else ();
    GlDraw.vertex ~x:(-0.1) ~y:(-0.1) ~z:(-0.07) ();
    if textures then GlTex.coord2 (0.68,0.0) else ();
    GlDraw.vertex ~x:(-0.1) ~y:(-0.1) ~z:0.07 ();
    if textures then GlTex.coord2 (0.68,0.26) else ();
    GlDraw.vertex ~x:(-0.1) ~y:0.03 ~z:0.07 ();
    if textures then GlTex.coord2 (0.4,0.26) else ();
    GlDraw.vertex ~x:(-0.1) ~y:0.03 ~z:(-0.07) ();
    GlDraw.normal ~x:1.0 ();
    if textures then GlTex.coord2 (0.4,0.0) else ();
    GlDraw.vertex ~x:0.1 ~y:(-0.1) ~z:(-0.07) ();
    if textures then GlTex.coord2 (0.4,0.26) else ();
    GlDraw.vertex ~x:0.1 ~y:0.03 ~z:(-0.07) ();
    if textures then GlTex.coord2 (0.68,0.26) else ();
    GlDraw.vertex ~x:0.1 ~y:0.03 ~z:0.07 ();
    if textures then GlTex.coord2 (0.68,0.0) else ();
    GlDraw.vertex ~x:0.1 ~y:(-0.1) ~z:0.07 ();
    GlDraw.normal ~x:0.0 ~y:(-1.0) ();
    if textures then GlTex.coord2 (0.0,0.26) else ();
    GlDraw.vertex ~x:(-0.1) ~y:(-0.1) ~z:(-0.07) ();
    if textures then GlTex.coord2 (0.4,0.26) else ();
    GlDraw.vertex ~x:0.1 ~y:(-0.1) ~z:(-0.07) ();
    if textures then GlTex.coord2 (0.4,0.54) else ();
    GlDraw.vertex ~x:0.1 ~y:(-0.1) ~z:0.07 ();
    if textures then GlTex.coord2 (0.0,0.54) else ();
    GlDraw.vertex ~x:(-0.1) ~y:(-0.1) ~z:0.07 ();
    GlDraw.normal ~x:0.0 ~y:0.0 ~z:(-1.0) ();
    if textures then GlTex.coord2 (0.0,0.0) else ();
    GlDraw.vertex ~x:(-0.1) ~y:(-0.1) ~z:(-0.07) ();
    if textures then GlTex.coord2 (0.0,0.26) else ();
    GlDraw.vertex ~x:(-0.1) ~y:0.03 ~z:(-0.07) ();
    if textures then GlTex.coord2 (0.4,0.26) else ();
    GlDraw.vertex ~x:0.1 ~y:0.03 ~z:(-0.07) ();
    if textures then GlTex.coord2 (0.4,0.0) else ();
    GlDraw.vertex ~x:0.1 ~y:(-0.1) ~z:(-0.07) ();
    GlDraw.normal ~x:0.0 ~y:0.0 ~z:1.0 ();
    if textures then GlTex.coord2 (0.0,0.0) else ();
    GlDraw.vertex ~x:(-0.1) ~y:(-0.1) ~z:0.07 ();
    if textures then GlTex.coord2 (0.4,0.0) else ();
    GlDraw.vertex ~x:0.1 ~y:(-0.1) ~z:0.07 ();
    if textures then GlTex.coord2 (0.4,0.26) else ();
    GlDraw.vertex ~x:0.1 ~y:0.03 ~z:0.07 ();
    if textures then GlTex.coord2 (0.0,0.26) else ();
    GlDraw.vertex ~x:(-0.1) ~y:0.03 ~z:0.07 ();
  GlDraw.ends ();

  let nr = 10 in  (* how many face to use for the rounded top *)
  (* the rounded side faces *)
  GlDraw.begins `triangle_fan;
    GlDraw.normal ~x:(-1.0) ();
    if textures then GlTex.coord2 (0.54,0.26) else ();
    GlDraw.vertex ~x:(-0.1) ~y:0.03 ~z:0.0 ();
    for i = 0 to nr do
      let t = 3.1415926535897 *. (float i) /. (float nr) in
      if textures then GlTex.coord2 (0.54+.0.14*.cos t, 0.26+.0.14*.sin t)
      else ();
      GlDraw.vertex ~x:(-0.1) ~y:(0.03+.0.07*.sin t) ~z:(0.07*.cos t) ();
    done;
  GlDraw.ends ();
  GlDraw.begins `triangle_fan;
    GlDraw.normal ~x:1.0 ();
    if textures then GlTex.coord2 (0.54,0.26) else ();
    GlDraw.vertex ~x:0.1 ~y:0.03 ~z:0.0 ();
    for i = 0 to nr do
      let t = 3.1415926535897 *. (float (nr-i)) /. (float nr) in
      if textures then GlTex.coord2 (0.54+.0.14*.cos t, 0.26+.0.14*.sin t)
      else ();
      GlDraw.vertex ~x:0.1 ~y:(0.03+.0.07*.sin t) ~z:(0.07*.cos t) ();
    done;
  GlDraw.ends ();
  (* the rounded top *)
  GlDraw.begins `quad_strip;
    for i = 0 to nr do
      let t = 3.1415926535897 *. (float (nr-i)) /. (float nr) in
      let c = cos t and s = sin t in
      GlDraw.normal ~x:0.0 ~y:s ~z:c ();
      if textures then GlTex.coord2 (0.4,0.54+.0.22*.(float i)/.(float nr))
      else ();
      GlDraw.vertex ~x:0.1    ~y:(0.03+.0.07*.s) ~z:(0.07*.c) ();
      if textures then GlTex.coord2 (0.0,0.54+.0.22*.(float i)/.(float nr))
      else ();
      GlDraw.vertex ~x:(-0.1) ~y:(0.03+.0.07*.s) ~z:(0.07*.c) ();
    done;
  GlDraw.ends ();
  if textures then
    Gl.disable `texture_2d
  else ()


(* draw a teleporter *)
let drawteleporter alpha textures greyscale texid =
  if textures then begin
    Gl.enable `texture_2d;
    GlDraw.color ~alpha (1.0,1.0,1.0);
    GlTex.env (`mode `modulate);
    Tex.bind texid;
  end else if greyscale then
    GlDraw.color ~alpha (0.05,0.05,0.05)
  else
    GlDraw.color ~alpha (0.0,0.0,0.1);

  GlLight.material ~face:`front (`ambient_and_diffuse(0.0,0.0,0.0,alpha));
  if greyscale then
    GlLight.material ~face:`front (`specular(3.0,3.0,3.0,alpha))
  else
    GlLight.material ~face:`front (`specular(3.0,3.0,5.0,alpha));
  GlLight.material ~face:`front (`shininess 10.0);
  if greyscale then begin
    GlLight.material ~face:`back (`ambient_and_diffuse(0.4,0.4,0.4,alpha));
    GlLight.material ~face:`back (`specular(1.2,1.2,1.2,alpha));
  end else begin
    GlLight.material ~face:`back (`ambient_and_diffuse(0.4,0.4,0.5,alpha));
    GlLight.material ~face:`back (`specular(0.8,0.8,2.0,alpha));
  end;
  GlLight.material ~face:`back (`shininess 10.0);
  Gl.disable `cull_face;   (* we might be inside the teleporter! *)

  (* it's just a cube with six faces *)
  GlDraw.begins `quads;
    GlDraw.normal ~x:(-1.0) ();
    if textures then GlTex.coord2 (0.0,0.0) else ();
    GlDraw.vertex ~x:(-0.445) ~y:(-0.445) ~z:(-0.445) ();
    if textures then GlTex.coord2 (1.0,0.0) else ();
    GlDraw.vertex ~x:(-0.445) ~y:(-0.445) ~z:0.445 ();
    if textures then GlTex.coord2 (1.0,1.0) else ();
    GlDraw.vertex ~x:(-0.445) ~y:0.445    ~z:0.445 ();
    if textures then GlTex.coord2 (0.0,1.0) else ();
    GlDraw.vertex ~x:(-0.445) ~y:0.445    ~z:(-0.445) ();
    GlDraw.normal ~x:1.0 ();
    if textures then GlTex.coord2 (0.0,0.0) else ();
    GlDraw.vertex ~x:0.445    ~y:(-0.445) ~z:(-0.445) ();
    if textures then GlTex.coord2 (1.0,0.0) else ();
    GlDraw.vertex ~x:0.445    ~y:0.445    ~z:(-0.445) ();
    if textures then GlTex.coord2 (1.0,1.0) else ();
    GlDraw.vertex ~x:0.445    ~y:0.445    ~z:0.445 ();
    if textures then GlTex.coord2 (0.0,1.0) else ();
    GlDraw.vertex ~x:0.445    ~y:(-0.445) ~z:0.445 ();
    GlDraw.normal ~x:0.0 ~y:(-1.0) ();
    if textures then GlTex.coord2 (0.0,0.0) else ();
    GlDraw.vertex ~x:(-0.445) ~y:(-0.445) ~z:(-0.445) ();
    if textures then GlTex.coord2 (1.0,0.0) else ();
    GlDraw.vertex ~x:0.445    ~y:(-0.445) ~z:(-0.445) ();
    if textures then GlTex.coord2 (1.0,1.0) else ();
    GlDraw.vertex ~x:0.445    ~y:(-0.445) ~z:0.445 ();
    if textures then GlTex.coord2 (0.0,1.0) else ();
    GlDraw.vertex ~x:(-0.445) ~y:(-0.445) ~z:0.445 ();
    GlDraw.normal ~x:0.0 ~y:1.0 ();
    if textures then GlTex.coord2 (0.0,0.0) else ();
    GlDraw.vertex ~x:(-0.445) ~y:0.445    ~z:(-0.445) ();
    if textures then GlTex.coord2 (1.0,0.0) else ();
    GlDraw.vertex ~x:(-0.445) ~y:0.445    ~z:0.445 ();
    if textures then GlTex.coord2 (1.0,1.0) else ();
    GlDraw.vertex ~x:0.445    ~y:0.445    ~z:0.445 ();
    if textures then GlTex.coord2 (0.0,1.0) else ();
    GlDraw.vertex ~x:0.445    ~y:0.445    ~z:(-0.445) ();
    GlDraw.normal ~x:0.0 ~y:0.0 ~z:(-1.0) ();
    if textures then GlTex.coord2 (0.0,0.0) else ();
    GlDraw.vertex ~x:(-0.445) ~y:(-0.445) ~z:(-0.445) ();
    if textures then GlTex.coord2 (1.0,0.0) else ();
    GlDraw.vertex ~x:(-0.445) ~y:0.445    ~z:(-0.445) ();
    if textures then GlTex.coord2 (1.0,1.0) else ();
    GlDraw.vertex ~x:0.445    ~y:0.445    ~z:(-0.445) ();
    if textures then GlTex.coord2 (0.0,1.0) else ();
    GlDraw.vertex ~x:0.445    ~y:(-0.445) ~z:(-0.445) ();
    GlDraw.normal ~x:0.0 ~y:0.0 ~z:1.0 ();
    if textures then GlTex.coord2 (0.0,0.0) else ();
    GlDraw.vertex ~x:(-0.445) ~y:(-0.445) ~z:0.445 ();
    if textures then GlTex.coord2 (1.0,0.0) else ();
    GlDraw.vertex ~x:0.445    ~y:(-0.445) ~z:0.445 ();
    if textures then GlTex.coord2 (1.0,1.0) else ();
    GlDraw.vertex ~x:0.445    ~y:0.445    ~z:0.445 ();
    if textures then GlTex.coord2 (0.0,1.0) else ();
    GlDraw.vertex ~x:(-0.445) ~y:0.445    ~z:0.445 ();
  GlDraw.ends ();

  Gl.enable `cull_face;
  if textures then
    Gl.disable `texture_2d
  else ()



(* initialize stuff before rendering *)
let init () =
  (* set up textures *)
  chargetexid := Tex.loadPPM "Data/charge.ppm";
  if !chargetexid = Nativeint.zero then begin
    prerr_endline ("Problem loading image Data/charge.ppm");
    exit 0
  end else ();
  chargetexgreyid := Tex.loadPPM "Data/charge_grey.ppm";
  if !chargetexgreyid = Nativeint.zero then begin
    prerr_endline ("Problem loading image Data/charge_grey.ppm");
    exit 0
  end else ();
  bribetexid := Tex.loadPPM "Data/bribe.ppm";
  if !bribetexid = Nativeint.zero then begin
    prerr_endline ("Problem loading image Data/bribe.ppm");
    exit 0
  end else ();
  bribetexgreyid := Tex.loadPPM "Data/bribe_grey.ppm";
  if !bribetexgreyid = Nativeint.zero then begin
    prerr_endline ("Problem loading image Data/bribe_grey.ppm");
    exit 0
  end else ();
  teleportertexid := Tex.loadPPM "Data/teleporter.ppm";
  if !teleportertexid = Nativeint.zero then begin
    prerr_endline ("Problem loading image Data/teleporter.ppm");
    exit 0
  end else ();
  teleportertexgreyid := Tex.loadPPM "Data/teleporter_grey.ppm";
  if !teleportertexgreyid = Nativeint.zero then begin
    prerr_endline ("Problem loading image Data/teleporter_grey.ppm");
    exit 0
  end else ();

  (* set up display lists for objects *)
  let plains = GlList.gen_lists ~len:3 in
  plainlists := DisplayLists plains;
  let texs = GlList.gen_lists ~len:3 in
  texlists := DisplayLists texs;
  let plaingreys = GlList.gen_lists ~len:3 in
  plaingreylists := DisplayLists plaingreys;
  let texgreys = GlList.gen_lists ~len:3 in
  texgreylists := DisplayLists texgreys;

  (* Charge *)
  GlList.begins (GlList.nth plains ~pos:0) ~mode:`compile;
  drawcharge 1.0 false false Nativeint.zero;
  GlList.ends ();
  GlList.begins (GlList.nth texs ~pos:0) ~mode:`compile;
  drawcharge 1.0 true false !chargetexid;
  GlList.ends ();
  GlList.begins (GlList.nth plaingreys ~pos:0) ~mode:`compile;
  drawcharge 1.0 false true Nativeint.zero;
  GlList.ends ();
  GlList.begins (GlList.nth texgreys ~pos:0) ~mode:`compile;
  drawcharge 1.0 true true !chargetexgreyid;
  GlList.ends ();

  (* Bribe *)
  GlList.begins (GlList.nth plains ~pos:1) ~mode:`compile;
  drawbribe 1.0 false false Nativeint.zero;
  GlList.ends ();
  GlList.begins (GlList.nth texs ~pos:1) ~mode:`compile;
  drawbribe 1.0 true false !bribetexid;
  GlList.ends ();
  GlList.begins (GlList.nth plaingreys ~pos:1) ~mode:`compile;
  drawbribe 1.0 false true Nativeint.zero;
  GlList.ends ();
  GlList.begins (GlList.nth texgreys ~pos:1) ~mode:`compile;
  drawbribe 1.0 true true !bribetexgreyid;
  GlList.ends ();

  (* Teleporter *)
  GlList.begins (GlList.nth plains ~pos:2) ~mode:`compile;
  drawteleporter 1.0 false false Nativeint.zero;
  GlList.ends ();
  GlList.begins (GlList.nth texs ~pos:2) ~mode:`compile;
  drawteleporter 1.0 true false !teleportertexid;
  GlList.ends ();
  GlList.begins (GlList.nth plaingreys ~pos:2) ~mode:`compile;
  drawteleporter 1.0 false true Nativeint.zero;
  GlList.ends ();
  GlList.begins (GlList.nth texgreys ~pos:2) ~mode:`compile;
  drawteleporter 1.0 true true !teleportertexgreyid;
  GlList.ends ()


(* update the free object given delta_t and a list of collision hulls that
 * it might hit *)
let update ob dt hulls =
  if ob.dmtimer > 0.0 then
    ()   (* dematterized objects don't move *)
  else begin
    let per_sec = 1.0/.dt in
    (* first check if the object is already in collision with a hull, or
     * is resting on top of one *)
    let atrest =
      List.fold_left
        ~f:(fun atrest hull -> atrest || incollision ob.hull hull
                                    || (topcollision ob.hull hull 0.01) < 0.0)
        ~init:false hulls in
    if not atrest then begin
      ob.vy <- max (ob.vy -. (if ob.kind = Teleporter then 1.0 else 5.0)*.dt)
                   (-0.5*.per_sec);
      let (dx,dy,dz) = deflect ob.hull 0.0 (dt*.ob.vy) 0.0 hulls in
      ob.hull.y <- ob.hull.y +. dy
    end else
      ob.vy <- 0.0
  end


(* return the hull-action pair for the given free object *)
let actionpair ob =
  (ob.hull, 
   match ob.kind with
     Charge -> TeleCharge
   | Bribe -> GetBribe
   | Teleporter -> Teleport)


(* render the free object given flags for texture and greyscale modes *)
let render ob textureson greyscaleon =
  if ob.dmtimer > 0.0 then  (* if the object is dematterized *)
    let alpha = 1.0 -. ob.dmtimer in  (* draw it partially transparent *)
    GlMat.push ();
    GlMat.translate ~x:ob.hull.x ~y:ob.hull.y ~z:ob.hull.z ();
    begin match ob.kind with
      Charge ->
        drawcharge alpha textureson greyscaleon
                   (if greyscaleon then !chargetexgreyid else !chargetexid)
    | Bribe ->
        drawbribe alpha textureson greyscaleon
                  (if greyscaleon then !bribetexgreyid else !bribetexid)
    | Teleporter ->
        drawteleporter alpha textureson greyscaleon
              (if greyscaleon then !teleportertexgreyid else !teleportertexid)
    end;
    GlMat.pop ()

  else begin (* otherwise, use display lists to draw the solid object *)
    GlMat.push ();
    GlMat.translate ~x:ob.hull.x ~y:ob.hull.y ~z:ob.hull.z ();

    let DisplayLists dls =
      if textureson then begin
        if greyscaleon then !texgreylists
        else !texlists
      end else begin
        if greyscaleon then !plaingreylists
        else !plainlists
      end in

    begin match ob.kind with
      Charge     -> GlList.call (GlList.nth dls ~pos:0)
    | Bribe      -> GlList.call (GlList.nth dls ~pos:1)
    | Teleporter -> GlList.call (GlList.nth dls ~pos:2)
    end;
    GlMat.pop ()
  end

