(* level.ml: interface for level stuff *)

open Collision

(* either a cube is empty or it is solid, with an array of 6 walls in the
 * order x+, x-, y+, y-, z+, z- *)
type solidity = Empty | Solid of Wall.t array

type cube = {
  mutable w: solidity;
  mutable fixed: Fixedobj.t list;
  mutable free: Freeobj.t list;
  mutable badg: Badguy.t list;
  mutable dmtimer: float;
  mutable scent: float;
}

type t = {
  mutable currtime: float;

  (* the layout of this level *)
  lenx: int;
  leny: int;
  lenz: int;
  layout: cube array; (* actually 3d array, in FORTRAN order *)
  (* lists of dematterized things *)
  mutable dmcubes: cube list;
  mutable dmfixed: Fixedobj.t list;
  mutable dmfree: Freeobj.t list;
  (* a list of ALL the free objects in the level *)
  mutable freeobj: Freeobj.t list;
  (* a list of ALL the badguys in the level *)
  mutable badguys: Badguy.t list;

  (* Basic parts of spiff's initial state for this level *)
  mutable initspiffx: float;
  mutable initspiffy: float;
  mutable initspiffz: float;
  mutable initspiffheading: float;
}


(* Simple primitives *********************************************************)

let getcube lev i j k =
  lev.layout.(i + lev.lenx*(j + lev.leny*k))


let setcube lev i j k c =
  lev.layout.(i + lev.lenx*(j + lev.leny*k)) <- c


(* return whether or not the given cube is partially transparent/permeable *)
let permeable lev i j k =
  let c = getcube lev i j k in
  match c.w with
    Empty ->    true
  | Solid(_) -> c.dmtimer > 0.0  (* true if solid but dematterized *)


(* Basic file stuff **********************************************************)

let protolevel = {
  currtime = 0.0;
  lenx = 0;
  leny = 0;
  lenz = 0;
  layout = [| |];
  dmcubes = [];
  dmfixed = [];
  dmfree = [];
  freeobj = [];
  badguys = [];
  initspiffx = 0.0;
  initspiffy = 0.0;
  initspiffz = 0.0;
  initspiffheading = 0.0;
}

(* find out how many levels there are *)
let numlevels () =
  let i = ref 0 in
  while Sys.file_exists ("level" ^ (string_of_int !i)) do
    incr i
  done;
  if !i <= 0 then begin
    prerr_endline "Couldn't find levels in dungeon directory";
    prerr_endline ("(I'm looking for 'level0' in " ^ (Sys.getcwd ()) ^ ")");
    exit 1
  end else
    !i

(* read a level description from the given file *)
let load_level filename =
  let inchan = open_in_bin filename in
  (Marshal.from_channel inchan : t)

(* read level descriptions from the Dungeon directory *)
let load_newgame () =
  let numlev = numlevels () in
  let dungeon = Array.make numlev protolevel in
  for i = 0 to numlev-1 do
    try dungeon.(i) <- load_level ("level" ^ (string_of_int i))
    with _ -> prerr_endline ("Problem reading level" ^ (string_of_int i)
                             ^ " in " ^ (Sys.getcwd ()));
              exit 1
  done;
  dungeon

(* set the salient fields of spiff to the initial values for this level *)
let init_spiff lev spiff =
  spiff.Spiff.hull.x <- lev.initspiffx;
  spiff.Spiff.hull.y <- lev.initspiffy;
  spiff.Spiff.hull.z <- lev.initspiffz;
  spiff.Spiff.heading <- lev.initspiffheading;
  spiff.Spiff.vx <- 0.0;
  spiff.Spiff.vy <- 0.0;
  spiff.Spiff.vz <- 0.0


(* Updating the level ********************************************************)

(* Update the matterization of all cubes and objects in the level. *)
let updatedm lev spiff dt =
  (* basically, go through the global lists decrementing the dematterization
   * timers, removing any objects from the lists that are now fully
   * matterized again. *)
  lev.dmcubes <- List.fold_left ~init:[] lev.dmcubes
    ~f:(fun l cube -> cube.dmtimer <- cube.dmtimer -. 0.2*.dt;
                      if cube.dmtimer <= 0.0 then begin
                        cube.dmtimer <- 0.0;
                        l
                      end else cube::l);
  lev.dmfixed <- List.fold_left ~init:[] lev.dmfixed
    ~f:(fun l fx -> fx.Fixedobj.dmtimer <- fx.Fixedobj.dmtimer -. 0.2*.dt;
                      if fx.Fixedobj.dmtimer <= 0.0 then begin
                        fx.Fixedobj.dmtimer <- 0.0;
                        l
                      end else fx::l);
  (* free objects rematterize much faster, to allow Spiff to get them... *)
  lev.dmfree <- List.fold_left ~init:[] lev.dmfree
    ~f:(fun l fr -> fr.Freeobj.dmtimer <- fr.Freeobj.dmtimer -. 0.4*.dt;
                      if fr.Freeobj.dmtimer <= 0.0 then begin
                        fr.Freeobj.dmtimer <- 0.0;
                        l
                      end else fr::l)


(* returns a list of cubes within inf-norm 0.5 of a given hull, and a
 * matching list of things to avoid (walls, fixed objects) *)
let cubelist lev hull =
  let starti = int_of_float (hull.x -. hull.lx -. 0.5)
  and endi   = int_of_float (hull.x +. hull.lx +. 0.5)
  and startj = int_of_float (hull.y -. hull.ly -. 0.5)
  and endj   = int_of_float (hull.y +. hull.ly +. 0.5)
  and startk = int_of_float (hull.z -. hull.lz -. 0.5)
  and endk   = int_of_float (hull.z +. hull.lz +. 0.5) in
  let lc = ref []
  and al = ref [] in
  for i = starti to endi do
    let cx = 0.5 +. float_of_int i in
    for j = startj to endj do
      let cy = 0.5 +. float_of_int j in
      for k = startk to endk do
        let cz = 0.5 +. float_of_int k in
        lc := (getcube lev i j k)::!lc;
        if not (permeable lev i j k) then
          al := {x=cx;y=cy;z=cz;lx=0.5;ly=0.5;lz=0.5}::!al
        else ()
      done
    done
  done;
  let localcubes = !lc in
  let avoidlist =
    List.fold_left ~f:(fun l cube ->
                         List.fold_left ~f:(fun l fx ->
                                             if fx.Fixedobj.dmtimer = 0.0 then
                                               fx.Fixedobj.hull::l
                                             else l)
                                        ~init:l cube.fixed)
                   ~init:!al localcubes in
  (localcubes, avoidlist)


(* move free objects in the level *)
let movefreeobj lev dt =
  List.iter lev.freeobj ~f:(fun fr ->
    let i = int_of_float fr.Freeobj.hull.x
    and j = int_of_float fr.Freeobj.hull.y
    and k = int_of_float fr.Freeobj.hull.z in

    (* we have to assemble a list of possibly interacting things *)
    let (localcubes,avoidlist) = cubelist lev fr.Freeobj.hull in
    (* what about dematterized free objects??? *)
    let frlist = List.fold_left ~f:(fun l c -> List.rev_append c.free l)
                                ~init:[] localcubes in
    let hulllist = List.rev_append
      (List.map ~f:(fun fr2 -> fr2.Freeobj.hull) frlist) avoidlist in

    (* ask the object to move itself appropriately *)
    Freeobj.update fr dt hulllist;

    (* and then check if it moved to a different cube *)
    let newi = int_of_float fr.Freeobj.hull.x
    and newj = int_of_float fr.Freeobj.hull.y
    and newk = int_of_float fr.Freeobj.hull.z in
    if newi != i || newj != j || newk != k then begin
      (* we have to move bg from the current cube to the new cube's list *)
      let c = getcube lev i j k in
      c.free <- List.fold_left ~f:(fun l f -> if f == fr then l else f::l)
                               ~init:[] c.free;
      let newc = getcube lev newi newj newk in
      newc.free <- fr::newc.free
    end else ()
  )


(* move badguys in the level *)
let movebadguys lev spiff dt =
  List.iter lev.badguys ~f:(fun bg ->
    (* we have to assemble a list of possible interacting hulls *)
    let (localcubes,avoidlist) = cubelist lev bg.Badguy.hull in
    let bglist = List.fold_left ~f:(fun l c -> List.rev_append c.badg l)
                                ~init:[] localcubes in
    let avoidlist = List.rev_append
      (List.map ~f:(fun bg2 -> bg2.Badguy.hull) bglist) avoidlist in
    (* we also will pass in the scent pairs for nearby cubes *)
    let i = int_of_float bg.Badguy.hull.x
    and j = int_of_float bg.Badguy.hull.y
    and k = int_of_float bg.Badguy.hull.z in
    let scents = [|
      (getcube lev (i-1) j (k-1)).scent;
      (getcube lev (i-1) j k).scent;
      (getcube lev (i-1) j (k+1)).scent;
      (getcube lev i j (k-1)).scent;
      (getcube lev i j k).scent;
      (getcube lev i j (k+1)).scent;
      (getcube lev (i+1) j (k-1)).scent;
      (getcube lev (i+1) j k).scent;
      (getcube lev (i+1) j (k+1)).scent |] in
    Badguy.update bg dt avoidlist scents;
    let newi = int_of_float bg.Badguy.hull.x
    and newj = int_of_float bg.Badguy.hull.y
    and newk = int_of_float bg.Badguy.hull.z in
    if newi != i || newj != j || newk != k then begin
      (* we have to move bg from the current cube to the new cube's list *)
      let c = getcube lev i j k in
      c.badg <- List.fold_left ~f:(fun l b -> if b == bg then l else b::l)
                               ~init:[] c.badg;
      let newc = getcube lev newi newj newk in
      newc.badg <- bg::newc.badg
    end else ();

    (* update Badguy's scent in local cubes *)
    List.iter ~f:(fun cube -> cube.scent <- cube.scent -.0.1*.dt*.lev.currtime)
              localcubes;
  )


exception FoundFree of Freeobj.t

(* move Spiff, returning actions that are incompletely handled *)
let movespiff lev spiff dt =
  (* get lists of nearby things *)
  let (localcubes,avoidlist) = cubelist lev spiff.Spiff.hull in
  let frlist = List.fold_left ~f:(fun l c -> List.rev_append c.free l)
                                ~init:[] localcubes
  and bglist = List.fold_left ~f:(fun l c -> List.rev_append c.badg l)
                              ~init:[] localcubes in
  let actlist = List.rev_append (List.map ~f:Badguy.actionpair bglist) 
                                (List.map ~f:Freeobj.actionpair frlist) in

  (* update Spiff's scent in local cubes *)
  List.iter ~f:(fun cube -> cube.scent <- cube.scent +. dt*.lev.currtime)
            localcubes;
  
  let action = Spiff.update spiff dt avoidlist actlist in
  match action with
    TeleCharge -> begin
      (* HACK: we'll delete the first telecharge in the list *)
      (* Really should just be the one that Spiff collided with *)
      try List.iter ~f:(fun fr -> if fr.Freeobj.kind = Freeobj.Charge then
                                    raise (FoundFree fr)
                                  else ()) frlist;
      with FoundFree(charge) -> begin
        (* Remove from global list of free objects *)
        lev.freeobj <- List.fold_left ~init:[] lev.freeobj
                       ~f:(fun l fr -> if (fr = charge) then l else fr::l);
        (* Remove from nearby cubes. Note we require a free object to appear *)
        (* in only one cube -- otherwise this might not eliminate all copies *)
        List.iter ~f:(fun cube ->
                        cube.free <- List.fold_left ~init:[] cube.free
                          ~f:(fun l fr -> if (fr = charge) then l else fr::l))
                  localcubes
      end; end;
      DoNothing

  | GetBribe -> begin
      (* HACK: we'll delete the first bribe in the list *)
      (* Really should just be the one that Spiff collided with *)
      try List.iter ~f:(fun fr -> if fr.Freeobj.kind = Freeobj.Bribe then
                                    raise (FoundFree fr)
                                  else ()) frlist;
      with FoundFree(bribe) -> begin
        (* Remove from global list of free objects *)
        lev.freeobj <- List.fold_left ~init:[] lev.freeobj
                       ~f:(fun l fr -> if (fr = bribe) then l else fr::l);
        (* Remove from nearby cubes. Note we require a free object to appear *)
        (* in only one cube -- otherwise this might not eliminate all copies *)
        List.iter ~f:(fun cube ->
                        cube.free <- List.fold_left ~init:[] cube.free
                          ~f:(fun l fr -> if (fr = bribe) then l else fr::l))
                  localcubes
      end; end;
      DoNothing

  | LoseChance -> (* game isn't lost, but we get sent back to start of level *)
      init_spiff lev spiff;
      LoseChance

  | _ -> action


let update lev spiff dt =
  lev.currtime <- lev.currtime +. dt;
  (* update matterization *)
  updatedm lev spiff dt;
  (* update free objects *)
  movefreeobj lev dt;
  (* update badguys *)
  movebadguys lev spiff dt;
  (* let Spiff try to move, and return the resulting action *)
  movespiff lev spiff dt
  

(* dematterization ***********************************************************)

(* attempt to dematterize stuff intersection the given hull *)
let dematterize lev dmhull =
  (* first seelct just one cube that may be dematterized *)
  let i = int_of_float dmhull.x
  and j = int_of_float dmhull.y
  and k = int_of_float dmhull.z in
  (* we have to make sure that Spiff cannot make a hole through the
     outer limits of the dungeon. *)
  let i = if i < 1 then 1 else if i > lev.lenx-2 then lev.lenx-2 else i
  and j = if j < 1 then 1 else if j > lev.leny-2 then lev.leny-2 else j
  and k = if k < 1 then 1 else if k > lev.lenz-2 then lev.lenz-2 else k in
  (* dematterize cube i j k *)
  let cube = getcube lev i j k in
  if cube.dmtimer = 0.0 then
    lev.dmcubes <- cube::lev.dmcubes
  else ();
  cube.dmtimer <- 1.0;
  (* and its contents *)
  lev.dmfixed <-
    List.fold_left
      ~f:(fun l fx -> let returnval =
                        if fx.Fixedobj.dmtimer = 0.0 then fx::l else l in
                      fx.Fixedobj.dmtimer <- 1.0;
                      returnval)
      ~init:lev.dmfixed
      cube.fixed;
  lev.dmfree <-
    List.fold_left
      ~f:(fun l fr -> let returnval =
                        if fr.Freeobj.dmtimer = 0.0 then fr::l else l in
                      fr.Freeobj.dmtimer <- 1.0;
                      returnval)
      ~init:lev.dmfree
      cube.free;
  (* also dematterize contents of neighbouring cubes that collide with hull *)
  let (localcubes,_) = cubelist lev dmhull in
  List.iter localcubes
    ~f:(fun cube ->
          lev.dmfixed <-
            List.fold_left
              ~f:(fun l fx ->
                    if incollision fx.Fixedobj.hull dmhull then begin
                      let returnval =
                        if fx.Fixedobj.dmtimer = 0.0 then fx::l else l in
                        fx.Fixedobj.dmtimer <- 1.0;
                        returnval
                    end else l)
              ~init:lev.dmfixed
              cube.fixed;

          lev.dmfree <-
            List.fold_left
              ~f:(fun l fr ->
                    if incollision fr.Freeobj.hull dmhull then begin
                      let returnval =
                        if fr.Freeobj.dmtimer = 0.0 then fr::l else l in
                        fr.Freeobj.dmtimer <- 1.0;
                        returnval
                    end else l)
              ~init:lev.dmfree
              cube.free;
    )


(* Rendering *****************************************************************)

(* we will assemble lists of things to render *)
type renderthing = RenderCube of (int*int*int) | RenderFixed of Fixedobj.t
                 | RenderFree of Freeobj.t | RenderBadguy of Badguy.t


(* queue of cube coordinates for deciding what to render *)
type cubequeue = {
  mutable qhead: int;
  mutable qtail: int;
  qarray: (int * int * int) array;
}

let makequeue maxlen =
  { qhead = 0;
    qtail = 0;
    qarray = Array.make maxlen (0,0,0) }

let removequeue q =
  let e = q.qarray.(q.qhead) in
  q.qhead <- (q.qhead+1) mod (Array.length q.qarray);
  e

let addqueue q newentries =
  List.iter ~f:(fun e -> q.qarray.(q.qtail) <- e;
                         q.qtail <- (q.qtail+1) mod (Array.length q.qarray))
            newentries

let emptyqueue q =
  q.qhead = q.qtail


(* status of a cube during visibility algorithm *)
type visiblestatus = VisUnknown | MaybeVisible | Visible | Clipped | Occluded

(* helper functions for culling *)
let ifnotclipped vf =
  vf = Visible || vf = Occluded

let ifoccluding lev visflag i j k =
  let n = i + lev.lenx*(j + lev.leny*k) in
  visflag.(n) = Clipped ||
  visflag.(n) = Occluded ||
  (visflag.(n) = Visible && not (permeable lev i j k))

(* check whether or not the given point xyz is inside the view frustum *)
let checkfrustum x y z sinhead coshead sinpitch cospitch l r b t n f =
  (* we assume xyz have already been translated so eye is at origin *)
  (* rotate corner according to heading *)
  let x =   coshead*.x +. sinhead*.z
  and z = -.sinhead*.x +. coshead*.z in
  (* rotate corner according to pitch *)
  let y = cospitch*.y -. sinpitch*.z
  and z = sinpitch*.y +. cospitch*.z in
  (* check for sign of z *)
  if -2.0*.f*.n <= (f+.n)*.z  then
    false
  else begin
    (* now check x *)
    let x = 2.0*.n*.x +. (r+.l)*.z and w = r-.l in
    if x <= -.w || x >= w then
      false
    else begin
      (* now check y *)
      let y = 2.0*.n*.y +. (t+.b)*.z and h = t-.b in
      if y <= -.h || y >= h then
        false
      else
        true
    end
  end

(* check whether or not the given cube ijk intersects the view frustum *)
let insidefrustum i j k eyex eyey eyez sh ch sp cp l r b t n f =
  (* translate corner according to eye *)
  let x0 = (float i) -. eyex
  and y0 = (float j) -. eyey
  and z0 = (float k) -. eyez in
  let x1 = x0 +. 1.0
  and y1 = y0 +. 1.0
  and z1 = z0 +. 1.0 in
  (* Since no cube is going to completely span the frustum, it suffices to
   * just check the corners *)
  checkfrustum x0 y0 z0 sh ch sp cp l r b t n f ||
  checkfrustum x1 y1 z1 sh ch sp cp l r b t n f ||
  checkfrustum x1 y0 z0 sh ch sp cp l r b t n f ||
  checkfrustum x0 y1 z1 sh ch sp cp l r b t n f ||
  checkfrustum x0 y1 z0 sh ch sp cp l r b t n f ||
  checkfrustum x1 y0 z1 sh ch sp cp l r b t n f ||
  checkfrustum x0 y0 z1 sh ch sp cp l r b t n f ||
  checkfrustum x1 y1 z0 sh ch sp cp l r b t n f


(* do a partial BFS from the given queue to find things to render, given
 * partial rendering lists already. The reglist will contain all the
 * solid objects in approximate (up to cube resolution) back-to-front order,
 * and dmlist will contain all the partially transparent objects in
 * approximate (up to cube resolution) back-to-front order. *)
let rec visbfs lev eyex eyey eyez sinhead coshead sinpitch cospitch
           vleft vright vbot vtop vnear vfar todoqueue reglist dmlist visflag =
  if emptyqueue todoqueue then
    (reglist,dmlist)  (* no more cubes in queue to process *)

  else begin
    (* get the first cube from the queue *)
    let (i,j,k) = removequeue todoqueue in
    let n = i + lev.lenx*(j + lev.leny*k) in

    (* find out which faces of the cube are facing the viewer *)
    let xp = (eyex > float (i+1)) && i < lev.lenx-1
    and xm = (eyex < float i)     && i > 0
    and yp = (eyey > float (j+1)) && j < lev.leny-1
    and ym = (eyey < float j)     && j > 0
    and zp = (eyez > float (k+1)) && k < lev.lenz-1
    and zm = (eyez < float k)     && k > 0 in
    let numvis = (if xp then 1 else 0) + (if xm then 1 else 0) +
                 (if yp then 1 else 0) + (if ym then 1 else 0) +
                 (if zp then 1 else 0) + (if zm then 1 else 0) in

    (* for each visible face, check if the adjoining cube has already
     * passed clipping test *)
    let dj = lev.lenx and dk = lev.lenx*lev.leny in
    let nbrvis = 
      (if xp && ifnotclipped visflag.(n+1) then 1 else 0) +
      (if xm && ifnotclipped visflag.(n-1) then 1 else 0) +
      (if yp && ifnotclipped visflag.(n+dj) then 1 else 0) +
      (if ym && ifnotclipped visflag.(n-dj) then 1 else 0) +
      (if zp && ifnotclipped visflag.(n+dk) then 1 else 0) +
      (if zm && ifnotclipped visflag.(n-dk) then 1 else 0) in

    (* now decide if this cube is clipped *)
    if nbrvis < numvis && not
       (insidefrustum i j k eyex eyey eyez sinhead coshead sinpitch cospitch
                      vleft vright vbot vtop vnear vfar)
    then
      visflag.(n) <- Clipped
    else begin
      (* and if it isn't, check if it's occluded *)
      let numocc =
        (if xp && ifoccluding lev visflag (i+1) j k then 1 else 0) +
        (if xm && ifoccluding lev visflag (i-1) j k then 1 else 0) +
        (if yp && ifoccluding lev visflag i (j+1) k then 1 else 0) +
        (if ym && ifoccluding lev visflag i (j-1) k then 1 else 0) +
        (if zp && ifoccluding lev visflag i j (k+1) then 1 else 0) +
        (if zm && ifoccluding lev visflag i j (k-1) then 1 else 0) in
      if numvis > 0 && numocc = numvis then
        visflag.(n) <- Occluded
      else
        visflag.(n) <- Visible
    end;

    (* if cube isn't clipped, add appropriate neighbours to queue *)
    if visflag.(n) != Clipped then begin
      (* add appropriate adjoining cubes to todoqueue *)
      if (not xp) && i < lev.lenx-1 && visflag.(n+1) = VisUnknown then begin
        visflag.(n+1) <- MaybeVisible;
        addqueue todoqueue [(i+1,j,k)]
      end else ();
      if (not xm) && i > 0          && visflag.(n-1) = VisUnknown then begin
        visflag.(n-1) <- MaybeVisible;
        addqueue todoqueue [(i-1,j,k)]
      end else ();
      if (not yp) && j < lev.leny-1 && visflag.(n+dj) = VisUnknown then begin
        visflag.(n+dj) <- MaybeVisible;
        addqueue todoqueue [(i,j+1,k)]
      end else ();
      if (not ym) && j > 0          && visflag.(n-dj) = VisUnknown then begin
        visflag.(n-dj) <- MaybeVisible;
        addqueue todoqueue [(i,j-1,k)]
      end else ();
      if (not zp) && k < lev.lenz-1 && visflag.(n+dk) = VisUnknown then begin
        visflag.(n+dk) <- MaybeVisible;
        addqueue todoqueue [(i,j,k+1)]
      end else ();
      if (not zm) && k > 0          && visflag.(n-dk) = VisUnknown then begin
        visflag.(n-dk) <- MaybeVisible;
        addqueue todoqueue [(i,j,k-1)]
      end else ();
    end;

    (* finally, if the cube is visible add to the rendering lists *)
    if visflag.(n) = Visible then begin    
      (* figure out what to add to the rendering lists *)
      let c = lev.layout.(n) in
      let reglist = 
        if c.dmtimer = 0.0 && c.w != Empty then
          (RenderCube (i,j,k))::reglist
        else begin
          (* add to reglist all the fully solid objects contained in c *)
          let reglist = List.fold_left
            ~f:(fun l fx ->
                  if fx.Fixedobj.dmtimer = 0.0 then (RenderFixed fx)::l
                  else l)
            ~init:reglist c.fixed in
          let reglist = List.fold_left
            ~f:(fun l fr ->
                  if fr.Freeobj.dmtimer = 0.0 then (RenderFree fr)::l
                  else l)
            ~init:reglist c.free in
          List.fold_left
            ~f:(fun l bg -> (RenderBadguy bg)::l)
            ~init:reglist c.badg
        end in

      let dmlist = 
        if c.dmtimer > 0.0 && c.w != Empty then
          (RenderCube (i,j,k))::dmlist
        else dmlist in
      let dmlist =
        if c.dmtimer > 0.0 || c.w = Empty then begin
          (* add in all semitransparent objects contained in c *)
          let dmlist = List.fold_left
            ~f:(fun l fx ->
                  if fx.Fixedobj.dmtimer > 0.0 then (RenderFixed fx)::l
                  else l)
            ~init:dmlist c.fixed in
          List.fold_left
            ~f:(fun l fr ->
                  if fr.Freeobj.dmtimer > 0.0 then (RenderFree fr)::l
                  else l)
            ~init:dmlist c.free
        end else
          dmlist in

      (* tail-recurse on rest of queue with the new additions *)
      visbfs lev eyex eyey eyez sinhead coshead sinpitch cospitch
             vleft vright vbot vtop vnear vfar todoqueue reglist dmlist visflag
    end else
      (* if not visible, just continue with rest of queue *)
      visbfs lev eyex eyey eyez sinhead coshead sinpitch cospitch
             vleft vright vbot vtop vnear vfar todoqueue reglist dmlist visflag
  end


(* Return two lists of cubes to be rendered (with their contents). The first
 * is fully solid, sorted by depth with the head of the list the closest to 
 * the eye. The second is a list of dematterized cubes, sorted the opposite
 * way (the head is furthest). This does view frustum culling and occlusion
 * culling.
 *)
let sortcubes lev eyex eyey eyez sinhead coshead sinpitch cospitch
              vleft vright vbot vtop vnear vfar =
  (* create an array of visibility flags for use in the algorithm *)
  let visflag = Array.make (lev.lenx*lev.leny*lev.lenz) VisUnknown in
  (* initialize a queue of cubes to check with the cube containing the eye *)
  let i = int_of_float eyex
  and j = int_of_float eyey
  and k = int_of_float eyez in
  let todoqueue = makequeue (lev.lenx*lev.leny*lev.lenz) in
  addqueue todoqueue [(i,j,k)];
  visflag.(i+lev.lenx*(j+lev.leny*k)) <- Visible;
  (* finally do the partial BFS to find visible things *) 
  let (reglist,dmlist) =
    visbfs lev eyex eyey eyez sinhead coshead sinpitch cospitch
           vleft vright vbot vtop vnear vfar todoqueue [] [] visflag in
  (* Reverse the regular list to get it in front-to-back order *)
  (* We do this to hopefully improve performance - less computation should
   * be performed for far-away pixels that fail the depth test *)
  (List.rev reglist, dmlist)


(* draw the walls of a cube, possible dematterized *)
let drawcube lev textureson stereoon (i,j,k) =
  let c = getcube lev i j k in 
  begin match c.w with
    Empty -> ()
  | Solid(walls) -> begin
      let alpha = 1.0 -. c.dmtimer in
      GlMat.push ();
      GlMat.translate ~x:(float i) ~y:(float j) ~z:(float k) ();
      GlMat.push ();
      GlMat.translate ~x:1.0 ~y:0.5 ~z:0.5 ();
      GlMat.rotate ~angle:180.0 ~y:1.0 ();
      Wall.render walls.(0) textureson stereoon alpha;
      GlMat.pop ();
      GlMat.push ();
      GlMat.translate ~x:0.0 ~y:0.5 ~z:0.5 ();
      Wall.render walls.(1) textureson stereoon alpha;
      GlMat.pop ();
      GlMat.push ();
      GlMat.translate ~x:0.5 ~y:1.0 ~z:0.5 ();
      GlMat.rotate ~angle:270.0 ~z:1.0 ();
      Wall.render walls.(2) textureson stereoon alpha;
      GlMat.pop ();
      GlMat.push ();
      GlMat.translate ~x:0.5 ~y:0.0 ~z:0.5 ();
      GlMat.rotate ~angle:90.0 ~z:1.0 ();
      Wall.render walls.(3) textureson stereoon alpha;
      GlMat.pop ();
      GlMat.push ();
      GlMat.translate ~x:0.5 ~y:0.5 ~z:1.0 ();
      GlMat.rotate ~angle:90.0 ~y:1.0 ();
      Wall.render walls.(4) textureson stereoon alpha;
      GlMat.pop ();
      GlMat.push ();
      GlMat.translate ~x:0.5 ~y:0.5 ~z:0.0 ();
      GlMat.rotate ~angle:270.0 ~y:1.0 ();
      Wall.render walls.(5) textureson stereoon alpha;
      GlMat.pop ();
      GlMat.pop ()
    end
  end


(* render the current state of the level *)
let render lev spiff w h b lightingon textureson stereoon dx =
  GlDraw.viewport ~x:0 ~y:b ~w:(w-b) ~h:(h-b);
  Gl.enable `depth_test;
  Gl.enable `cull_face;
  if lightingon then begin
    Gl.enable `lighting;
    GlLight.light_model (`ambient(0.1,0.1,0.1,1.0));
    GlLight.light ~num:0 (`ambient(0.9,0.9,0.9,1.0));
    GlLight.light ~num:0 (`diffuse(1.0,1.0,1.0,1.0));
    GlLight.light ~num:0 (`specular(1.0,1.0,1.0,1.0));
    Gl.enable `light0;
  end else ();
  GlLight.fog (`mode(`exp));
  GlLight.fog (`density(0.4));
  Gl.enable `fog;
  GlDraw.shade_model `smooth;

  let (eyex, eyey, eyez, eyehead, eyepitch) = Spiff.get_eye spiff in
  let sinhead = sin (0.017453292 *. eyehead)
  and coshead = cos (0.017453292 *. eyehead)
  and sinpitch = sin (0.017453292 *. eyepitch)
  and cospitch = cos (0.017453292 *. eyepitch) in
  let eyex = eyex -. dx *. coshead
  and eyez = eyez -. dx *. sinhead in

  GlMat.mode `projection;
  GlMat.load_identity ();
  let aspect = (float_of_int h) /. (float_of_int w) in
  let vl = 0.1*.dx-.0.029
  and vr = 0.1*.dx+.0.029
  and vb = -.aspect*.0.029
  and vt = aspect*.0.029
  and vn = 0.03
  and vf = 90.0 in
  GlMat.frustum ~x:(vl,vr) ~y:(vb,vt) ~z:(vn,vf);

  GlMat.mode `modelview;
  GlMat.load_identity ();

  GlMat.rotate ~angle:eyepitch ~x:1.0 ();
  if lightingon then
    GlLight.light ~num:0 (`position(0.0,0.0,1.0,0.0))
  else ();
  GlMat.rotate ~angle:eyehead ~y:1.0 ();
  GlMat.translate ~x:(-.eyex) ~y:(-.eyey) ~z:(-.eyez) ();

  (* find lists of everything to be rendered, sorted appropriately *)
  let (reglist,dmlist) =
    sortcubes lev eyex eyey eyez sinhead coshead sinpitch cospitch
              vl vr vb vt vn vf in

  (* first render the solid stuff *)
  List.iter ~f:(fun r -> match r with
                  RenderCube ijk  -> drawcube lev textureson stereoon ijk
                | RenderFixed fx  -> Fixedobj.render fx textureson stereoon
                | RenderFree fr   -> Freeobj.render fr textureson stereoon
                | RenderBadguy bg -> Badguy.render bg textureson stereoon
               ) reglist;

  (* now render the dematterized stuff *)
  GlFunc.depth_mask false;  (* make depth buffer read-only *)
  Gl.enable `blend;            (* turn on blending *)
  GlFunc.blend_func ~src:`src_alpha ~dst:`one_minus_src_alpha;
  List.iter ~f:(fun r -> match r with
                  RenderCube ijk -> drawcube lev textureson stereoon ijk
                | RenderFixed fx -> Fixedobj.render fx textureson stereoon
                | RenderFree fr  -> Freeobj.render fr textureson stereoon
                | _ -> ()) dmlist;

  Gl.disable `depth_test;
  Gl.disable `cull_face;
  if lightingon then begin
    Gl.disable `light0;
    Gl.disable `lighting
  end else ();
  Gl.disable `fog;

  (* and finally dim screen if we're inside a solid dematterized cube *)
  let c = getcube lev (int_of_float eyex)
                      (int_of_float eyey)
                      (int_of_float eyez) in
  if c.w != Empty && c.dmtimer > 0.0 then begin
    GlMat.mode `modelview;
    GlMat.load_identity ();
    GlMat.mode `projection;
    GlMat.load_identity ();
    GlMat.ortho ~x:(0.0,1.0) ~y:(0.0,1.0) ~z:(-1.0,1.0);
    GlDraw.color ~alpha:(1.0 -. c.dmtimer) (0.0,0.0,0.0);
    GlDraw.begins `quads;
    GlDraw.vertex ~x:0.0 ~y:0.0 ();
    GlDraw.vertex ~x:1.0 ~y:0.0 ();
    GlDraw.vertex ~x:1.0 ~y:1.0 ();
    GlDraw.vertex ~x:0.0 ~y:1.0 ();
    GlDraw.ends ();
  end else ();
  GlFunc.depth_mask true;  (* restore normal operations *)
  Gl.disable `blend


