(* spiff.ml: implementation for Spiff's update and on-screen control
 * rendering *)

open Collision


(* where Spiff is trying to run (in his local coordinate system: North is
 * straight ahead, whatever direction that may be in the level) *)
type acceldir =
  Dir0 | DirN | DirNE | DirE | DirSE | DirS | DirSW | DirW | DirNW

type t = {
  hull: hull;
  mutable gait: float;  (* not currently used... *)
  mutable vx: float;
  mutable vy: float;
  mutable vz: float;
  mutable dir: acceldir;
  mutable jump: bool;
  mutable heading: float;
  mutable pitch: float;
  mutable mouse_sensitivity: float;

  mutable chances: int;
  mutable dmcharge: float;
  mutable telecharge: int;
}


(* Return default new Spiff *)
let newspiff () =
  { hull = {x=0.0; y=0.0; z=0.0; lx=0.08; ly=0.3; lz=0.08};
    gait = 0.0;
    vx = 0.0;
    vy = 0.0;
    vz = 0.0;
    dir = Dir0;
    jump = false;
    heading = 0.0;
    pitch = 0.0;
    mouse_sensitivity = 0.15;
    chances = 5;
    dmcharge = 1.0;
    telecharge = 0;}


(* updates Spiff's state given delta time, a list of collision hulls to
 * avoid (walls and solid objects) and a list of (collision hull, action)
 * pairs. *)
let update s dt avoidlist actionlist =
  (* update dematterizer charge *)
  s.dmcharge <-  min (s.dmcharge +. 0.07 *. dt) 1.0;

  (* figure out motion *)

  let per_sec = 1.0/.dt in

  (* first check if Spiff has his feet on top of anything *)
  let h_onfloor =
    List.fold_left
      ~f:(fun h ob -> min h (topcollision s.hull ob 0.01))
      ~init:1.0 avoidlist in

  (* handle horizontal motion - even if Spiff doesn't have purchase, he can
   * change his velocity just a little bit *)
  let decay = if h_onfloor < 0.0 then 0.01*.per_sec
                                 else 0.99 in
  let accel = if h_onfloor < 0.0 then 100.0 *. dt
                                 else dt in
  let radians = 0.017453293 *. s.heading in
  let headx = sin radians and headz = -.cos radians in
  begin match s.dir with
    Dir0  -> ()
  | DirN  ->   s.vx <- s.vx +. accel*.headx;
               s.vz <- s.vz +. accel*.headz;
  | DirNE ->   s.vx <- s.vx +. accel*.0.65*.(headx-.headz);
               s.vz <- s.vz +. accel*.0.65*.(headx+.headz);
  | DirNW ->   s.vx <- s.vx +. accel*.0.65*.(headx+.headz);
               s.vz <- s.vz -. accel*.0.65*.(headx-.headz);
  | DirE  ->   s.vx <- s.vx -. accel*.0.75*.headz;
               s.vz <- s.vz +. accel*.0.75*.headx;
  | DirW  ->   s.vx <- s.vx +. accel*.0.75*.headz;
               s.vz <- s.vz -. accel*.0.75*.headx;
  | DirSE ->   s.vx <- s.vx +. accel*.0.45*.(-.headz-.headx);
               s.vz <- s.vz +. accel*.0.45*.(-.headz+.headx);
  | DirSW ->   s.vx <- s.vx -. accel*.0.45*.(-.headz+.headx);
               s.vz <- s.vz +. accel*.0.45*.(-.headz-.headx);
  | DirS  ->   s.vx <- s.vx -. accel*.0.5*.headx;
               s.vz <- s.vz -. accel*.0.5*.headz;
  end;
  s.vx <- s.vx *. decay;
  s.vz <- s.vz *. decay;

  (* handle vertical motion *)
  if h_onfloor < 0.0 then begin
    s.vy <- if s.jump then 2.4 else 0.0;
    if h_onfloor < -.0.005 then   (* make sure Spiff doesn't hit the ground *)
      s.hull.y <- s.hull.y +. 0.005
    else ()
  end else
    s.vy <- s.vy -. 5.0*.dt;

  (* now clamp velocity to make sure bad things don't happen *)
  if s.vx < -.0.5*.per_sec then s.vx <- -0.5*.per_sec
  else if s.vx > 0.5*.per_sec then s.vx <- 0.5*.per_sec else ();
  if s.vy < -.0.5*.per_sec then s.vy <- -0.5*.per_sec
  else if s.vy > 0.5*.per_sec then s.vy <- 0.5*.per_sec else ();
  if s.vz < -.0.5*.per_sec then s.vz <- -0.5*.per_sec
  else if s.vz > 0.5*.per_sec then s.vz <- 0.5*.per_sec else ();

  (* check if Spiff is in collision with a solid object (that just
   * rematterized) and if so then LoseChance or LoseGame *)
  if List.fold_left ~f:(fun incol hull -> incol || incollision hull s.hull)
                    ~init:false avoidlist then begin
    s.chances <- max 0 (s.chances-1);
    if s.chances = 0 then
      LoseGame
    else
      LoseChance
  end else begin

    (* check for collisions we're supposed to avoid *)
    let (dx,dy,dz) =
      deflect s.hull (dt*.s.vx) (dt*.s.vy) (dt*.s.vz) avoidlist in

    (* update velocity and position *)
    s.vx <- dx *. per_sec;
    s.vy <- dy *. per_sec;
    s.vz <- dz *. per_sec;
    s.hull.x <- s.hull.x +. dx;
    s.hull.y <- s.hull.y +. dy;
    s.hull.z <- s.hull.z +. dz;

    (* now check for collisions demanding special action *)
    let act = 
      List.fold_left
        ~f:(fun act (ob2,act2) ->
              if incollision s.hull ob2 then
                match act, act2 with
                  TeleCharge, _ -> TeleCharge   (* first priority *)
                | _, TeleCharge -> TeleCharge
                | _, Teleport   -> Teleport     (* second priority *)
                | Teleport, _   -> Teleport
                | GetBribe, _   -> GetBribe     (* third priority *)
                | _, GetBribe   -> GetBribe
                | _             -> act2
              else act)
        ~init:DoNothing actionlist in
    match act with
      TeleCharge -> (s.telecharge <- s.telecharge + 1; act)
    | GetBribe -> (s.chances <- s.chances + 1; act)
    | Teleport -> (if s.telecharge < 9 then DoNothing
                   else (s.telecharge <- 0; act))
    | LoseChance -> (s.chances <- max 0 (s.chances-1);
                     if s.chances = 0 then LoseGame else LoseChance)
    | _ -> act
  end


(* returns a collision hull for the blast area of the dematterizer *)
let dematterhull s =
  let x = s.hull.x  and y = s.hull.y  and z = s.hull.z in
  let prad = 0.017453293 *. s.pitch in
  let dy = -. sin prad   and c = cos prad in
  let hrad = 0.017453293 *. s.heading in
  let dx = c *. sin hrad and dz = -.c *. cos hrad in
  (* we now have a unit norm vector pointing in direction Spiff is looking *)
  (* However, we want to make it hard to dematterize the floor beneath our
   * feet unless we're really trying to do that. Worry about that later *)
  {x=x+.0.6*.dx; y=y+.0.6*.dy; z=z+.0.6*.dz; lx=0.4; ly=0.4; lz=0.4}


(* initialize stuff before rendering *)
let texid = ref Nativeint.zero
and texgreyid = ref Nativeint.zero

let init () =
  (* load in Spiff image for the chances display *)
  texid := Tex.loadPPM "Data/spiff.ppm";
  if !texid = Nativeint.zero then begin
    prerr_endline ("Problem loading image Data/spiff.ppm");
    exit 0
  end else ();
  texgreyid := Tex.loadPPM "Data/spiff_grey.ppm";
  if !texgreyid = Nativeint.zero then begin
    prerr_endline ("Problem loading image Data/spiff_grey.ppm");
    exit 0
  end else ()


(* return position and direction of Spiff's eye *)
let get_eye s =
  (s.hull.x, s.hull.y +. 0.22, s.hull.z, s.heading, s.pitch)


(* translate modelview to Spiff's eye position *)
let transform_to_eye s =
  GlMat.rotate ~angle:s.pitch ~x:1.0 ();
  GlMat.rotate ~angle:s.heading ~y:1.0 ();
  GlMat.translate ~x:(-.s.hull.x)
                  ~y:(-.(s.hull.y +. 0.22))
                  ~z:(-.s.hull.z) ()


(* called after the 3d scene has been rendered, this draws
 * the dematterizer charge, chances, and teleporter charge indicators.
 * pass in screen dimensions and whether or not this in stereo mode *)
let render s w h b stereoon =
  (* render Spiff's feet, dematterizer, etc. in 3d view *)
  (* GlDraw.viewport ~x:0 ~y:b ~w:(w-b) ~h:(h-b); *)
  (* eventually want to add a picture of the dematterizer, and Spiff's
   * feet depending on the pitch *)

  (* render dematterizer charge indicator *)
  GlDraw.viewport ~x:0 ~y:0 ~w:(w-b) ~h:b;
  GlMat.mode `modelview;
  GlMat.load_identity ();
  GlMat.mode `projection;
  GlMat.load_identity ();
  GlMat.ortho ~x:(0.0,19.0) ~y:(0.0,1.0) ~z:(-1.0,1.0);
  let endx = 0.1 +. 18.9 *. s.dmcharge in
  GlDraw.begins `quad_strip;
  if stereoon then begin
    GlDraw.shade_model `flat;
    GlDraw.color (1.0, 1.0, 1.0);
  end else begin
    GlDraw.shade_model `smooth;
    GlDraw.color (1.0, 0.0, 0.0);
  end;
  GlDraw.vertex ~x:0.1 ~y:0.1 ();
  GlDraw.vertex ~x:0.1 ~y:0.9 ();
  if not stereoon then
    GlDraw.color (1.0, 1.0, 0.0)
  else ();
  if s.dmcharge >= 0.4 then begin
    GlDraw.vertex ~x:7.66 ~y:0.1 ();
    GlDraw.vertex ~x:7.66 ~y:0.9 ();
    if not stereoon then
      GlDraw.color (0.0, 1.0, 0.0)
    else ();
  end else ();
  GlDraw.vertex ~x:endx ~y:0.1 ();
  GlDraw.vertex ~x:endx ~y:0.9 ();
  GlDraw.ends ();
  GlDraw.shade_model `flat;

  (* render chances *)
  (* This should properly be done with pixmaps, not textures, but it appears
   * that the Linux/nVidia OpenGL implementation differs from the Sun
   * implementation in how pixmaps are treated. Oh well. *)
  GlDraw.viewport ~x:(w-b) ~y:b ~w:b ~h:(h-b);
  GlMat.mode `modelview;
  GlMat.load_identity ();
  GlMat.mode `projection;
  GlMat.load_identity ();
  GlMat.ortho ~x:(0.0,1.0) ~y:(0.0,10.0) ~z:(-1.0,1.0);
  Gl.enable `texture_2d;
  Tex.bind (if stereoon then !texgreyid else !texid);
  GlDraw.color (1.0,1.0,1.0);
  GlDraw.begins `quads;
  for i = 1 to (min s.chances 10) do
    GlTex.coord2 (0.0,0.0);
    GlDraw.vertex ~x:0.0 ~y:(float (i-1)) ();
    GlTex.coord2 (0.0,1.0);
    GlDraw.vertex ~x:0.0 ~y:(float i) ();
    GlTex.coord2 (1.0,1.0);
    GlDraw.vertex ~x:1.0 ~y:(float i) ();
    GlTex.coord2 (1.0,0.0);
    GlDraw.vertex ~x:1.0 ~y:(float (i-1)) ();
  done;
  GlDraw.ends ();
  Gl.disable `texture_2d;

  (* render teleporter charges *)
  GlDraw.viewport ~x:(w-b) ~y:0 ~w:b ~h:b;
  GlMat.mode `modelview;
  GlMat.load_identity ();
  GlMat.mode `projection;
  GlMat.load_identity ();
  GlMat.ortho ~x:(0.0,1.0) ~y:(0.0,1.0) ~z:(-1.0,1.0);
  GlDraw.begins `quads;
  for i = 0 to (min 8 (s.telecharge-1)) do
    let x = 0.1 +. 0.3 *. float_of_int (i mod 3)
    and y = 0.1 +. 0.3 *. float_of_int (i / 3) in
    if stereoon then
      GlDraw.color (1.0, 1.0, 1.0)
    else
      GlDraw.color (x*.1.4, y*.1.4, 1.0);
    GlDraw.vertex ~x          ~y ();
    GlDraw.vertex ~x:(x+.0.2) ~y ();
    GlDraw.vertex ~x:(x+.0.2) ~y:(y+.0.2) ();
    GlDraw.vertex ~x          ~y:(y+.0.2) ();
  done;
  GlDraw.ends ()


(* event handlers for controlling Spiff *)
let movemouse s dx dy =
  s.heading <- s.heading +. s.mouse_sensitivity *. float_of_int dx;
  s.pitch   <- s.pitch -. 0.5 *. s.mouse_sensitivity *. float_of_int dy;
  if s.pitch < -55.0 then s.pitch <- -55.0 else
  if s.pitch > 55.0  then s.pitch <- 55.0 else ()

let fire s =
  if s.dmcharge >= 0.4 then begin
    s.dmcharge <- s.dmcharge -. 0.4;
    true
  end else
    false

let startfwd s =
  match s.dir with
    Dir0  -> s.dir <- DirN
  | DirE  -> s.dir <- DirNE
  | DirSE -> s.dir <- DirNE
  | DirS  -> s.dir <- DirN
  | DirSW -> s.dir <- DirNW
  | DirW  -> s.dir <- DirNW
  | _     -> ()

let startback s =
  match s.dir with
    Dir0  -> s.dir <- DirS
  | DirN  -> s.dir <- DirS
  | DirNE -> s.dir <- DirSE
  | DirE  -> s.dir <- DirSE
  | DirW  -> s.dir <- DirSW
  | DirNW -> s.dir <- DirSW
  | _     -> ()

let startleft s =
  match s.dir with
    Dir0  -> s.dir <- DirW
  | DirN  -> s.dir <- DirNW
  | DirNE -> s.dir <- DirNW
  | DirE  -> s.dir <- DirW
  | DirSE -> s.dir <- DirSW
  | DirS  -> s.dir <- DirSW
  | _     -> ()

let startright s =
  match s.dir with
    Dir0  -> s.dir <- DirE
  | DirN  -> s.dir <- DirNE
  | DirS  -> s.dir <- DirSE
  | DirSW -> s.dir <- DirSE
  | DirW  -> s.dir <- DirE
  | DirNW -> s.dir <- DirNE
  | _     -> ()

let endfwd s =
  match s.dir with
    DirN  -> s.dir <- Dir0 
  | DirNE -> s.dir <- DirE
  | DirNW -> s.dir <- DirW
  | _     -> ()

let endback s =
  match s.dir with
    DirSE -> s.dir <- DirE
  | DirS  -> s.dir <- Dir0 
  | DirSW -> s.dir <- DirW
  | _     -> ()

let endleft s =
  match s.dir with
    DirSW -> s.dir <- DirS
  | DirW  -> s.dir <- Dir0 
  | DirNW -> s.dir <- DirN
  | _     -> ()

let endright s =
  match s.dir with
    DirNE -> s.dir <- DirN
  | DirE  -> s.dir <- Dir0 
  | DirSE -> s.dir <- DirS
  | _     -> ()

let startjump s =
  s.jump <- true

let endjump s =
  s.jump <- false

