(* spiff.mli: interface for Spiff update and control panel rendering *)

type acceldir

type t = {
  hull: Collision.hull;
  mutable gait: float;  (* timer for doing gait *)
  mutable vx: float;
  mutable vy: float;
  mutable vz: float;
  mutable dir: acceldir;
  mutable jump: bool;
  mutable heading: float;
  mutable pitch: float;
  mutable mouse_sensitivity: float;

  mutable chances: int;
  mutable dmcharge: float;
  mutable telecharge: int;
}


(* Return the default new Spiff *)
val newspiff: unit -> t


(* Updates Spiff's state, given time increment, a list of collision hulls
 * to avoid, and a list of (collision hull, action) pairs to test against.
 * Return an action resulting from collision. *)
val update: t -> float -> Collision.hull list ->
            (Collision.hull * Collision.action) list -> Collision.action


(* returns a collision hull representing the blast of the dematterizer *)
val dematterhull: t -> Collision.hull


(* initialize stuff before rendering *)
val init: unit -> unit


(* return eye position, heading, and pitch *)
val get_eye: t -> float * float * float * float * float

(* transform modelview to Spiff's eye *)
val transform_to_eye: t -> unit


(* Called after the 3d scene has been rendered, this adds Spiff to it,
 * as well as the dm charge, chances, and telecharge indicators. The
 * int arguments are width, height, and border of GL window in pixels.
 * Last argument - if stereo rendering is on *)
val render: t -> int -> int -> int -> bool -> unit


(* event handlers for controlling Spiff *)
val movemouse: t -> int -> int -> unit
val fire: t -> bool
val startfwd: t -> unit
val startback: t -> unit
val startleft: t -> unit
val startright: t -> unit
val endfwd: t -> unit
val endback: t -> unit
val endleft: t -> unit
val endright: t -> unit
val startjump: t -> unit
val endjump: t -> unit

