/* texhelp.c: helper functions for managing textures from OCAML/LABLGL.
 * This provides routines to load in a mipmap texture from a PPM image on disk
 * (returning an id), to bind a loaded texture, and to destroy a texture.
 */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <caml/mlvalues.h>


/* attempt to load a PPM file from the given filename */
GLubyte *readPPM (const char *fname, GLint *glw, GLint *glh)
{
  FILE *in;
  int c, w, h, maxc, n;
  GLubyte *img;

  in = fopen (fname, "rb");
  if (in == NULL)
    return NULL;

  /* make sure we're getting raw rgb data */
  if (fgetc(in) != 'P') { fclose(in); return NULL; }
  if (fgetc(in) != '6') { fclose(in); return NULL; }

  for (;;) {
    while (isspace(c = fgetc(in)));
    if (c == EOF) { fclose(in); return NULL; }
    else if (c == '#') while ((c = fgetc(in)) != '\n');
    else {
      ungetc (c, in);
      break;
    }
  }
  if (fscanf (in, "%d", &w) < 1) { fclose(in); return NULL; }

  for (;;) {
    while (isspace(c = fgetc(in)));
    if (c == EOF) { fclose(in); return NULL; }
    else if (c == '#') while ((c = fgetc(in)) != '\n');
    else {
      ungetc (c, in);
      break;
    }
  }
  if (fscanf (in, "%d", &h) < 1) { fclose(in); return NULL; }

  for (;;) {
    while (isspace(c = fgetc(in)));
    if (c == EOF) { fclose(in); return NULL; }
    else if (c == '#') while ((c = fgetc(in)) != '\n');
    else {
      ungetc (c, in);
      break;
    }
  }
  if (fscanf (in, "%d", &maxc) < 1) { fclose(in); return NULL; }
  if (maxc <=0 || maxc >255) { fclose(in); return NULL; }

  /* single character of whitespace */
  fgetc(in);

  /* allocate space for image */
  img = (GLubyte*) malloc (w*h*3);
  if (img == NULL) { fclose(in); return NULL; }

  /* read it in */
  n = fread (img, 1, w*h*3, in);
  fclose (in);
  if (n != w*h*3) { free(img); return NULL; }

  /* let us know the size of the image */
  *glw = w;
  *glh = h;

  return img;
}


/* Attempt to open a PPM image from the given filename, build a 2d mipmap
 * texture from it, and create the id of a texture object representing it.
 * The id is a Nativeint in OCAML - not a regular int.
 */
value loadPPMTexture (value filename)
{
  GLubyte *img;
  GLsizei w, h;
  GLuint texname;

  img = readPPM (String_val(filename), &w, &h);
  if (img == NULL) {
    return copy_nativeint(0);
  }

  /* get an id */
  glGenTextures (1, &texname);
  glBindTexture (GL_TEXTURE_2D, texname);

  /* create the mipmap texture */
  glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
  gluBuild2DMipmaps (GL_TEXTURE_2D, GL_RGB, w, h, GL_RGB, GL_UNSIGNED_BYTE,
                     img);

  /* free storage */
  free (img);

  return copy_nativeint(texname);
}



/* bind a 2d texture */
value bind2dTexture (value texname)
{
  GLuint id = (GLuint) Nativeint_val(texname);
  glBindTexture (GL_TEXTURE_2D, (GLuint) Nativeint_val(texname));
  return Val_unit;
}


/* delete a bound texture */
value deleteOneTexture (value texname)
{
  GLuint tex = (GLuint) Nativeint_val(texname);
  glDeleteTextures (1, &tex);
  return Val_unit;
}

