/***************************************************************************
                          tinfrastructure.cpp  -  description
                             -------------------
    begin                : Sun Dec 15 2002
    copyright            : (C) 2002 by Chong Jiayi
    email                : jychong@stanford.edu
 ***************************************************************************/



#include "tinfrastructure.h"



TInfrastructure::TInfrastructure(int Width, int Height, int PosX, int PosY, char * Title){
	this->width = Width;
	this->height = Height;
	this->posX = PosX;
	this->posY = PosY;
	this->title = Title;
	makeInverseSqrtLookupTable();
}



TInfrastructure::~TInfrastructure() {
	//clean up
	for(int i = 0; i < textureList.size(); i++) { //loop through and delete all the loaded textures
		if(textureList[i].decIndex != UNDEFINED) glDeleteTextures(1, &(textureList[i].decIndex));
		if(textureList[i].bumpIndex != UNDEFINED) glDeleteTextures(1, &(textureList[i].bumpIndex));                                                                                          
	}
}

int TInfrastructure::CreateGL() {
	//initialize video and stuff
	const SDL_VideoInfo* info = NULL;
	Uint32 flags = false;
	int size = 0;
	/* Initialize SDL */
	if ( SDL_Init(SDL_INIT_VIDEO) < 0 ) {
		fprintf(stderr, "Couldn't init SDL: %s\n", SDL_GetError());
		return false;
	}

	SDL_WM_SetCaption(title, NULL);

	SDL_GL_SetAttribute( SDL_GL_RED_SIZE, 8 );
	SDL_GL_SetAttribute( SDL_GL_GREEN_SIZE, 8 );
	SDL_GL_SetAttribute( SDL_GL_BLUE_SIZE, 8 );    
	SDL_GL_SetAttribute( SDL_GL_ALPHA_SIZE, 8 );
	SDL_GL_SetAttribute( SDL_GL_STENCIL_SIZE, 8 );	
	SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );   


	/* Let's get some video information. */
	info = SDL_GetVideoInfo( );
	
	flags = SDL_OPENGL;
	if ( SDL_SetVideoMode(width, height, info->vfmt->BitsPerPixel, flags) == NULL ) {
		fprintf(stderr, "Couldn't init SDL: %s\n", SDL_GetError());		
		return false;
	}

	int rBits = 0, gBits = 0, bBits = 0, alphaBits = 0, stencilSize = 0;
	SDL_GL_GetAttribute(SDL_GL_RED_SIZE, &rBits);
	SDL_GL_GetAttribute(SDL_GL_BLUE_SIZE, &bBits);
	SDL_GL_GetAttribute(SDL_GL_GREEN_SIZE, &gBits);
	SDL_GL_GetAttribute(SDL_GL_STENCIL_SIZE, &stencilSize);
	SDL_GL_GetAttribute(SDL_GL_ALPHA_SIZE, &alphaBits);	
	
	printf("Red Bits:%d Green Bits:%d Blue Bits:%d Alpha Bits:%d Stencil Bits:%d\n", rBits, gBits, bBits, alphaBits, stencilSize);
	ReSizeGLScene(width, height);	
	//openGL initialization
	glEnable(GL_TEXTURE_2D);							// Enable Texture Mapping
	glShadeModel(GL_SMOOTH);							// Enable Smooth Shading
	glClearColor(0.00f, 0.00f, 0.00f, 0.0f);			// Black Background
	glClearDepth(1.0f);									// Depth Buffer Setup
	glEnable(GL_DEPTH_TEST);							// Enables Depth Testing
	glDepthFunc(GL_LEQUAL);								// The Type Of Depth Testing To Do
	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);	// Really Nice Perspective Calculations
	glDisable(GL_LIGHTING);
	glFrontFace(GL_CW);
	glEnable(GL_CULL_FACE );
	glClearStencil(0);									// Stencil Buffer Setup

  glActiveTextureARB(GL_TEXTURE0_ARB);
  glBindTexture(GL_TEXTURE_CUBE_MAP_EXT, TO_NORMALIZE_CUBE_MAP);
  makeNormalizeVectorCubeMap(32);
  glEnable(GL_TEXTURE_CUBE_MAP_EXT);
  glActiveTextureARB(GL_TEXTURE1_ARB);
  
	return true;
}

// Resize And Initialize The GL Window
void TInfrastructure::ReSizeGLScene(GLsizei width, GLsizei height)
{
	if (height==0) { 										// Prevent A Divide By Zero By
		height=1;										// Making Height Equal One
	}

	glViewport(0,0,width,height);						// Reset The Current Viewport

	glMatrixMode(GL_PROJECTION);						// Select The Projection Matrix
	glLoadIdentity();									// Reset The Projection Matrix

	// Calculate The Aspect Ratio Of The Window
	gluPerspective(45.0f,(GLfloat)width/(GLfloat)height,0.1f,100.0f);

	glMatrixMode(GL_MODELVIEW);							// Select The Modelview Matrix
	glLoadIdentity();									// Reset The Modelview Matrix
}

void TInfrastructure::KillGLWindow() {
	SDL_Quit();
}

Normal * TInfrastructure::convertHeightFieldToNormalMap(GLubyte *pixels, int w, int h, int wr, int hr, float scale) {
  Normal *nmap;
  float sqlen, reciplen, nx, ny, nz;

  const float oneOver255 = 1.0f/255.0f;

  float c, cx, cy;

  nmap = new Normal[w * h]; 

  for (int i=0; i<h; i++) {
    for (int j=0; j<w; j++) {
      /* Expand [0,255] texel values to the [0,1] range. */
      c = pixels[i*wr + j] * oneOver255;
      /* Expand the texel to its right. */
      cx = pixels[i*wr + (j+1)%wr] * oneOver255;
      /* Expand the texel one up. */
      cy = pixels[((i+1)%hr)*wr + j] * oneOver255;
       /* Normalize the vector. */
      Vector curDC;
      curDC.x = scale * (c - cy);
      curDC.y = scale * (c - cx);
      curDC.z = 1.0f;
      Vnormal(&curDC);
      /* Repack the normalized vector into an RGB unsigned byte
         vector in the normal map image. */
      nmap[i*w+j].nx = 128 + 127*curDC.x;
      nmap[i*w+j].ny = 128 + 127*curDC.y;
      nmap[i*w+j].nz = 128 + 127*curDC.z;

      nmap[i*w+j].mag = 255;
    }
  }

  return nmap;
}


/* Based on Mark Kilgard's Paper "A Practical and Robust Bump-mapping Technique
    for Today's GPUs", http://developer.nvidia.com
   Given a normal map, create a downsampled version of the normal map
   at half the width and height.  Use a 2x2 box filter to create each
   downsample.  gluBuild2DMipmaps is not suitable because each downsampled
   texel must also be renormalized. */
Normal * TInfrastructure::downSampleNormalMap(Normal *old, int w2, int h2, int w, int h) {
  const float oneOver127 = 1.0/127.0;
  const float oneOver255 = 1.0/255.0;

  Normal *nmap;
  float x, y, z, l, invl;
  float mag00, mag01, mag10, mag11;
  int ii, jj;

  /* Allocate space for the downsampled normal map level. */
  nmap = new Normal[w * h]; 

  for (int i=0; i<h2; i+=2) {
    for (int j=0; j<w2; j+=2) {

      /* The "%w2" and "%h2" modulo arithmetic makes sure that
         Nx1 and 1xN normal map levels are handled correctly. */

      /* Fetch the magnitude of the four vectors to be downsampled. */
      mag00 = oneOver255 * old[ (i  )    *w2 +  (j  )    ].mag;
      mag01 = oneOver255 * old[ (i  )    *w2 + ((j+1)%h2)].mag;
      mag10 = oneOver255 * old[((i+1)%w2)*w2 +  (j  )    ].mag;
      mag11 = oneOver255 * old[((i+1)%w2)*w2 + ((j+1)%h2)].mag;

      /* Sum 2x2 footprint of red component scaled back to [-1,1] floating point range. */
      x =  mag00 * (oneOver127 * old[ (i  )    *w2 +  (j  )    ].nx - 1.0);
      x += mag01 * (oneOver127 * old[ (i  )    *w2 + ((j+1)%h2)].nx - 1.0);
      x += mag10 * (oneOver127 * old[((i+1)%w2)*w2 +  (j  )    ].nx - 1.0);
      x += mag11 * (oneOver127 * old[((i+1)%w2)*w2 + ((j+1)%h2)].nx - 1.0);

      /* Sum 2x2 footprint of green component scaled back to [-1,1] floating point range. */
      y =  mag00 * (oneOver127 * old[ (i  )    *w2 +  (j  )    ].ny - 1.0);
      y += mag01 * (oneOver127 * old[ (i  )    *w2 + ((j+1)%h2)].ny - 1.0);
      y += mag10 * (oneOver127 * old[((i+1)%w2)*w2 +  (j  )    ].ny - 1.0);
      y += mag11 * (oneOver127 * old[((i+1)%w2)*w2 + ((j+1)%h2)].ny - 1.0);

      /* Sum 2x2 footprint of blue component scaled back to [-1,1] floating point range. */
      z =  mag00 * (oneOver127 * old[ (i  )    *w2 +  (j  )    ].nz - 1.0);
      z += mag01 * (oneOver127 * old[ (i  )    *w2 + ((j+1)%h2)].nz - 1.0);
      z += mag10 * (oneOver127 * old[((i+1)%w2)*w2 +  (j  )    ].nz - 1.0);
      z += mag11 * (oneOver127 * old[((i+1)%w2)*w2 + ((j+1)%h2)].nz - 1.0);

      /* Compute length of the (x,y,z) vector. */
      l = sqrt(x*x + y*y + z*z);
      if (l == 0.0) {
        x = 0.0;
        y = 0.0;
        z = 1.0;
      } else {
        invl = 1.0/l;
        x = x*invl;
        y = y*invl;
        z = z*invl;
      }

      ii = i >> 1;
      jj = j >> 1;

      /* Pack the normalized vector into an RGB unsigned byte vector
         in the downsampled image. */
      nmap[ii*w+jj].nx = 128 + 127*x;
      nmap[ii*w+jj].ny = 128 + 127*y;
      nmap[ii*w+jj].nz = 128 + 127*z;

      /* Store the magnitude of the average vector in the alpha
         component so we keep track of the magntiude. */
      l = l/4.0;
      if (l > 1.0) {
        nmap[ii*w+jj].mag = 255;
      } else {
        nmap[ii*w+jj].mag = 255*l;
      }
    }
  }

  delete old;

  return nmap;
}

/* Convert the supplied height-field image into a normal map (a normalized
   vector compressed to the [0,1] range in RGB and A=1.0).  Load the
   base texture level, then recursively downsample and load successive
   normal map levels (being careful to expand, average, renormalize,
   and unexpand each RGB value an also accumulate the average vector
   shortening in alpha). */
void TInfrastructure::convertHeightFieldAndLoadNormalMapTexture(GLubyte *pixels, int w, int h, int wr, int hr, float scale) {
  Normal *nmap;
  int level;

  nmap = convertHeightFieldToNormalMap(pixels, w, h, wr, hr, scale);

  level = 0;

  /* Load original maximum resolution normal map. */

  /* The BGRA color component ordering is fastest for NVIDIA. */
  glTexImage2D(GL_TEXTURE_2D, level, GL_COMPRESSED_RGBA_S3TC_DXT5_EXT, w, h, level,
    GL_BGRA_EXT, GL_UNSIGNED_BYTE, &nmap->nz);

  /* Downsample the normal map for mipmap levels down to 1x1. */
  while (w > 1 || h > 1) {
    int nw, nh;

    level++;

    /* Half width and height but not beyond one. */
    nw = w >> 1;
    nh = h >> 1;
    if (nw == 0) nw = 1;
    if (nh == 0) nh = 1;

    nmap = downSampleNormalMap(nmap, w, h, nw, nh);

    glTexImage2D(GL_TEXTURE_2D, level, GL_COMPRESSED_RGBA_S3TC_DXT5_EXT, nw, nh, 0,
      GL_BGRA_EXT, GL_UNSIGNED_BYTE, &nmap->nz);

    /* Make the new width and height the old width and height. */
    w = nw;
    h = nh;
  }

   delete nmap;
}


/* Given a cube map face index, cube map size, and integer 2D face position,
 * return the cooresponding normalized vector.
 */
void TInfrastructure::getCubeVector(int i, int cubesize, int x, int y, Vector *curVec) {
  float s, t, sc, tc, mag;

  s = ((float)x + 0.5) / (float)cubesize;
  t = ((float)y + 0.5) / (float)cubesize;
  sc = s*2.0 - 1.0;
  tc = t*2.0 - 1.0;

  switch (i) {
  case 0:
    curVec->x = 1.0f;
    curVec->y = -tc;
    curVec->z = -sc;
    break;
  case 1:
    curVec->x = -1.0f;
    curVec->y = -tc;
    curVec->z = sc;
    break;
  case 2:
    curVec->x = sc;
    curVec->y = 1.0;
    curVec->z = tc;
    break;
  case 3:
    curVec->x = sc;
    curVec->y = -1.0f;
    curVec->z = -tc;
    break;
  case 4:
    curVec->x = sc;
    curVec->y = -tc;
    curVec->z = 1.0f;
    break;
  case 5:
    curVec->x = -sc;
    curVec->y = -tc;
    curVec->z = -1.0f;
    break;
  }

  Vnormal(curVec);
}

/* Initialize a cube map texture object that generates RGB values
 * that when expanded to a [-1,1] range in the register combiners
 * form a normalized vector matching the per-pixel vector used to
 * access the cube map.
 */
void TInfrastructure::makeNormalizeVectorCubeMap(int size) {
  Vector curVec;
  GLubyte *pixels;

  pixels = (GLubyte*) new GLubyte[size * size * 3]; 
  if (pixels == NULL) {
    cerr << "npeturb: malloc failed in makeNormalizedVectorCubeMap" << endl;
    exit(1);
  }

  glTexParameteri(GL_TEXTURE_CUBE_MAP_EXT, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
  glTexParameteri(GL_TEXTURE_CUBE_MAP_EXT, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
  glTexParameteri(GL_TEXTURE_CUBE_MAP_EXT, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  glTexParameteri(GL_TEXTURE_CUBE_MAP_EXT, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

  for (int i = 0; i < 6; i++) {
    for (int y = 0; y < size; y++) {
      for (int x = 0; x < size; x++) {
        getCubeVector(i, size, x, y, &curVec);
        pixels[3*(y*size+x) + 0] = 128 + 127*curVec.x;
        pixels[3*(y*size+x) + 1] = 128 + 127*curVec.y;
        pixels[3*(y*size+x) + 2] = 128 + 127*curVec.z;
      }
    }
    glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_X_EXT+i, 0, GL_COMPRESSED_RGB_ARB,
      size, size, 0, GL_RGB, GL_UNSIGNED_BYTE, pixels);
  }

  delete pixels;
}

/*** LOAD TEXTURE IMAGES ***/

gliGenericImage * TInfrastructure::readImage(char *filename)
{
	return readTGAImage(filename);
}

void TInfrastructure::loadTextureDecalImage(char *filename, int mipmaps)
{
  gliGenericImage *image;

  image = readImage(filename);
  if (image->format == GL_COLOR_INDEX) {
    /* Rambo 8-bit color index into luminance. */
    image->format = GL_LUMINANCE;
  }
  if (mipmaps) {
    gluBuild2DMipmaps(GL_TEXTURE_2D, GL_COMPRESSED_RGB_S3TC_DXT1_EXT, 
      image->width, image->height,
      image->format, GL_UNSIGNED_BYTE, image->pixels);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
      GL_LINEAR_MIPMAP_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  } else {
    glTexImage2D(GL_TEXTURE_2D, 0, GL_COMPRESSED_RGB_S3TC_DXT1_EXT, 
      image->width, image->height, 0,
      image->format, GL_UNSIGNED_BYTE, image->pixels);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  }

}

void TInfrastructure::loadTextureNormalMap(char *filename, float scale)
{
  gliGenericImage *image;
  int w, h, wr, hr, badSize;

  image = readImage(filename);
  if (image->components != 1) {
    fprintf(stderr, "npeturb: \"%s\" must be a luminance height field image\n", filename);
    exit(1);
  }

  w = image->width;
  h = image->height;

  badSize = 0;
  if ( (w & (w-1))) {
    if ( ((w-1) & (w-2))) {
      /* Width not 2^n or 2^+1. */
      badSize = 1;
    } else {
      /* Width is power of two plus one, use border */
      wr = w;
      w = w-1;
    }
  } else {
    /* Width is a power of two, wrap normal map width. */
    wr = w;
  }

  if ( (h & (h-1))) {
    if ( ((h-1) & (h-2))) {
      /* Height not 2^n or 2^+1. */
      badSize = 1;
    } else {
      /* Height is power of two plus one, use border */
      hr = h;
      h = h-1;
    }
  } else {
    /* Height is a power of two, wrap normal map height. */
    hr = h;
  }

  if (badSize) {
    fprintf(stderr,
      "npeturb: normal map \"%s\" must have 2^n or 2^n+1 dimensions, not %dx%d\n",
      filename, w, h);
    exit(1);
  }

  convertHeightFieldAndLoadNormalMapTexture(image->pixels,
    w, h, wr, hr, scale);
}

/* Loads in the decal and bump-map texture and then returns their information in a TextureInfo structure*/
TextureInfo *TInfrastructure::LoadTexture(char *decalFile, int mipmaps, char *bumpFile, float scale) {
	TextureInfo *retInfo = new TextureInfo;
	retInfo->bumpIndex = UNDEFINED;
	retInfo->decIndex = UNDEFINED;

	if(decalFile != NULL) { //load in decal texture
		glGenTextures(1, &(retInfo->decIndex));
		glBindTexture(GL_TEXTURE_2D, retInfo->decIndex);
		loadTextureDecalImage(decalFile, mipmaps);
	}

	if(bumpFile != NULL) { //load in bump texture
		glGenTextures(1, &(retInfo->bumpIndex));
		glBindTexture(GL_TEXTURE_2D, retInfo->bumpIndex);
		loadTextureNormalMap(bumpFile, scale);
	}
	
  	textureList.push_back(*retInfo);
	return retInfo;
}

char *TInfrastructure::readShaderFile(const char* filename) {
	FILE *file;
	char * buffer;

	off_t size;
	file = fopen(filename,"r");
	if (!file) return NULL;
	fseek(file,0L,SEEK_END);
	size = ftell(file);
	fclose(file);
	// Read the file into memory
	file = fopen(filename, "rb");

	buffer = new char[size];
	if (!buffer) return NULL;
	fread(buffer, size, sizeof(char), file);
	fclose(file);
	return buffer;
}

void TInfrastructure::PrepareStencilShadowing() {
	glDepthMask(GL_FALSE);
	glDepthFunc(GL_LEQUAL);

	glEnable(GL_STENCIL_TEST);
	glColorMask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE);
	glStencilFunc(GL_ALWAYS, 1, 0xffffffff);
}

void TInfrastructure::EndStencilShadowing() {
	glDepthFunc(GL_LEQUAL);
	glDepthMask(GL_TRUE);
	glDisable(GL_STENCIL_TEST);
	glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
	glShadeModel(GL_SMOOTH);
	glFrontFace(GL_CW);
}

/* draw a shadowing rectangle covering the entire screen */
void TInfrastructure::RenderShadowRect(GLfloat r, GLfloat g, GLfloat b, GLfloat alpha) {
   //default values can be: r = 0.0, g = 0.0, b = 0.0, alpha = 0.4
	glFrontFace(GL_CCW);
	glColorMask(1, 1, 1, 1);

	glDisable(GL_DEPTH_TEST);
	glColor4f(r, g, b, alpha);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
//	glStencilFunc(GL_NOTEQUAL, 0, 0xffffffff);
	glStencilFunc(GL_LESS, 0, 0xffffffff);
	glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
	glPushMatrix();
	glLoadIdentity();

	glBegin(GL_TRIANGLE_STRIP);
		glVertex3f(-0.1f, 0.1f,-0.10f);
		glVertex3f(-0.1f,-0.1f,-0.10f);
		glVertex3f( 0.1f, 0.1f,-0.10f);
		glVertex3f( 0.1f,-0.1f,-0.10f);
	glEnd();

	glPopMatrix();
	glDisable(GL_BLEND);
	glEnable(GL_DEPTH_TEST);
}