/***************************************************************************
                          tobject.cpp  -  description
                             -------------------
    begin                : Tue Dec 17 2002
    copyright            : (C) 2002 by Chong Jiayi
    email                : jychong@stanford.edu
 ***************************************************************************/


#include "tobject.h"

#define BIG_NUMBER 100.0f


void TObject::Initialize(TInfrastructure * curInfrastructure) {
	this->vertices = NULL;
	this->orthobasis = NULL;
	this->vertexConnect = NULL;
	this->infrastructure = curInfrastructure;
	this->texInfo = NULL;
	this->triangleNum = 0;
	this->curRenderMode = 0;
}

void TObject::Cleanup(){
	//object memory clean up
	if(this->vertices != NULL) delete this->vertices;
	if(this->orthobasis != NULL) delete this->orthobasis;
} 

//renders the object into framebuffer
void TObject::Render() {
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadMatrixf(this->transformMatrix);
	glDrawArrays(GL_TRIANGLES, 0, 3 * triangleNum);

	glPopMatrix();
}

//loads a texture and creates the corresponding bump map for the current object
void TObject::LoadTexture(char *decal_filename, char *bump_filename, int mipmaps, float scale) {
	texInfo = infrastructure->LoadTexture(decal_filename, mipmaps, bump_filename, scale);
	
}

//prepares the necessary vertex array stuff for rendering
void TObject::SetupVertexArraysForRendering(int mode) {
	switch(mode) {
		case DECAL_MODE:
			glClientActiveTextureARB(GL_TEXTURE1_ARB);
			glDisableClientState(GL_TEXTURE_COORD_ARRAY);
			glClientActiveTextureARB(GL_TEXTURE0_ARB);
			glTexCoordPointer(2, GL_FLOAT, ARRAY_STRIDE, &(vertices->s));
			glEnableClientState(GL_TEXTURE_COORD_ARRAY);
		break;

		case DIFFUSE_BUMP_MODE:
			glClientActiveTextureARB(GL_TEXTURE0_ARB);
			glTexCoordPointer(3, GL_FLOAT, ARRAY_STRIDE, &(vertices->tangentSpaceLightVector));
			glEnableClientState(GL_TEXTURE_COORD_ARRAY);
			glClientActiveTextureARB(GL_TEXTURE1_ARB);
			glTexCoordPointer(2, GL_FLOAT, ARRAY_STRIDE, &(vertices->s));
			glEnableClientState(GL_TEXTURE_COORD_ARRAY);
			break;

		case SPECULAR_BUMP_MODE:
			glClientActiveTextureARB(GL_TEXTURE0_ARB);
			glTexCoordPointer(3, GL_FLOAT, ARRAY_STRIDE, &(vertices->tangentSpaceHalfAngleVector));
			glEnableClientState(GL_TEXTURE_COORD_ARRAY);
			glClientActiveTextureARB(GL_TEXTURE1_ARB);
			glTexCoordPointer(2, GL_FLOAT, ARRAY_STRIDE, &(vertices->s));
			glEnableClientState(GL_TEXTURE_COORD_ARRAY);
		break;
	}

	glVertexPointer(3, GL_FLOAT, ARRAY_STRIDE, vertices);
	glEnableClientState(GL_VERTEX_ARRAY);
	curRenderMode = mode;
}

void TObject::setVertexPos(int index, GLfloat x, GLfloat y, GLfloat z) {
	(this->vertices + index)->position.x = x;
	(this->vertices + index)->position.y = y;
	(this->vertices + index)->position.z = z;
}

void TObject::setTextureCoords(int index, GLfloat s, GLfloat t) {
	(this->vertices + index)->s = s;
	(this->vertices + index)->t = t;
}

void TObject::setNormalCoords(int index, GLfloat x, GLfloat y, GLfloat z) {
	(this->orthobasis + index)->normal.x = x;
	(this->orthobasis + index)->normal.y = y;
	(this->orthobasis + index)->normal.z = z;
}

/* Selects a single texture for decal rendering*/
void TObject::SelectSingleDecalTexture() {
	glColor3f(1.0f, 1.0f, 1.0f);
	glActiveTextureARB(GL_TEXTURE0_ARB);
	glDisable(GL_TEXTURE_CUBE_MAP_EXT);
	glEnable(GL_TEXTURE_2D);
	
	glBindTexture(GL_TEXTURE_2D, this->texInfo->decIndex);
	glDisable(GL_LIGHTING);
	glDisable(GL_BLEND);
}

/* Selects 2 textures: 1 cube map and 1 bump map for use in
    per-pixel bump-mapping, call this before doing any register
    combiners stuff
*/
void TObject::SelectCubeAndBumpTexture() {
	glColor3f(1.0f, 1.0f, 1.0f);
	glActiveTextureARB(GL_TEXTURE0_ARB);
   glDisable(GL_TEXTURE_2D);
	glEnable(GL_TEXTURE_CUBE_MAP_EXT);
	glBindTexture(GL_TEXTURE_CUBE_MAP_EXT, this->infrastructure->GetCubeMapName());

	glActiveTextureARB(GL_TEXTURE1_ARB);
	glDisable(GL_TEXTURE_CUBE_MAP_EXT);
	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, this->texInfo->bumpIndex);
}

void TObject::CalculateOrthoBasis() {
	Vertex *curVertex = this->vertices;
	OrthoNormalBasis * curOrtho = this->orthobasis;
	for(int i = 0; i < this->triangleNum; i++) {
		Plane curTrianglePlane[3];
		Vertex *v1 = curVertex;
		Vertex *v2 = curVertex + 1;
		Vertex *v3 = curVertex + 2;

		Vector inV1[3], inV2[3], inV3[3];
      //make partial planes relative to x, y, and z
		inV1[0].x = v1->position.x;
		inV1[0].y = v1->s;
		inV1[0].z = v1->t;
		inV1[1].x = v2->position.x;
		inV1[1].y = v2->s;
		inV1[1].z = v2->t;
		inV1[2].x = v3->position.x;
		inV1[2].y = v3->s;
		inV1[2].z = v3->t;

		inV2[0].x = v1->position.y;
		inV2[0].y = v1->s;
		inV2[0].z = v1->t;
		inV2[1].x = v2->position.y;
		inV2[1].y = v2->s;
		inV2[1].z = v2->t;
		inV2[2].x = v3->position.y;
		inV2[2].y = v3->s;
		inV2[2].z = v3->t;

		inV3[0].x = v1->position.z;
		inV3[0].y = v1->s;
		inV3[0].z = v1->t;
		inV3[1].x = v2->position.z;
		inV3[1].y = v2->s;
		inV3[1].z = v2->t;
		inV3[2].x = v3->position.z;
		inV3[2].y = v3->s;
		inV3[2].z = v3->t;


		
		makePlane(inV1, &inV1[1], &inV1[2], &curTrianglePlane[0]);
		makePlane(inV2, &inV2[1], &inV2[2], &curTrianglePlane[1]);
		makePlane(inV3, &inV3[1], &inV3[2], &curTrianglePlane[2]);	
		//now calculate the partial derivatives
		curOrtho->tangent.x = -(curTrianglePlane[0].B / curTrianglePlane[0].A);
		curOrtho->tangent.y = -(curTrianglePlane[1].B / curTrianglePlane[1].A);
		curOrtho->tangent.z = -(curTrianglePlane[2].B / curTrianglePlane[1].A);
		Vnormal((&(curOrtho->tangent)));
		(curOrtho + 1)->tangent = curOrtho->tangent;
		(curOrtho + 2)->tangent = curOrtho->tangent;
      
		curOrtho->binormal.x = -(curTrianglePlane[0].C / curTrianglePlane[0].A);
		curOrtho->binormal.y = -(curTrianglePlane[1].C / curTrianglePlane[1].A);
		curOrtho->binormal.z = -(curTrianglePlane[2].C / curTrianglePlane[1].A);
		Vnormal((&(curOrtho->tangent)));
		(curOrtho + 1)->binormal = curOrtho->binormal;
		(curOrtho + 2)->binormal = curOrtho->binormal;
		
		curVertex = curVertex + 3;
		curOrtho = curOrtho + 3;

	}
}

void TObject::CalculateSphericalOrthoBasis() {
	OrthoNormalBasis * curOrtho = this->orthobasis;
	GLfloat curMat[16];
	for(int i = 0; i < (this->triangleNum * 3); i++) {
		Vector *curNormal = &(curOrtho->normal);
		float horizAngle = atan2(curNormal->z, curNormal->x) / MATH_PI * 180, vertAngle = atan2(curNormal->x, curNormal->y) / MATH_PI * 180;
		glMatrixMode(GL_MODELVIEW);
		glPushMatrix();
		glLoadIdentity();
		glRotatef(horizAngle, 0.0, 1.0, 0.0);
		glRotatef(vertAngle, 0.0, 0.0, 1.0);
		glGetFloatv(GL_MODELVIEW_MATRIX, curMat);
		glPopMatrix();

		Vector vec1, vec2, dstVec1, dstVec2;
		Vset(&vec1, 1.0f, 0.0f, 0.0f);
		Vset(&vec2, 0.0f, 1.0f, 0.0f);
		transformVector(&dstVec1, &vec1, curMat);
		transformVector(&dstVec2, &vec2, curMat);
		curOrtho->tangent = dstVec1;
		curOrtho->binormal = dstVec2;
		curOrtho++;
	}
}

/* Sets up the neighbouring connectives for shadow volume projections*/
void TObject::SetConnectives() {
	Connective *curConnective = this->vertexConnect;
	Vertex *curVertex = this->vertices;

	for(int i = 0; i < (this->triangleNum); i++) {
		curConnective->neighbourIndex = UNDEFINED;
		(curConnective + 1)->neighbourIndex = UNDEFINED;
		(curConnective + 2)->neighbourIndex = UNDEFINED;

		//edge1
		Vertex *cmpVertex = this->vertices;
		for(int j = 0; j < this->triangleNum; j++) {
			if(j != i && !SameTriangle(&curVertex->position, &(curVertex + 1)->position, &(curVertex + 2)->position, &(cmpVertex)->position, &(cmpVertex + 1)->position, &(cmpVertex + 2)->position) ) {
				if(ShareEdge(curVertex, curVertex + 1, cmpVertex, cmpVertex + 1)) {
					curConnective->neighbourIndex = j;
					break;                                                                                                                           
				}
				if(ShareEdge(curVertex, curVertex + 1, cmpVertex + 1, cmpVertex + 2)) {
					curConnective->neighbourIndex = j;
					break;
				}
				if(ShareEdge(curVertex, curVertex + 1, cmpVertex + 2, cmpVertex)) {
					curConnective->neighbourIndex = j;
					break;
				}
				cmpVertex = cmpVertex + 3;
			}	
		}
			//edge2
		cmpVertex = this->vertices;
		for(int j = 0; j < this->triangleNum; j++) {
			if(j != i && !SameTriangle(&curVertex->position, &(curVertex + 1)->position, &(curVertex + 2)->position, &(cmpVertex)->position, &(cmpVertex + 1)->position, &(cmpVertex + 2)->position) ) {
				if(ShareEdge(curVertex + 1, curVertex + 2, cmpVertex, cmpVertex + 1)) {
					(curConnective + 1)->neighbourIndex = j;
					break;
				}
				if(ShareEdge(curVertex + 1, curVertex + 2, cmpVertex + 1, cmpVertex + 2)) {
					(curConnective + 1)->neighbourIndex = j;
					break;
				}
				if(ShareEdge(curVertex + 1, curVertex + 2, cmpVertex + 2, cmpVertex)) {
					(curConnective + 1)->neighbourIndex = j;
					break;
				}
				cmpVertex = cmpVertex + 3;
			}
		}
		//edge3
		cmpVertex = this->vertices;
		for(int j = 0; j < this->triangleNum; j++) {
			if(j != i && !SameTriangle(&curVertex->position, &(curVertex + 1)->position, &(curVertex + 2)->position, &(cmpVertex)->position, &(cmpVertex + 1)->position, &(cmpVertex + 2)->position) ) {
				if(ShareEdge(curVertex + 2, curVertex, cmpVertex, cmpVertex + 1)) {
					(curConnective + 2)->neighbourIndex = j;
					break;
				}
				if(ShareEdge(curVertex + 2, curVertex, cmpVertex + 1, cmpVertex + 2)) {
					(curConnective + 2)->neighbourIndex = j;
					break;
				}
				if(ShareEdge(curVertex + 2, curVertex, cmpVertex + 2, cmpVertex)) {
					(curConnective + 2)->neighbourIndex = j;
					break;
				}
				cmpVertex = cmpVertex + 3;
			}
		}                    
   	
		curConnective = curConnective + 3;
		curVertex = curVertex + 3;
	}
}

bool TObject::ShareEdge(Vertex *edge11, Vertex * edge12, Vertex *edge21, Vertex *edge22) {

	Vector v1Pos1 = edge11->position;
	Vector v1Pos2 = edge12->position;
	Vector v2Pos1 = edge21->position;
	Vector v2Pos2 = edge22->position;

	if(Vequal(&v1Pos1, &v2Pos1) && Vequal(&v1Pos2, &v2Pos2)) return true;
	if(Vequal(&v1Pos1, &v2Pos2) && Vequal(&v1Pos2, &v2Pos1)) return true;

	return false;
}

bool TObject::SameTriangle(Vector *v11, Vector *v12, Vector *v13, Vector *v21, Vector *v22, Vector *v23) {
	if(Vequal(v11, v21) && Vequal(v12, v22) && Vequal(v13, v23)) return true;
	if(Vequal(v11, v21) && Vequal(v12, v23) && Vequal(v13, v22)) return true;
	if(Vequal(v11, v22) && Vequal(v12, v21) && Vequal(v13, v23)) return true;
	if(Vequal(v11, v22) && Vequal(v12, v23) && Vequal(v13, v21)) return true;
	if(Vequal(v11, v23) && Vequal(v12, v21) && Vequal(v13, v22)) return true;
	if(Vequal(v11, v23) && Vequal(v12, v22) && Vequal(v13, v21)) return true;

	return false;
}

void TObject::CalculatePlanes() {
	Connective *curConnective = this->vertexConnect;
	Vertex *curVertex = this->vertices;
	
	for(int i = 0; i < this->triangleNum; i++) {
		Vector *v1 = &(curVertex->position);
		Vector *v2 = &((curVertex + 1)->position);
		Vector *v3 = &((curVertex + 2)->position);

		makePlane(v1, v2, v3, &(curConnective->plane));
		(curConnective + 1)->plane = curConnective->plane;
		(curConnective + 2)->plane = curConnective->plane;

		curConnective = curConnective + 3;
		curVertex = curVertex + 3;
	}
}


void TObject::PrepareTransform() {
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
}

void TObject::EndTransform() {
	glMatrixMode(GL_MODELVIEW);
	glGetFloatv(GL_MODELVIEW_MATRIX, this->transformMatrix);
	invertMatrixf(this->inverseTransformMatrix, this->transformMatrix);
	glPopMatrix();
}



/* Prepares register combiners for bump-mapping, based on Mark Kilgard's Paper "A Practical and Robust Bump-mapping Technique
    for Today's GPUs", http://developer.nvidia.com */

void TObject::EnableNVRegCombinersDiffuse(GLfloat aRed, GLfloat aGreen, GLfloat aBlue, GLfloat aAlpha, GLfloat dRed, GLfloat dGreen, GLfloat dBlue, GLfloat dAlpha) {
	GLfloat curAmbient[4];
	GLfloat curDiffuse[4];

	curAmbient[0] = aRed;
	curAmbient[1] = aGreen;
	curAmbient[2] = aBlue;
	curAmbient[3] = aAlpha;

	curDiffuse[0] = dRed;
	curDiffuse[1] = dGreen;
	curDiffuse[2] = dBlue;
	curDiffuse[3] = dAlpha;

	glCombinerParameteriNV(GL_NUM_GENERAL_COMBINERS_NV, 2);
	glCombinerParameterfvNV(GL_CONSTANT_COLOR0_NV, curAmbient);
	glCombinerParameterfvNV(GL_CONSTANT_COLOR1_NV, curDiffuse);
	/*** GENERAL Combiner ZERO, RGB portion. ***/
	/* Argb = 3x3 matrix column1 = expand(texture0rgb) = N' */
	glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_A_NV,
	GL_TEXTURE0_ARB, GL_EXPAND_NORMAL_NV, GL_RGB);
	/* Brgb = expand(texture1rgb) = L */
	glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_B_NV,
	GL_TEXTURE1_ARB, GL_EXPAND_NORMAL_NV, GL_RGB);
	/* spare0rgb = Argb dot Brgb
	=expand(texture0rgb) dot expand(texture1rgb) = L dot N' */
	glCombinerOutputNV(GL_COMBINER0_NV, GL_RGB,
	GL_SPARE0_NV, GL_DISCARD_NV, GL_DISCARD_NV,
	GL_NONE, GL_NONE, GL_TRUE, GL_FALSE, GL_FALSE);
	/*** GENERAL Combiner ZERO, Alpha portion. ***/
	/* Aa = 1 */
	glCombinerInputNV(GL_COMBINER0_NV, GL_ALPHA, GL_VARIABLE_A_NV,
	GL_ZERO, GL_UNSIGNED_INVERT_NV, GL_ALPHA);
	/* Ba = expand(texture1b) = Lz */
	glCombinerInputNV(GL_COMBINER0_NV, GL_ALPHA, GL_VARIABLE_B_NV,
	GL_TEXTURE1_ARB, GL_EXPAND_NORMAL_NV, GL_BLUE);
	/* Ca = 1 */
	glCombinerInputNV(GL_COMBINER0_NV, GL_ALPHA, GL_VARIABLE_C_NV,
	GL_ZERO, GL_UNSIGNED_INVERT_NV, GL_ALPHA);
	/* Da = expand(texture1b) = Lz */
	glCombinerInputNV(GL_COMBINER0_NV, GL_ALPHA, GL_VARIABLE_D_NV,
	GL_TEXTURE1_ARB, GL_EXPAND_NORMAL_NV, GL_BLUE);
	/* spare0a = 4*(1*Lz + 1*Lz) = 8*expand(texture1b) */
	glCombinerOutputNV(GL_COMBINER0_NV, GL_ALPHA,
	GL_DISCARD_NV, GL_DISCARD_NV, GL_SPARE0_NV,
	GL_SCALE_BY_FOUR_NV, GL_NONE, GL_FALSE, GL_FALSE, GL_FALSE);
	/*** GENERAL Combiner ONE, RGB portion. ***/
	/* Argb = spare0rgb = L dot N' */
	glCombinerInputNV(GL_COMBINER1_NV, GL_RGB, GL_VARIABLE_A_NV,
	GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	/* Brgb = expand(texture0a) = normal map denormalization factor */
	glCombinerInputNV(GL_COMBINER1_NV, GL_RGB, GL_VARIABLE_B_NV,
	GL_TEXTURE0_ARB, GL_UNSIGNED_IDENTITY_NV, GL_ALPHA);
	/* spare0rgb = Argb Brgb = L dot N' scaled by the normal map denormalization factor */
	glCombinerOutputNV(GL_COMBINER1_NV, GL_RGB,
	GL_SPARE0_NV, GL_DISCARD_NV, GL_DISCARD_NV,
	GL_NONE, GL_NONE, GL_TRUE, GL_FALSE, GL_FALSE);
	/*** GENERAL Combiner ONE, Alpha portion. ***/
	/* Discard all outputs. */
	glCombinerOutputNV(GL_COMBINER1_NV, GL_ALPHA,
	GL_DISCARD_NV, GL_DISCARD_NV, GL_DISCARD_NV,
	GL_NONE, GL_NONE, GL_FALSE, GL_FALSE, GL_FALSE);
	/*** FINAL Combiner. ***/
	/* A = spare0a = per-pixel self-shadowing term */
	glFinalCombinerInputNV(GL_VARIABLE_A_NV,
	GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_ALPHA);
	/* B = EF */
	glFinalCombinerInputNV(GL_VARIABLE_B_NV,
	GL_E_TIMES_F_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	/* C = zero */
	glFinalCombinerInputNV(GL_VARIABLE_C_NV,
	GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	/* D = C0 = ambient illumination contribution */
	glFinalCombinerInputNV(GL_VARIABLE_D_NV,
	GL_CONSTANT_COLOR0_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	/* E = C1 = diffuse material characteristic */
	glFinalCombinerInputNV(GL_VARIABLE_E_NV,
	GL_CONSTANT_COLOR1_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	/* F = spare0rgb = diffuse illumination contribution = L dot N' */
	glFinalCombinerInputNV(GL_VARIABLE_F_NV,
	GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	/* diffuse RGB color = A*E*F + D = diffuse modulated by self-shadowing term and the
	diffuse material characteristic + ambient */
	/* G = spare0a = self-shadowing term = 8*expand(texture1b) */
	glFinalCombinerInputNV(GL_VARIABLE_G_NV,
	GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_ALPHA);
	glEnable(GL_REGISTER_COMBINERS_NV);
}

void TObject::EnableNVRegCombinersSpecular() {
	glCombinerParameteriNV(GL_NUM_GENERAL_COMBINERS_NV, 2);
	/*** GENERAL Combiner ZERO, RGB portion. ***/
	/* Argb = 3x3 matrix column1 = expand(texture0rgb) = N' */
	glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_A_NV,
	GL_TEXTURE0_ARB, GL_EXPAND_NORMAL_NV, GL_RGB);
	/* Brgb = expand(texture1rgb) = H */
	glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_B_NV,
	GL_TEXTURE1_ARB, GL_EXPAND_NORMAL_NV, GL_RGB);
	/* spare0rgb = Argb dot Brgb = expand(texture0rgb) dot expand(texture1rgb)
	= N' dot H */
	glCombinerOutputNV(GL_COMBINER0_NV, GL_RGB,
	GL_SPARE0_NV, GL_DISCARD_NV, GL_DISCARD_NV,
	GL_NONE, GL_NONE, GL_TRUE, GL_FALSE, GL_FALSE);
	/*** GENERAL Combiner ZERO, Alpha portion. ***/
	/* Aa = texture1b = unexpanded Hz */
	glCombinerInputNV(GL_COMBINER0_NV, GL_ALPHA, GL_VARIABLE_A_NV,
	GL_TEXTURE1_ARB, GL_UNSIGNED_IDENTITY_NV, GL_BLUE);
	/* Ba = 1 */
	glCombinerInputNV(GL_COMBINER0_NV, GL_ALPHA, GL_VARIABLE_B_NV,
	GL_ZERO, GL_UNSIGNED_INVERT_NV, GL_ALPHA);
	/* spare0a = 1 * texture1b = unexpanded Hz */
	glCombinerOutputNV(GL_COMBINER0_NV, GL_ALPHA,
	GL_SPARE0_NV, GL_DISCARD_NV, GL_DISCARD_NV,
	GL_NONE, GL_NONE, GL_FALSE, GL_FALSE, GL_FALSE);
	/*** GENERAL Combiner ONE, RGB portion. ***/
	/* Argb = 0 */
	glCombinerInputNV(GL_COMBINER1_NV, GL_RGB, GL_VARIABLE_A_NV,
	GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	/* Brgb = 0 */
	glCombinerInputNV(GL_COMBINER1_NV, GL_RGB, GL_VARIABLE_B_NV,
	GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	/* Crgb = spare0rgb = H dot N' */
	glCombinerInputNV(GL_COMBINER1_NV, GL_RGB, GL_VARIABLE_C_NV,
	GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	/* Drgb = spare0rgb = H dot N' */
	glCombinerInputNV(GL_COMBINER1_NV, GL_RGB, GL_VARIABLE_D_NV,
	GL_SPARE0_NV, GL_SIGNED_IDENTITY_NV, GL_RGB);
	/* spare0rgb = ((spare0a >= 0.5) ? spare0rgb^2 : 0)
	= ((H dot N > 0) ? (H dot N')^2 : 0) */
	glCombinerOutputNV(GL_COMBINER1_NV, GL_RGB,
	GL_DISCARD_NV, GL_DISCARD_NV, GL_SPARE0_NV,
	GL_NONE, GL_NONE, GL_FALSE, GL_FALSE, GL_TRUE);
	/*** GENERAL Combiner ONE, Alpha portion. ***/
	/* Discard all outputs. */
	glCombinerOutputNV(GL_COMBINER1_NV, GL_ALPHA,
	GL_DISCARD_NV, GL_DISCARD_NV, GL_DISCARD_NV,
	GL_NONE, GL_NONE, GL_FALSE, GL_FALSE, GL_FALSE);
	/*** FINAL Combiner. ***/
	/* A = EF */
	glFinalCombinerInputNV(GL_VARIABLE_A_NV,
	GL_E_TIMES_F_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	/* B = EF */
	glFinalCombinerInputNV(GL_VARIABLE_B_NV,
	GL_E_TIMES_F_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	/* C = zero */
	glFinalCombinerInputNV(GL_VARIABLE_C_NV,
	GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	/* D = zero = no extra specular illumination contribution */
	glFinalCombinerInputNV(GL_VARIABLE_D_NV,
	GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	/* F = spare0rgb = (H dot N')^2 */
	glFinalCombinerInputNV(GL_VARIABLE_E_NV,
	GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	/* F = spare0rgb = (H dot N')^2 */
	glFinalCombinerInputNV(GL_VARIABLE_F_NV,
	GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	/* specular RGB color = A*B = (E*F)*(E*F) = (H dot N')^8 */
	/* G = 0 */
	glFinalCombinerInputNV(GL_VARIABLE_G_NV,
	GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_ALPHA);
	glEnable(GL_REGISTER_COMBINERS_NV);
}

void TObject::SetLightPos(GLfloat x, GLfloat y, GLfloat z) {
	this->lightPos.x = x;
	this->lightPos.y = y;
	this->lightPos.z = z;
}

void TObject::SetEyePos(GLfloat x, GLfloat y, GLfloat z) {
	this->eyePos.x = x;
	this->eyePos.y = y;
	this->eyePos.z = z;
}

void TObject::SetupLightVectorsForRendering() {
	Vertex *curVertex = this->vertices;
	OrthoNormalBasis *curBasis = this->orthobasis;
	Connective *curConnective = this->vertexConnect;

	Vector lightVec;
	Vector eyeVec;
	transformVector(&lightVec, &(this->lightPos), (this->inverseTransformMatrix) );
	transformVector(&eyeVec, &(this->eyePos), (this->inverseTransformMatrix) );
		
	for(int i = 0; i < this->triangleNum * 3; i++) {
		//do light vector calculations
		Vsub((&lightVec), (&(curVertex->position)), (&(curVertex->lightPos)) );
		curVertex->tangentSpaceLightVector.x = Vdot((&(curVertex->lightPos)), (&(curBasis->tangent)) );
		curVertex->tangentSpaceLightVector.y = Vdot((&(curVertex->lightPos)), (&(curBasis->binormal)) );
		curVertex->tangentSpaceLightVector.z = Vdot((&(curVertex->lightPos)), (&(curBasis->normal)) );
		Vnormal((&(curVertex->tangentSpaceLightVector)) );
		//now do specular half-angle calculations
		Vsub((&eyeVec), (&(curVertex->position)), (&(curVertex->eyePos)) );
		curVertex->tangentSpaceHalfAngleVector.x = Vdot((&(curVertex->eyePos)), (&(curBasis->tangent)) );
		curVertex->tangentSpaceHalfAngleVector.y = Vdot((&(curVertex->eyePos)), (&(curBasis->binormal)) );
		curVertex->tangentSpaceHalfAngleVector.z = Vdot((&(curVertex->eyePos)), (&(curBasis->normal)) );  
		Vnormal((&(curVertex->tangentSpaceHalfAngleVector)) );
		Vadd((&(curVertex->tangentSpaceHalfAngleVector)), &(curVertex->tangentSpaceLightVector), &(curVertex->tangentSpaceHalfAngleVector) );
		Vnormal((&(curVertex->tangentSpaceHalfAngleVector)) );

		curVertex = curVertex + 1;
		curBasis = curBasis + 1;
	}

	//now check if the plane is visible to the light for shadow volume stuff
	for(int j = 0; j < this->triangleNum; j++) {
		if(vecPlaneOrientation(&lightVec, &curConnective->plane) > 0) {
			curConnective->visible = true;
		}
		else curConnective->visible = false;     

		(curConnective + 1)->visible = curConnective->visible;
		(curConnective + 2)->visible = curConnective->visible;
		curConnective =  curConnective + 3;
	}

}

void TObject::EnablePlainDecal()
{
	glActiveTextureARB( GL_TEXTURE1_ARB );
	glDisable( GL_TEXTURE_CUBE_MAP_EXT );
	glDisable( GL_TEXTURE_2D );

	glActiveTextureARB( GL_TEXTURE0_ARB );
	glDisable( GL_TEXTURE_CUBE_MAP_EXT );
	glDisable( GL_TEXTURE_2D );

	glDisable(GL_REGISTER_COMBINERS_NV);
	glDisable(GL_BLEND);
	glDepthFunc(GL_LEQUAL);

	glDisable(GL_LIGHTING);

	curRenderMode = DECAL_MODE;
}

void TObject:: EnableShadowedDecal() {
	glActiveTextureARB( GL_TEXTURE1_ARB );
	glDisable( GL_TEXTURE_CUBE_MAP_EXT );
	glDisable( GL_TEXTURE_2D );

	glActiveTextureARB( GL_TEXTURE0_ARB );
	glDisable( GL_TEXTURE_CUBE_MAP_EXT );
	glDisable( GL_TEXTURE_2D );

	glDisable(GL_REGISTER_COMBINERS_NV);
	glDisable(GL_BLEND);
	glDisable(GL_LIGHTING);

	glDepthFunc(GL_EQUAL);
	glEnable(GL_STENCIL_TEST);
	glStencilFunc(GL_GEQUAL, 0, 0xffffffff);
	glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);

	curRenderMode = DECAL_MODE;
}

void TObject:: EnableStencilDecal(GLfloat r, GLfloat g, GLfloat b) {
	glActiveTextureARB( GL_TEXTURE1_ARB );
	glDisable( GL_TEXTURE_CUBE_MAP_EXT );
	glDisable( GL_TEXTURE_2D );

	glActiveTextureARB( GL_TEXTURE0_ARB );
	glDisable( GL_TEXTURE_CUBE_MAP_EXT );
	glDisable( GL_TEXTURE_2D );

	glDisable(GL_REGISTER_COMBINERS_NV);
	glDisable(GL_BLEND);
	glDisable(GL_LIGHTING);

	glDepthFunc(GL_EQUAL);
	glEnable(GL_STENCIL_TEST);
	glStencilFunc(GL_GEQUAL, 0, 0xffffffff);
	glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);

	curRenderMode = DECAL_MODE;
}

void TObject::EnableDiffusePass(GLfloat aRed, GLfloat aGreen, GLfloat aBlue, GLfloat aAlpha, GLfloat dRed, GLfloat dGreen, GLfloat dBlue, GLfloat dAlpha) {
	EnableNVRegCombinersDiffuse(aRed, aGreen, aBlue, aAlpha, dRed, dGreen, dBlue, dAlpha);
	glDepthFunc(GL_EQUAL);

	glEnable(GL_BLEND);
	glBlendFunc(GL_DST_COLOR, GL_ZERO);

	curRenderMode = DIFFUSE_BUMP_MODE;

}

void TObject::EnableSpecularPass() {
//	glDisable(GL_STENCIL_TEST);
	EnableNVRegCombinersSpecular();
	glDepthFunc(GL_EQUAL);

	glEnable(GL_BLEND);
	glBlendFunc(GL_DST_ALPHA, GL_ONE);

	curRenderMode = SPECULAR_BUMP_MODE;
}

void TObject::CastShadowDepthPass() {
	Vertex *curVertex = this->vertices;
	Connective *curConnective = this->vertexConnect;
	Connective *theConnectives = this->vertexConnect;

	Vector lightVec;
	transformVector(&lightVec, &(this->lightPos), (this->inverseTransformMatrix) );

	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadMatrixf(this->transformMatrix);

	// first pass, decrease stencil value
	glCullFace(GL_FRONT);
	glStencilOp(GL_KEEP, GL_KEEP, GL_INCR);

	ShadowPass(curConnective, curVertex, theConnectives, lightVec, true);
	
	// second pass, decreases the stencil value
	glCullFace(GL_BACK);
	glStencilOp(GL_KEEP, GL_KEEP, GL_DECR);

	ShadowPass(curConnective, curVertex, theConnectives, lightVec, true);


	glPopMatrix();
}

void TObject::ShadowPass(Connective *curConnective, Vertex *curVertex, Connective *theConnectives, Vector lightVec, bool flag) {
	for(int i = 0; i < this->triangleNum; i++) {
		if(curConnective->visible == flag) {
			for(int j = 0; j < 3; j++) {
//				if((curConnective + j)->neighbourIndex == UNDEFINED || (theConnectives + ((curConnective + j)->neighbourIndex) * 3)->visible == false ) {
					Vector v1 = (curVertex + j)->position;
					Vector v2 = (curVertex + ((j + 1) % 3) )->position;
					Vector v3, v4;
					Vsub(&v1, &lightVec, &v3);
					Vscale(&v3,  BIG_NUMBER);
					Vsub(&v2, &lightVec, &v4);
					Vscale(&v4,  BIG_NUMBER);

					glBegin(GL_QUADS);
						glVertex3f(v1.x, v1.y, v1.z);
						glVertex3f(v2.x, v2.y, v2.z);
						glVertex3f(v2.x + v4.x, v2.y + v4.y, v2.z + v4.z);
						glVertex3f(v1.x + v3.x, v1.y + v3.y, v1.z + v3.z);
					glEnd();

//				}
			}
		}
		curConnective = curConnective + 3;
		curVertex = curVertex + 3;
	}
}

