/***************************************************************************
                          vectorutils.cpp  -  description
                             -------------------
    begin                : Tue Dec 17 2002
    copyright            : (C) 2002 by Chong Jiayi
    email                : jychong@stanford.edu
 ***************************************************************************/


#include "vectorutils.h"
#include <iostream.h>

void Vzero(Vector *v) {
    v->x = 0.0;
    v->y = 0.0;
    v->z = 0.0;
}

void Vset(Vector *v, float x, float y, float z) {
    v->x = x;
    v->y = y;
    v->z = z;
}

bool Vequal(const Vector *v1, const Vector *v2) {
	return( (v1->x == v2->x) && (v1->y == v2->y) && (v1->z == v2->z));
}

void Vsub(Vector *src1, Vector *src2, Vector *dst) {
    dst->x = src1->x - src2->x;
    dst->y = src1->y - src2->y;
    dst->z = src1->z - src2->z;
}

void Vcopy(const Vector *v1, Vector *v2)
{
	*v2 = *v1;
}

void Vcross(Vector *v1, Vector *v2, Vector *cross) {
    cross->x = (v1->y * v2->z) - (v1->z * v2->y);
    cross->y = (v1->z * v2->x) - (v1->x * v2->z);
    cross->z = (v1->x * v2->y) - (v1->y * v2->x);
}

float Vsqlength(const Vector *v) {
  return v->x * v->x + v->y * v->y + v->z * v->z;
}

float Vlength(const Vector *v) {
    return (float) Sqrt(Vsqlength(v));
}

void Vscale(Vector *v, float div) {
    v->x *= div;
    v->y *= div;
    v->z *= div;
}

void Vnormal(Vector *v) {
	float mag = Vlength(v);
	if(mag > 0.0f) Vscale(v, 1.0f / mag );
}

float Vdot(const Vector *v1, const Vector *v2) {
    return v1->x*v2->x + v1->y*v2->y + v1->z*v2->z;
}

void Vadd(Vector *src1, Vector *src2, Vector *dst) {
    dst->x = src1->x + src2->x;
    dst->y = src1->y + src2->y;
    dst->z = src1->z + src2->z;
}

void Vinterpolate(Vector *from, Vector *to, Vector *result, int smoothness, int index) {
	float inv = 1.0f / (float)(smoothness);
	result->x = (to->x - from->x) * inv * index + from->x;
	result->y = (to->y - from->y) * inv * index + from->y;
	result->z = (to->z - from->z) * inv * index + from->z;
}

void PlaneInterpolate(Plane *from, Plane *to, Plane *result, int smoothness, int index) {
	float inv = 1.0f / (float)(smoothness);
	result->A = (to->A - from->A) * inv * index + from->A;
	result->B = (to->B - from->B) * inv * index + from->B;
	result->C = (to->C - from->C) * inv * index + from->D;
	result->D = (to->D - from->D) * inv * index + from->D;
}

/*** FAST INVERSE SQUARE ROOT ***/

/* From "Graphics Gems V", Alan Paeth (Editor)
 * ISBN 0125434553/9649-1547332-306386
 * Published by Ap Profession, 1995
 */

/* Compute the Inverse Square Root
 * of an IEEE Single Precision Floating-Point number.
 *
 * Written by Ken Turkowski.
 */

/* Specified parameters */
#define LOOKUP_BITS    6   /* Number of mantissa bits for lookup */
#define EXP_POS       23   /* Position of the exponent */
#define EXP_BIAS     127   /* Bias of exponent */
/* The mantissa is assumed to be just down from the exponent */

/* Derived parameters */
#define LOOKUP_POS   (EXP_POS-LOOKUP_BITS)  /* Position of mantissa lookup */
#define SEED_POS     (EXP_POS-8)            /* Position of mantissa seed */
#define TABLE_SIZE   (2 << LOOKUP_BITS)     /* Number of entries in table */
#define LOOKUP_MASK  (TABLE_SIZE - 1)           /* Mask for table input */
#define GET_EXP(a)   (((a) >> EXP_POS) & 0xFF)  /* Extract exponent */
#define SET_EXP(a)   ((a) << EXP_POS)           /* Set exponent */
#define GET_EMANT(a) (((a) >> LOOKUP_POS) & LOOKUP_MASK)  /* Extended mantissa
                                                           * MSB's */
#define SET_MANTSEED(a) (((unsigned long)(a)) << SEED_POS)  /* Set mantissa
                                                             * 8 MSB's */

static unsigned char iSqrt[TABLE_SIZE];

union _flint {
  unsigned long    i;
  float            f;
} _fi, _fo;

void makeInverseSqrtLookupTable(void) {
  register long f;
  register unsigned char *h;
  union _flint fi, fo;

  h = iSqrt;
  for (f = 0, h = iSqrt; f < TABLE_SIZE; f++) {
    fi.i = ((EXP_BIAS-1) << EXP_POS) | (f << LOOKUP_POS);
    fo.f = (float) (1.0 / sqrt(fi.f));
    *h++ = (unsigned char)
           (((fo.i + (1<<(SEED_POS-2))) >> SEED_POS) & 0xFF); /* rounding */
  }
  iSqrt[TABLE_SIZE / 2] = 0xFF;    /* Special case for 1.0 */
}

float Sqrt(float x) {
	return (1 / invSqrt(x));
}

/* The following returns the inverse square root. */
float invSqrt(float x) {
  unsigned long a = ((union _flint*)(&x))->i;
  float arg = x;
  union _flint seed;
  float r;

  seed.i = SET_EXP(((3*EXP_BIAS-1) - GET_EXP(a)) >> 1)
         | SET_MANTSEED(iSqrt[GET_EMANT(a)]);

  /* Seed: accurate to LOOKUP_BITS */
  r = seed.f;

  /* First iteration: accurate to 2*LOOKUP_BITS */
  r = (float) ((3.0 - r * r * arg) * r * 0.5);

#if 0  /* Wow!  We don't need this much precision! */
  /* Second iteration: accurate to 4*LOOKUP_BITS */
  r = (float) ((3.0 - r * r * arg) * r * 0.5);
#endif

  return r;
}

/*
 * Compute inverse of 4x4 transformation SINGLE-PRECISION matrix.
 * Code contributed by Jacques Leroy <jle@star.be>
 * Code lifted from Brian Paul's Mesa freeware OpenGL implementation.
 * Return GL_TRUE for success, GL_FALSE for failure (singular matrix)
 */
GLboolean invertMatrixf(GLfloat *out, const GLfloat *m) {
/* NB. OpenGL Matrices are COLUMN major. */
#define SWAP_ROWS(a, b) { GLdouble *_tmp = a; (a)=(b); (b)=_tmp; }
#define MAT(m,r,c) (m)[(c)*4+(r)]

  GLdouble wtmp[4][8];
  GLdouble m0, m1, m2, m3, s;
  GLdouble *r0, *r1, *r2, *r3;

  r0 = wtmp[0], r1 = wtmp[1], r2 = wtmp[2], r3 = wtmp[3];

  r0[0] = MAT(m,0,0), r0[1] = MAT(m,0,1),
  r0[2] = MAT(m,0,2), r0[3] = MAT(m,0,3),
  r0[4] = 1.0, r0[5] = r0[6] = r0[7] = 0.0,

  r1[0] = MAT(m,1,0), r1[1] = MAT(m,1,1),
  r1[2] = MAT(m,1,2), r1[3] = MAT(m,1,3),
  r1[5] = 1.0, r1[4] = r1[6] = r1[7] = 0.0,

  r2[0] = MAT(m,2,0), r2[1] = MAT(m,2,1),
  r2[2] = MAT(m,2,2), r2[3] = MAT(m,2,3),
  r2[6] = 1.0, r2[4] = r2[5] = r2[7] = 0.0,

  r3[0] = MAT(m,3,0), r3[1] = MAT(m,3,1),
  r3[2] = MAT(m,3,2), r3[3] = MAT(m,3,3),
  r3[7] = 1.0, r3[4] = r3[5] = r3[6] = 0.0;

  /* choose pivot - or die */
  if (fabs(r3[0])>fabs(r2[0])) SWAP_ROWS(r3, r2);
  if (fabs(r2[0])>fabs(r1[0])) SWAP_ROWS(r2, r1);
  if (fabs(r1[0])>fabs(r0[0])) SWAP_ROWS(r1, r0);
  if (0.0 == r0[0]) {
    return GL_FALSE;
  }

  /* eliminate first variable     */
  m1 = r1[0]/r0[0]; m2 = r2[0]/r0[0]; m3 = r3[0]/r0[0];
  s = r0[1]; r1[1] -= m1 * s; r2[1] -= m2 * s; r3[1] -= m3 * s;
  s = r0[2]; r1[2] -= m1 * s; r2[2] -= m2 * s; r3[2] -= m3 * s;
  s = r0[3]; r1[3] -= m1 * s; r2[3] -= m2 * s; r3[3] -= m3 * s;
  s = r0[4];
  if (s != 0.0) { r1[4] -= m1 * s; r2[4] -= m2 * s; r3[4] -= m3 * s; }
  s = r0[5];
  if (s != 0.0) { r1[5] -= m1 * s; r2[5] -= m2 * s; r3[5] -= m3 * s; }
  s = r0[6];
  if (s != 0.0) { r1[6] -= m1 * s; r2[6] -= m2 * s; r3[6] -= m3 * s; }
  s = r0[7];
  if (s != 0.0) { r1[7] -= m1 * s; r2[7] -= m2 * s; r3[7] -= m3 * s; }

  /* choose pivot - or die */
  if (fabs(r3[1])>fabs(r2[1])) SWAP_ROWS(r3, r2);
  if (fabs(r2[1])>fabs(r1[1])) SWAP_ROWS(r2, r1);
  if (0.0 == r1[1]) {
    return GL_FALSE;
  }

  /* eliminate second variable */
  m2 = r2[1]/r1[1]; m3 = r3[1]/r1[1];
  r2[2] -= m2 * r1[2]; r3[2] -= m3 * r1[2];
  r2[3] -= m2 * r1[3]; r3[3] -= m3 * r1[3];
  s = r1[4]; if (0.0 != s) { r2[4] -= m2 * s; r3[4] -= m3 * s; }
  s = r1[5]; if (0.0 != s) { r2[5] -= m2 * s; r3[5] -= m3 * s; }
  s = r1[6]; if (0.0 != s) { r2[6] -= m2 * s; r3[6] -= m3 * s; }
  s = r1[7]; if (0.0 != s) { r2[7] -= m2 * s; r3[7] -= m3 * s; }

  /* choose pivot - or die */
  if (fabs(r3[2])>fabs(r2[2])) SWAP_ROWS(r3, r2);
  if (0.0 == r2[2]) {
    return GL_FALSE;
  }

  /* eliminate third variable */
  m3 = r3[2]/r2[2];
  r3[3] -= m3 * r2[3], r3[4] -= m3 * r2[4],
  r3[5] -= m3 * r2[5], r3[6] -= m3 * r2[6],
  r3[7] -= m3 * r2[7];

  /* last check */
  if (0.0 == r3[3]) {
    return GL_FALSE;
  }

  s = 1.0/r3[3];              /* now back substitute row 3 */
  r3[4] *= s; r3[5] *= s; r3[6] *= s; r3[7] *= s;

  m2 = r2[3];                 /* now back substitute row 2 */
  s  = 1.0/r2[2];
  r2[4] = s * (r2[4] - r3[4] * m2), r2[5] = s * (r2[5] - r3[5] * m2),
  r2[6] = s * (r2[6] - r3[6] * m2), r2[7] = s * (r2[7] - r3[7] * m2);
  m1 = r1[3];
  r1[4] -= r3[4] * m1, r1[5] -= r3[5] * m1,
  r1[6] -= r3[6] * m1, r1[7] -= r3[7] * m1;
  m0 = r0[3];
  r0[4] -= r3[4] * m0, r0[5] -= r3[5] * m0,
  r0[6] -= r3[6] * m0, r0[7] -= r3[7] * m0;

  m1 = r1[2];                 /* now back substitute row 1 */
  s  = 1.0/r1[1];
  r1[4] = s * (r1[4] - r2[4] * m1), r1[5] = s * (r1[5] - r2[5] * m1),
  r1[6] = s * (r1[6] - r2[6] * m1), r1[7] = s * (r1[7] - r2[7] * m1);
  m0 = r0[2];
  r0[4] -= r2[4] * m0, r0[5] -= r2[5] * m0,
  r0[6] -= r2[6] * m0, r0[7] -= r2[7] * m0;

  m0 = r0[1];                 /* now back substitute row 0 */
  s  = 1.0/r0[0];
  r0[4] = s * (r0[4] - r1[4] * m0), r0[5] = s * (r0[5] - r1[5] * m0),
  r0[6] = s * (r0[6] - r1[6] * m0), r0[7] = s * (r0[7] - r1[7] * m0);

  MAT(out,0,0) = r0[4]; MAT(out,0,1) = r0[5],
  MAT(out,0,2) = r0[6]; MAT(out,0,3) = r0[7],
  MAT(out,1,0) = r1[4]; MAT(out,1,1) = r1[5],
  MAT(out,1,2) = r1[6]; MAT(out,1,3) = r1[7],
  MAT(out,2,0) = r2[4]; MAT(out,2,1) = r2[5],
  MAT(out,2,2) = r2[6]; MAT(out,2,3) = r2[7],
  MAT(out,3,0) = r3[4]; MAT(out,3,1) = r3[5],
  MAT(out,3,2) = r3[6]; MAT(out,3,3) = r3[7];

  return GL_TRUE;

#undef MAT
#undef SWAP_ROWS
}

void buildFrustumMatrix(GLdouble m[16],
                   GLdouble l, GLdouble r, GLdouble b, GLdouble t,
                   GLdouble n, GLdouble f)
{
  m[0] = (2.0*n) / (r-l);
  m[1] = 0.0;
  m[2] = 0.0;
  m[3] = 0.0;

  m[4] = 0.0;
  m[5] = (2.0*n) / (t-b);
  m[6] = 0.0;
  m[7] = 0.0;

  m[8] = (r+l) / (r-l);
  m[9] = (t+b) / (t-b);
  m[10] = -(f+n) / (f-n);
  m[11] = -1.0;

  m[12] = 0.0;
  m[13] = 0.0;
  m[14] = -(2.0*f*n) / (f-n);
  m[15] = 0.0;
}

void buildPerspectiveMatrix(GLdouble m[16],
                       GLdouble fovy, GLdouble aspect,
                       GLdouble zNear, GLdouble zFar)
{
  GLdouble xmin, xmax, ymin, ymax;

  ymax = zNear * tan(fovy * MATH_PI / 360.0f);
  ymin = -ymax;

  xmin = ymin * aspect;
  xmax = ymax * aspect;

  buildFrustumMatrix(m, xmin, xmax, ymin, ymax, zNear, zFar);
}

/* Build a 4x4 matrix transform based on the parameters for gluLookAt.
 */
void buildLookAtMatrix(GLdouble m[16], Vector eye, Vector center, Vector up)
{
   GLdouble x[3], y[3], z[3];
   Vector vecX, vecY, vecZ;
   GLdouble mag;

   /* Make rotation matrix */

   /* Z vector */
   Vsub(&eye, &center, &vecZ);
   Vnormal(&vecZ);

   z[0] = vecZ.x;
   z[1] = vecZ.y;
   z[2] = vecZ.z;
 
   Vcross(&up, &vecZ, &vecX);
   Vcross(&vecZ, &vecX, &vecY);
   Vnormal(&vecX);
   Vnormal(&vecY);

   x[0] = vecX.x;
   x[1] = vecX.y;
   x[2] = vecZ.z;
  
   y[0] = vecY.x;
   y[1] = vecY.y;
   y[2] = vecY.z;

#define M(row,col)  m[col*4+row]
   M(0,0) = x[0];  M(0,1) = x[1];  M(0,2) = x[2];  M(0,3) = -x[0]*eye.x + -x[1]*eye.y + -x[2]*eye.z;
   M(1,0) = y[0];  M(1,1) = y[1];  M(1,2) = y[2];  M(1,3) = -y[0]*eye.x + -y[1]*eye.y + -y[2]*eye.z;
   M(2,0) = z[0];  M(2,1) = z[1];  M(2,2) = z[2];  M(2,3) = -z[0]*eye.x + -z[1]*eye.y + -z[2]*eye.z;
   M(3,0) = 0.0;   M(3,1) = 0.0;   M(3,2) = 0.0;   M(3,3) = 1.0;
#undef M
}


/* Applies matrix "m" onto the "in" vector and outputs it as "out" */
void transformVector(Vector *out, Vector *in, float m[16]) {
  float w;

 out->x = *(m) * in->x + *(m+4) * in->y + *(m+8) * in->z + *(m+12);
 out->y = *(m+1) * in->x + *(m+5) * in->y + *(m+9) * in->z + *(m+13);
 out->z = *(m+2) * in->x + *(m+6) * in->y + *(m+10) * in->z + *(m+14);
}

/* Creates a plane out of 3 points*/
void makePlane(Vector *in1, Vector *in2, Vector *in3, Plane * outPlane) {
	Vector workVec1, workVec2, workVec3;
   //find to vectors parallel to the plane
	Vsub(in1, in2, (&workVec1));
	Vsub(in3, in2, (&workVec2));
	Vcross((&workVec1), (&workVec2), (&workVec3) );
	 outPlane->A = workVec3.x;
	 outPlane->B = workVec3.y;
	 outPlane->C = workVec3.z;
	 outPlane->D = -(outPlane->A * in1->x + outPlane->B * in1->y + outPlane->C * in1->z);
}

float vecPlaneOrientation(Vector *vec, Plane *plane) {
	return (vec->x * plane->A + vec->y * plane->B + vec->z * plane->C + plane->D);
}
