/***************************************************************************
                          tinfrastructure.cpp  -  description
                             -------------------
    begin                : Sun Dec 15 2002
    copyright            : (C) 2002 by Chong Jiayi
    email                : jychong@stanford.edu
 ***************************************************************************/



#include "tinfrastructure.h"

const int NUM_ASCII = 256;
const int FONT_OFFSET_X = 10;
const int FONT_WIDTH = 16;
const int FONT_HEIGHT = 16;


static int is_Space(int c)
{
	if (c == ' ' || c == '\t' || c == '\n' || c == '\r') return true;
	else return false;
}

static char * is_fgets(char *buf, int n, FILE *fp)
{
	char *wbuf;
	int c, bcnt = 0;

	if (n <= 0) return (NULL);

	wbuf = buf; n = n - 1;                   

	c = '\t';
	while(is_Space(c)) { c = getc(fp); }

	while (n != 0) {
		if (c == '\n' || c == '\r') break;
		if (c == EOF) 
			if (bcnt == 0) return NULL; else break;
		buf[bcnt] = c; bcnt++;
		n = n - 1;
		c = getc(fp);
	}

	buf[bcnt] = '\0';
	return buf;
}


TInfrastructure::TInfrastructure(int Width, int Height, int PosX, int PosY, char * Title){
	this->width = Width;
	this->height = Height;
	this->posX = PosX;
	this->posY = PosY;
	this->title = Title;
	cubeMapSelect = 0;
	
	for(int i = 0; i < 6; i++) {
		skyboxTexture[i] = 0;
	}
	//initialize network
	curNetworkObj = new TNetwork();
	
}



TInfrastructure::~TInfrastructure() {
	//clean up
	for(int i = 0; i < textureList.size(); i++) { //loop through and delete all the loaded textures
		if(textureList[i].decIndex != UNDEFINED) glDeleteTextures(1, (unsigned int *)(&(textureList[i].decIndex)) );
		if(textureList[i].bumpIndex != UNDEFINED) glDeleteTextures(1, (unsigned int *)(&(textureList[i].bumpIndex)));                                                                                          
	}
	
	for(i = 0; i < windowTexture.size(); i++) {
		glDeleteTextures(1, (unsigned int *)(&windowTexture[i]));
	}
	
	for(i = 0; i < bufferList.size(); i++) {
		glDeleteBuffersARB(1, &bufferList[i].index);
	}

	glDeleteTextures(1, (unsigned int*)(&attenuateTexture2D));
	glDeleteTextures(1, (unsigned int*)(&attenuateTexture1D));
	glDeleteTextures(1, (unsigned int*)(&textTexture));

	glDeleteLists(textRef, 256);
	
	for(i = 0; i < soundList.size(); i++) {
		Mix_FreeChunk(soundList[i]);
	}
	
	delete curParticle;

	delete curNetworkObj;
}

int TInfrastructure::CreateGL(bool fullScreen) {
	//initialize video and stuff
	const SDL_VideoInfo* info = NULL;
	Uint32 flags = false;
	int size = 0;
	/* Initialize SDL */
	if ( SDL_Init(SDL_INIT_VIDEO | SDL_INIT_AUDIO) < 0 ) {
		fprintf(stderr, "Couldn't init SDL: %s\n", SDL_GetError());
		return false;
	}
	
	if(Mix_OpenAudio(44100, AUDIO_S16SYS, 2, 2048) < 0) {
		printf("Warning: Couldn't set 44100 Hz 16-bit audio\n\- Reason: %s\n", SDL_GetError());
	}
	Mix_AllocateChannels(16) ;

	SDL_WM_SetCaption(title, NULL);

	SDL_GL_SetAttribute( SDL_GL_RED_SIZE, 8 );
	SDL_GL_SetAttribute( SDL_GL_GREEN_SIZE, 8 );
	SDL_GL_SetAttribute( SDL_GL_BLUE_SIZE, 8 );    
	SDL_GL_SetAttribute( SDL_GL_ALPHA_SIZE, 8 );
	SDL_GL_SetAttribute( SDL_GL_STENCIL_SIZE, 8 );	
	SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );   
	SDL_GL_SetAttribute( SDL_GL_ACCUM_RED_SIZE, 0);     // this and the next three lines set the bits allocated per pixel -
	SDL_GL_SetAttribute( SDL_GL_ACCUM_GREEN_SIZE, 0);   // - for the accumulation buffer to 0
	SDL_GL_SetAttribute( SDL_GL_ACCUM_BLUE_SIZE, 0);
	SDL_GL_SetAttribute( SDL_GL_ACCUM_ALPHA_SIZE, 0);


	/* Let's get some video information. */
	info = SDL_GetVideoInfo( );
	
	flags = SDL_OPENGL;
	if(fullScreen) flags = SDL_OPENGL | SDL_FULLSCREEN;
	
	if ( SDL_SetVideoMode(width, height, info->vfmt->BitsPerPixel, flags) == NULL ) {
		fprintf(stderr, "Couldn't init SDL: %s\n", SDL_GetError());		
		return false;
	}

	int rBits = 0, gBits = 0, bBits = 0, alphaBits = 0, stencilSize = 0;
	SDL_GL_GetAttribute(SDL_GL_RED_SIZE, &rBits);
	SDL_GL_GetAttribute(SDL_GL_BLUE_SIZE, &bBits);
	SDL_GL_GetAttribute(SDL_GL_GREEN_SIZE, &gBits);
	SDL_GL_GetAttribute(SDL_GL_STENCIL_SIZE, &stencilSize);
	SDL_GL_GetAttribute(SDL_GL_ALPHA_SIZE, &alphaBits);	
	
	printf("Red Bits:%d Green Bits:%d Blue Bits:%d Alpha Bits:%d Stencil Bits:%d\n", rBits, gBits, bBits, alphaBits, stencilSize);
	extgl_Initialize();
	
	if (extgl_Extensions.OpenGL14) cout << "OpenGL 1.4 extensions supported" << endl;
	else cout << "OpenGL 1.4 extensions not supported!" << endl;
	
	if (extgl_Extensions.EXT_texture_compression_s3tc) cout << "S3TC Texture Compression supported" << endl;
	else cout << "S3TC Texture Compression not supported!" << endl;
	
	if (extgl_Extensions.NV_register_combiners) cout << "NVIDIA Register Combiners supported" << endl;
	else cout << "NVIDIA Register Combiners not supported!" << endl;
	
	ReSizeGLScene(width, height);	
	//openGL initialization
	glEnable(GL_TEXTURE_2D);							// Enable Texture Mapping
	glShadeModel(GL_SMOOTH);							// Enable Smooth Shading
	glClearColor(0.00f, 0.00f, 0.00f, 0.0f);			// Black Background
	glClearDepth(1.0f);									// Depth Buffer Setup
	glEnable(GL_DEPTH_TEST);							// Enables Depth Testing
	glDepthFunc(GL_LEQUAL);								// The Type Of Depth Testing To Do
	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);	// Really Nice Perspective Calculations
	glDisable(GL_LIGHTING);
	glFrontFace(GL_CCW);
	glEnable(GL_CULL_FACE );
	glClearStencil(0);									// Stencil Buffer Setup

	glActiveTextureARB(GL_TEXTURE0_ARB);  //cube map for perpixel lighting
	glBindTexture(GL_TEXTURE_CUBE_MAP_ARB, TO_NORMALIZE_CUBE_MAP);
	makeNormalizeVectorCubeMap(32);
	glEnable(GL_TEXTURE_CUBE_MAP_ARB);
	glActiveTextureARB(GL_TEXTURE1_ARB);
	
	gliGenericImage *image2D;
	glGenTextures(1, (unsigned int *)(&attenuateTexture2D));    //load texture for light attenuation2D
	glBindTexture(GL_TEXTURE_2D, attenuateTexture2D);
	image2D = readImage("PointLight2D.tga");
	glTexImage2D(GL_TEXTURE_2D, 0, GL_COMPRESSED_RGB_S3TC_DXT1_EXT,
		image2D->width, image2D->height, 0, image2D->format, GL_UNSIGNED_BYTE, image2D->pixels);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	free(image2D);

	gliGenericImage *image1D;
	glGenTextures(1, (unsigned int *)(&attenuateTexture1D));    //load texture for light attenuation1D
	glBindTexture(GL_TEXTURE_1D, attenuateTexture1D);
	image1D = readImage("PointLight1D.tga");
	glTexImage1D(GL_TEXTURE_1D, 0, GL_COMPRESSED_RGB_S3TC_DXT1_EXT,
		image1D->width, 0, image1D->format, GL_UNSIGNED_BYTE, image1D->pixels);
	glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	free(image1D);

	glGenTextures(1, (unsigned int *)(&textTexture));    //load texture for fonts
	glBindTexture(GL_TEXTURE_2D, textTexture);
	SDL_Surface *	textImage2D = SDL_LoadBMP("font.bmp");
	glTexImage2D(GL_TEXTURE_2D, 0, 3, textImage2D->w, textImage2D->h, 0, GL_BGR, GL_UNSIGNED_BYTE, textImage2D->pixels);

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

	SDL_FreeSurface(textImage2D);

	//build up the lens flare textures
	for(int k = 0; k < 4; k++) {
		glGenTextures(1, (unsigned int *)(&lensflareTexture[k]));    //load texture for light attenuation2D
		glBindTexture(GL_TEXTURE_2D, lensflareTexture[k]);
		char outputFile[100];
		sprintf(outputFile, "./lensflare/lens%d.tga", k + 1);
		image2D = readImage(outputFile);
		glTexImage2D(GL_TEXTURE_2D, 0, GL_COMPRESSED_RGB_S3TC_DXT1_EXT,
			image2D->width, image2D->height, 0, image2D->format, GL_UNSIGNED_BYTE, image2D->pixels);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		free(image2D);
	}

	BuildFont();
	
	createWindowTextures(NUM_WINDOW_TEXTURES + 1, width, height);

	this->curParticle = new TParticle();
	
	LoadVertexProgramNV("./refract_1_NV.vp");
	LoadVertexProgramNV("./refract_2_NV.vp");
	
	timerVal = SDL_GetTicks();
	timerVal2 = SDL_GetTicks();
	srand(timerVal);
	
	return true;
}

// Resize And Initialize The GL Window
void TInfrastructure::ReSizeGLScene(GLsizei width, GLsizei height)
{
	if (height==0) { 										// Prevent A Divide By Zero By
		height=1;										// Making Height Equal One
	}

	glViewport(0,0,width,height);						// Reset The Current Viewport

	glMatrixMode(GL_PROJECTION);						// Select The Projection Matrix
	glLoadIdentity();									// Reset The Projection Matrix

	// Calculate The Aspect Ratio Of The Window
	iniProjectionMatrix();
	glMatrixMode(GL_MODELVIEW);							// Select The Modelview Matrix
	glLoadIdentity();									// Reset The Modelview Matrix
}

void TInfrastructure::KillGLWindow() {
	SDL_Quit();
}

/* Based on Mark Kilgard's Paper "A Practical and Robust Bump-mapping Technique
    for Today's GPUs", http://developer.nvidia.com
    Here we try to convert a .tga file into a normal map format that the nvidia hardware can read in. The normal map file loading code
    is based off Mark Kilgard's paper.
*/

Normal * TInfrastructure::convertHeightFieldToNormalMap(GLubyte *pixels, int w, int h, int wr, int hr, float scale) {
  Normal *nmap;
  float sqlen, reciplen, nx, ny, nz;
  const float recp255 = 1.0f/255.0f;
  float c, cx, cy;

  nmap = new Normal[w * h]; 

  for (int i=0; i<h; i++) {
    for (int j=0; j<w; j++) {
      int index = i * wr;
      c = pixels[index + j] * recp255;
      cx = pixels[index + (j+1)%wr] * recp255;
      cy = pixels[((i+1)%hr)*wr + j] * recp255;
       /* Normalize the vector. */
      Vector curDC;
      Vset(&curDC, scale * (c - cy), scale * (c - cx), 1.0f);
      Vnormal(&curDC);
      /* Repack the normalized vector into an RGB unsigned byte
         vector in the normal map image. */
      int index2 = i*w+j;
      nmap[index2].nx = 128 + 127*curDC.x;
      nmap[index2].ny = 128 + 127*curDC.y;
      nmap[index2].nz = 128 + 127*curDC.z;
      nmap[index2].mag = 255;
    }
  }

  return nmap;
}


void fetchVecMagnitude(float *mag00, float *mag01, float *mag10, float *mag11, Normal *curmap, int i, int j, int width, int height) {
	int vecIndex1 = i * width;
	int vecIndex2 = ((i+1)%width)*width;
	const float recp255 = 1.0/255.0;
	*mag00 = recp255 * curmap[vecIndex1 + j].mag;
	*mag01 = recp255 * curmap[vecIndex1 + ((j+1)%height)].mag;
	*mag10 = recp255 * curmap[vecIndex2 + j].mag;
	*mag11 = recp255 * curmap[vecIndex2 + ((j+1)%height)].mag;
}

void setMapVec(float mag00, float mag01, float mag10, float mag11, Normal *curmap, int i, int j, int width, int height, Vector *curVec) {
	const float recp127 = 1.0/127.0;
	int vecIndex1 = i * width;
	int vecIndex2 = ((i+1)%width)*width;
	#define SumFootPrintX mag00 * (recp127 * curmap[vecIndex1 + j].nx - 1.0) + mag01 * (recp127 * curmap[vecIndex1 + ((j+1)%height)].nx - 1.0) + mag10 * (recp127 * curmap[vecIndex2 + j].nx - 1.0) + mag11 * (recp127 * curmap[vecIndex2 + ((j+1)%height)].nx - 1.0)
	#define SumFootPrintY mag00 * (recp127 * curmap[vecIndex1 + j].ny - 1.0) + mag01 * (recp127 * curmap[vecIndex1 + ((j+1)%height)].ny - 1.0) + mag10 * (recp127 * curmap[vecIndex2 + j].ny - 1.0) + mag11 * (recp127 * curmap[vecIndex2 + ((j+1)%height)].ny - 1.0)
	 #define SumFootPrintZ mag00 * (recp127 * curmap[vecIndex1 + j].nz - 1.0) + mag01 * (recp127 * curmap[vecIndex1 + ((j+1)%height)].nz - 1.0) + mag10 * (recp127 * curmap[vecIndex2 + j].nz - 1.0) + mag11 * (recp127 * curmap[vecIndex2 + ((j+1)%height)].nz - 1.0)
	curVec->x = SumFootPrintX;
	curVec->y = SumFootPrintY;
	curVec->z = SumFootPrintZ;
	#undef SumFootPrintX
	#undef SumFootPrintY
	#undef SumFootPrintZ
}

/* Based on Mark Kilgard's Paper "A Practical and Robust Bump-mapping Technique
    for Today's GPUs", http://developer.nvidia.com
   Given a normal map, create a downsampled version of the normal map
   at half the width and height.  Use a 2x2 box filter to create each
   downsample.  gluBuild2DMipmaps is not suitable because each downsampled
   texel must also be renormalized. */
   
   /* Documentation taken from the paper as well to aid coding since it is very self-explanatory and important to understand*/
Normal * TInfrastructure::downSampleNormalMap(Normal *old, int w2, int h2, int width, int height) {
  const float recp127 = 1.0/127.0;
  const float recp255 = 1.0/255.0;

  Normal *nmap;
  Vector curVec;
  float invl, length;
  float mag00, mag01, mag10, mag11;
  int ii, jj;

  /* Allocate space for the downsampled normal map level. */
  nmap = new Normal[width * height];

  for (int i=0; i<h2; i+=2) {
    for (int j=0; j<w2; j+=2) {

      /* Fetch the magnitude of the four vectors to be downsampled. */
      int vecIndex1 = i * w2;
      int vecIndex2 = ((i+1)%w2)*w2;
      fetchVecMagnitude(&mag00, &mag01, &mag10, &mag11, old, i, j, w2, h2);
      setMapVec(mag00, mag01, mag10, mag11, old, i, j, w2, h2, &curVec);
      /* Compute length of the (x,y,z) vector. */
      length = Vlength(&curVec);
      if (length == 0.0) Vset(&curVec, 0.0f, 0.0f, 1.0f);
      else {
        invl = 1.0/length;
	Vscale(&curVec, invl);
      }

      /* Pack the normalized vector into an RGB unsigned byte vector
         in the downsampled image. */
      int factorX = (i >> 1) * width + (j >> 1);
      nmap[factorX].nx = 128 + 127*curVec.x;
      nmap[factorX].ny = 128 + 127*curVec.y;
      nmap[factorX].nz = 128 + 127*curVec.z;

      /* Store the magnitude of the average vector in the alpha
         component so we keep track of the magntiude. */
      length = length * 0.25;
      if (length > 1.0) nmap[factorX].mag = 255;
      else nmap[factorX].mag = 255*length;
    }
  }

  delete [] old;

  return nmap;
}

/* Convert the supplied height-field image into a normal map (a normalized
   vector compressed to the [0,1] range in RGB and A=1.0).  Load the
   base texture level, then recursively downsample and load successive
   normal map levels */
void TInfrastructure::convertHeightFieldAndLoadNormalMapTexture(GLubyte *pixels, int w, int h, int wr, int hr, float scale) {
  Normal *nmap = convertHeightFieldToNormalMap(pixels, w, h, wr, hr, scale);
  int level = 0, nw = 0, nh = 0;

  /* The BGRA color component ordering is fastest for NVIDIA. */
  glTexImage2D(GL_TEXTURE_2D, level, GL_RGBA, w, h, level,
  GL_BGRA_EXT, GL_UNSIGNED_BYTE, &nmap->nz);

  /* Downsample the normal map for mipmap levels down to 1x1. */
  while (w > 1 || h > 1) {
    level++;
    #define CLAMP(x) if(x == 0) x = 1
    nw = w >> 1;
    nh = h >> 1;
    CLAMP(nw); 
    CLAMP(nh);

    nmap = downSampleNormalMap(nmap, w, h, nw, nh);
    glTexImage2D(GL_TEXTURE_2D, level, GL_RGBA, nw, nh, 0,
      GL_BGRA_EXT, GL_UNSIGNED_BYTE, &nmap->nz);

    w = nw;
    h = nh;
  }

   delete [] nmap;
   #undef CLAMP
}


/* Given a cube map face index, cube map size, and integer 2D face position,
 * return the cooresponding normalized vector.
 */
void TInfrastructure::getCubeVector(int i, int cubesize, int x, int y, Vector *curVec) {
  float s, t, sc, tc;

  s = ((float)x + 0.5) / (float)cubesize;
  t = ((float)y + 0.5) / (float)cubesize;
  sc = s*2.0 - 1.0;
  tc = t*2.0 - 1.0;

  switch (i) {
  case 0:
    curVec->x = 1.0f;
    curVec->y = -tc;
    curVec->z = -sc;
    break;
  case 1:
    curVec->x = -1.0f;
    curVec->y = -tc;
    curVec->z = sc;
    break;
  case 2:
    curVec->x = sc;
    curVec->y = 1.0;
    curVec->z = tc;
    break;
  case 3:
    curVec->x = sc;
    curVec->y = -1.0f;
    curVec->z = -tc;
    break;
  case 4:
    curVec->x = sc;
    curVec->y = -tc;
    curVec->z = 1.0f;
    break;
  case 5:
    curVec->x = -sc;
    curVec->y = -tc;
    curVec->z = -1.0f;
    break;
  }

  Vnormal(curVec);
}

/* Initialize a cube map texture object that generates RGB values
 * that when expanded to a [-1,1] range in the register combiners
 * form a normalized vector matching the per-pixel vector used to
 * access the cube map.
 */

//helper function that writes the output to the cube map
void writeCubePixels(GLubyte *pixels, Vector *curVec, int x, int y, int size) {
	int index = 3*(y*size+x);
	pixels[index + 0] = 128 + 127*curVec->x;
	pixels[index + 1] = 128 + 127*curVec->y;
	pixels[index + 2] = 128 + 127*curVec->z;
}

void TInfrastructure::makeNormalizeVectorCubeMap(int size) {
  Vector curVec;
  GLubyte *pixels;

  pixels = (GLubyte*) new GLubyte[size * size * 3]; 
  if (pixels == NULL) {
    cerr << "Memory Allocation failed!" << endl;
    exit(1);
  }

  glTexParameteri(GL_TEXTURE_CUBE_MAP_ARB, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
  glTexParameteri(GL_TEXTURE_CUBE_MAP_ARB, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
  glTexParameteri(GL_TEXTURE_CUBE_MAP_ARB, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  glTexParameteri(GL_TEXTURE_CUBE_MAP_ARB, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

  for (int i = 0; i < 6; i++) {
    for (int y = 0; y < size; y++) {
      for (int x = 0; x < size; x++) {
        getCubeVector(i, size, x, y, &curVec);
        writeCubePixels(pixels, &curVec, x, y, size);
      }
    }
    glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_X_ARB+i, 0, GL_RGB,
      size, size, 0, GL_RGB, GL_UNSIGNED_BYTE, pixels);
  }

  delete [] pixels;
}

/*** LOAD TEXTURE IMAGES ***/



gliGenericImage * TInfrastructure::readImage(char *filename)
{
	return readTGAImage(filename);
}

void TInfrastructure::loadTextureDecalImage(char *filename, int mipmaps)
{
  gliGenericImage *image;

  image = readImage(filename);
  if (image->format == GL_COLOR_INDEX) {
    /* Rambo 8-bit color index into luminance. */
    image->format = GL_LUMINANCE;
  }
  if (mipmaps) {
    gluBuild2DMipmaps(GL_TEXTURE_2D, GL_COMPRESSED_RGB_S3TC_DXT1_EXT, 
      image->width, image->height,
      image->format, GL_UNSIGNED_BYTE, image->pixels);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
      GL_LINEAR_MIPMAP_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  } else {
    glTexImage2D(GL_TEXTURE_2D, 0, GL_COMPRESSED_RGB_S3TC_DXT1_EXT, 
      image->width, image->height, 0,
      image->format, GL_UNSIGNED_BYTE, image->pixels);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  }

}

/* Loads a texture normal map. Since OpenGL only allows us to load textures that have sizes of powers of 2, we have to check
    before we load. Checking/low-level utility code is referenced from nvidia's developer website: http://developer.nvidia.com*/
void TInfrastructure::loadTextureNormalMap(char *filename, float scale)
{
  gliGenericImage *image = NULL;
  int w = 0, h = 0, wr = 0, hr = 0, badSize = false;

  image = readImage(filename);
  if (image->components != 1) {
    cerr << filename << "not luminance height field image" << endl;
    exit(1);
  }

  w = image->width;
  h = image->height;

  if ( (w & (w-1))) {
    if ( ((w-1) & (w-2))) {
      badSize = true;
    } else {
      wr = w;
      w = w-1;
    }
  } else {
    wr = w;
  }

  if ( (h & (h-1))) {
    if ( ((h-1) & (h-2))) {
      badSize = true;
    } else {
      hr = h;
      h = h-1;
    }
  } else {
    hr = h;
  }

  if (badSize) {
       cerr << filename << "not  2^n/2^n + 1 dimenstions" << endl;
    exit(1);
  }

  convertHeightFieldAndLoadNormalMapTexture(image->pixels,
    w, h, wr, hr, scale);
}

/* Loads in the decal and bump-map texture and then returns their information in a TextureInfo structure*/
TextureInfo *TInfrastructure::LoadTexture(char *decalFile, int mipmaps, char *bumpFile, float scale) {
	bool decalSkip = false, bumpSkip = false;
	TextureInfo *retInfo = new TextureInfo;

	retInfo->bumpIndex = UNDEFINED;
	retInfo->decIndex = UNDEFINED;

	//see if the texture was already loaded
	if(decalFile != NULL && bumpFile != NULL) {
	   string curFile1 = decalFile;
	   string curFile2 = bumpFile;

	   for(int i = 0; i < textureList.size(); i++) {
			if(textureList[i].decalFilename == curFile1) { retInfo->decIndex = textureList[i].decIndex; decalSkip = true;}
			if(textureList[i].bumpFilename == curFile2) { retInfo->bumpIndex = textureList[i].bumpIndex; bumpSkip = true;}
		}
	}
	


	if(decalFile != NULL && decalSkip == false) { //load in decal texture
		cout << "Loading Decal Texture: " << decalFile << endl;
		glGenTextures(1, (unsigned int *)(&(retInfo->decIndex)));
		glBindTexture(GL_TEXTURE_2D, retInfo->decIndex);
		loadTextureDecalImage(decalFile, mipmaps);
	}

	if(bumpFile != NULL && bumpSkip == false) { //load in bump texture
		cout << "Loading Bump Texture: " << bumpFile << endl;
		glGenTextures(1, (unsigned int *)(&(retInfo->bumpIndex)));
		glBindTexture(GL_TEXTURE_2D, retInfo->bumpIndex);
		loadTextureNormalMap(bumpFile, scale);
	}

	
	retInfo->decalFilename = decalFile;
	retInfo->bumpFilename = bumpFile;
	retInfo->scale = scale;
	
  	textureList.push_back(*retInfo);
	return retInfo;
}

TextureInfo *TInfrastructure::LoadWidgetTexture(char *decalFile) {
	TextureInfo *retInfo = new TextureInfo;
	bool decalSkip = false;
	
	retInfo->bumpIndex = UNDEFINED;
	retInfo->decIndex = UNDEFINED;
	
	for(int i = 0; i < widgetTextureList.size(); i++) {
		if(widgetTextureList[i].decalFilename == decalFile) {
			retInfo->decIndex = widgetTextureList[i].decIndex;
			decalSkip = true;
		}
	}
	
	if(!decalSkip) {
		cout << "Loading Widget Texture: " << decalFile << endl;
		glGenTextures(1, (unsigned int *)(&(retInfo->decIndex)));
		glBindTexture(GL_TEXTURE_2D, retInfo->decIndex);
		loadTextureDecalImage(decalFile, false);
	}
	
	retInfo->decalFilename = decalFile;
	
	widgetTextureList.push_back(*retInfo);
	return retInfo;
}


unsigned char *TInfrastructure::readShaderFile(const char* filename, int *fileSize) {
	ifstream shaderFile(filename, ios::in | ios::binary);
//	ifstream shaderFile(filename);
	if(shaderFile.fail())
	{
		cerr << "Unable to open " << filename << endl;
		return NULL;
	}

	shaderFile.seekg(0, ios::end);
	int shaderSize = shaderFile.tellg();
	shaderFile.seekg(0, ios::beg);
	
	unsigned char * buffer = new unsigned char[shaderSize];
	shaderFile.read((char *)buffer, shaderSize);
	shaderFile.close();
	
	*fileSize = shaderSize;
	return buffer;
}

int TInfrastructure::LoadVertexProgramNV(char *filename) {
	GLuint newShader = 0;
	int errorVal = 0, fileSize = 0;
	char *curShaderString = readShaderFile(filename, &fileSize);
	
	glGenProgramsNV(1, &newShader);
	glBindProgramNV(GL_VERTEX_PROGRAM_NV, newShader);
	//load vertex program
	glLoadProgramNV(GL_VERTEX_PROGRAM_NV, newShader, fileSize, curShaderString);
	glGetIntegerv(GL_PROGRAM_ERROR_POSITION_NV, &errorVal);
	if(errorVal!=-1)
	{
		cerr << "Cannot load " << filename << " Vertex Shader due to error at: " << errorVal << endl;
		return -1;
	}
	cout << "Loaded " << filename << " Vertex Shader successfully." << endl;
	vertexProgramList.push_back(newShader);

	delete [] curShaderString;
	return newShader;
}


void TInfrastructure::PrepareStencilShadowing() {

	glDisable(GL_REGISTER_COMBINERS_NV);
	glDepthMask(GL_FALSE);
	glDepthFunc(GL_LESS);

	glEnable(GL_STENCIL_TEST);
	glColorMask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE);
	glStencilFunc(GL_ALWAYS, 0, ~0);

}

void TInfrastructure::PrepareShadowFinalPass() {
	glStencilFunc(GL_EQUAL, 0, ~0);
	glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
	glDepthFunc(GL_EQUAL);
	glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
	glShadeModel(GL_FLAT);
}

void TInfrastructure::EndStencilShadowing() {
	glDepthFunc(GL_LEQUAL);
	glDepthMask(GL_TRUE);
	glDisable(GL_STENCIL_TEST);
	glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
	glShadeModel(GL_SMOOTH);
	glFrontFace(GL_CW);
	glShadeModel(GL_SMOOTH);
}

/* draw a shadowing rectangle covering the entire screen */
void TInfrastructure::RenderShadowRect(GLfloat r, GLfloat g, GLfloat b, GLfloat alpha) {
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	gluPerspective(45.0f,(GLfloat)width/(GLfloat)height,0.1f,100.0f);
	glMatrixMode(GL_MODELVIEW);
	
   //default values can be: r = 0.0, g = 0.0, b = 0.0, alpha = 0.4
	glFrontFace(GL_CCW);
	glColorMask(1, 1, 1, 1);

	glDisable(GL_DEPTH_TEST);
	glColor4f(r, g, b, alpha);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glStencilFunc(GL_NOTEQUAL, 0, 0xffffffff);
	glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
	glPushMatrix();
	glLoadIdentity();

	glBegin(GL_TRIANGLE_STRIP);
		glVertex3f(-0.1f, 0.1f,-0.10f);
		glVertex3f(-0.1f,-0.1f,-0.10f);
		glVertex3f( 0.1f, 0.1f,-0.10f);
		glVertex3f( 0.1f,-0.1f,-0.10f);
	glEnd();

	glPopMatrix();
	glDisable(GL_BLEND);
	glEnable(GL_DEPTH_TEST);

	glMatrixMode(GL_PROJECTION);
	glPopMatrix();	
	glMatrixMode(GL_MODELVIEW);
}

void TInfrastructure::iniProjectionMatrix() {
	GLfloat Near = 1.0f;
	GLfloat fieldOfView = 38.0f; // should be RADIANS! but this number works
	GLfloat aspectRatio = (GLfloat) width / (GLfloat) height; 
	GLfloat Pinf[4][4];
	Pinf[1][0] = Pinf[2][0] = Pinf[3][0] = Pinf[0][1] =
	Pinf[2][1] = Pinf[3][1] = Pinf[0][2] = Pinf[1][2] =
	Pinf[0][3] = Pinf[1][3] = Pinf[3][3] = 0.0f;

	Pinf[0][0] = atan(fieldOfView) / aspectRatio;

	Pinf[1][1] = atan(fieldOfView);
	Pinf[3][2] = -2 * Near; 
	Pinf[2][2] = Pinf[2][3] = -1.0f;

	glMatrixMode(GL_PROJECTION);
	glMultMatrixf(&Pinf[0][0]);
}

void TInfrastructure::iniCustomProjectionMatrix(float fov, int curWidth, int curHeight) {
	GLfloat Near = 1.0f;
	GLfloat fieldOfView = fov * MATH_PI / 180.0;
	GLfloat aspectRatio = (GLfloat) curWidth / (GLfloat) curHeight; 
	GLfloat Pinf[4][4];
	Pinf[1][0] = Pinf[2][0] = Pinf[3][0] = Pinf[0][1] =
	Pinf[2][1] = Pinf[3][1] = Pinf[0][2] = Pinf[1][2] =
	Pinf[0][3] = Pinf[1][3] = Pinf[3][3] = 0.0f;

	Pinf[0][0] = atan(fieldOfView) / aspectRatio;

	Pinf[1][1] = atan(fieldOfView);
	Pinf[3][2] = -2 * Near; 
	Pinf[2][2] = Pinf[2][3] = -1.0f;

	glMatrixMode(GL_PROJECTION);
	glMultMatrixf(&Pinf[0][0]);
}


/* Reading texture-mapped fonts. Based on Nehe's Tutorial at http://nehe.gamedev.net This is the only(most efficient) way to read their file format*/
void TInfrastructure::BuildFont() {
	float cx = 0.0f;  
	float cy = 0.0f;  
	float ratio = 1.0f / 16.0;

	textRef  = glGenLists(NUM_ASCII);
	glBindTexture(GL_TEXTURE_2D, textTexture);

	for (int i = 0; i < NUM_ASCII; i++ ) {
		cx = 1.0 - ( float )( i % FONT_WIDTH ) * ratio;
		cy = 1.0 - ( float )( i / FONT_WIDTH) * ratio;
		glNewList(textRef + ( NUM_ASCII - 1 - i ), GL_COMPILE);

		glBegin(GL_QUADS);
			glTexCoord2f( cx - ratio, cy );
			glVertex2i(0, 0);
			glTexCoord2f(cx, cy);
			glVertex2i(FONT_WIDTH, 0);
			glTexCoord2f(cx, cy - ratio);
			glVertex2i(FONT_WIDTH, FONT_HEIGHT);
			glTexCoord2f(cx - ratio, cy - ratio);
			glVertex2i(0, FONT_HEIGHT);
		glEnd( );
		glTranslated(FONT_OFFSET_X, 0, 0);

		glEndList( );
	}

}

void TInfrastructure::screenPrint(int x, int y, char *curString, int fontSet, GLfloat scaleX, GLfloat scaleY) {
	glActiveTextureARB(GL_TEXTURE1_ARB );
	glDisable(GL_TEXTURE_CUBE_MAP_ARB );
	glDisable(GL_TEXTURE_2D);
	glDisable(GL_CULL_FACE );
	glDisable(GL_REGISTER_COMBINERS_NV);

	
	glActiveTextureARB( GL_TEXTURE0_ARB );
	glDisable( GL_TEXTURE_CUBE_MAP_ARB );
	glDisable(GL_CULL_FACE );
	glDisable(GL_REGISTER_COMBINERS_NV);
	glEnable( GL_TEXTURE_2D );

	glBindTexture( GL_TEXTURE_2D, textTexture);
	glDisable( GL_DEPTH_TEST );
	glMatrixMode( GL_PROJECTION );
		glPushMatrix( );
		glLoadIdentity( );
		glOrtho(0, width, 0, height, -1, 1);

	glMatrixMode( GL_MODELVIEW );
		glPushMatrix( );
		glLoadIdentity( );
		glTranslated( x, y, 0 );
		glScalef(scaleX, scaleY, 1.0);

	//choose the correct font set to render and render the string
	glListBase(textRef - 32 + (128 * fontSet));
	glCallLists(strlen(curString), GL_BYTE, curString);

	glMatrixMode( GL_PROJECTION );
	glPopMatrix( );
	glMatrixMode( GL_MODELVIEW );
	glPopMatrix( );
	glEnable( GL_DEPTH_TEST );
	glEnable(GL_CULL_FACE );
	
}

int TInfrastructure::loadSingleSkyTexture(char *filename) {
	int newTexIndex = 0;
	gliGenericImage *image2D;
	glGenTextures(1, (unsigned int *)(&newTexIndex) );
	glBindTexture(GL_TEXTURE_2D, newTexIndex);
	image2D = readImage(filename);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_COMPRESSED_RGB_S3TC_DXT1_EXT,
				image2D->width, image2D->height, 0,
				image2D->format, GL_UNSIGNED_BYTE, image2D->pixels);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	
	delete image2D;
	
	return newTexIndex;
}


void TInfrastructure::loadSkyBox(char *leftTexture, char *rightTexture, char *frontTexture, char *backTexture,
			char *upTexture, char *downTexture) {
	for(int i = 0; i < 6; i++) {
		if(skyboxTexture[i] != 0) {
			glDeleteTextures(1, (unsigned int *)(&(skyboxTexture[i])) );
		}
	}
	
	 skyboxTexture[SKYBOX_LEFT] = loadSingleSkyTexture(leftTexture);
	 skyboxTexture[SKYBOX_RIGHT] = loadSingleSkyTexture(rightTexture);
	 skyboxTexture[SKYBOX_FRONT] = loadSingleSkyTexture(frontTexture);
	 skyboxTexture[SKYBOX_BACK] = loadSingleSkyTexture(backTexture);
	 skyboxTexture[SKYBOX_UP] = loadSingleSkyTexture(upTexture);
	 skyboxTexture[SKYBOX_DOWN] = loadSingleSkyTexture(downTexture);
}

BufferInfo TInfrastructure::CreateBuffer(int numVertices) {
	if(extgl_Extensions.ARB_vertex_program) {
		BufferInfo newBuffer;
		newBuffer.verticeNum = numVertices;
		glGenBuffersARB(numVertices, &newBuffer.index);
		
		bufferList.push_back(newBuffer);
		return newBuffer;
	}
}

void TInfrastructure::createWindowTextures(int num, int width, int height) {
	glEnable(GL_TEXTURE_2D);
	for(int i = 0; i < num; i++) {
		int newHandle = 0;
		unsigned int *pTexture = new unsigned int [WINDOW_TEXTURE_SIZE * WINDOW_TEXTURE_SIZE * 3];
		memset(pTexture, 0, WINDOW_TEXTURE_SIZE * WINDOW_TEXTURE_SIZE * 3 * sizeof(unsigned int));
		glGenTextures(1, (unsigned int *)&newHandle);
		glBindTexture(GL_TEXTURE_2D, newHandle);
		glTexImage2D(GL_TEXTURE_2D, 0, 3,
			WINDOW_TEXTURE_SIZE, WINDOW_TEXTURE_SIZE, 0, GL_RGB, GL_UNSIGNED_INT, pTexture);
		// Set the texture quality^M
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
		windowTexture.push_back(newHandle);
		
		delete [] pTexture;
	}
	
	glDisable(GL_TEXTURE_2D);
}

void TInfrastructure::createCubeMapTextures() {
	//create 4 sets of Cube Maps of sizes: 32x32, 64x64,128x128 and 256x256
	glGenTextures(1, &CubeMap32Ref);
	createCubeMapTextureSet(32, CubeMap32Ref);

	glGenTextures(1, &CubeMap64Ref);
	createCubeMapTextureSet(64, CubeMap64Ref);

	glGenTextures(1, &CubeMap128Ref);
	createCubeMapTextureSet(128, CubeMap128Ref);

	glGenTextures(1, &CubeMap256Ref);
	createCubeMapTextureSet(256, CubeMap256Ref);
}

void TInfrastructure::createCubeMapTextureSet(int size, int ID) {
	unsigned int *pTexture = new unsigned int [size * size * 3];
	memset(pTexture, 0, size * size * 3 * sizeof(unsigned int));
	
	glBindTexture(GL_TEXTURE_CUBE_MAP_ARB, ID);
	glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_X_ARB,
					0, 3, size, size, 0,
					GL_RGB, GL_UNSIGNED_INT, pTexture);
	
	glTexImage2D(GL_TEXTURE_CUBE_MAP_NEGATIVE_X_ARB,
					0, 3, size, size, 0,
					GL_RGB, GL_UNSIGNED_INT, pTexture);

	glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_Y_ARB,
					0, 3, size, size, 0,
					GL_RGB, GL_UNSIGNED_INT, pTexture);
	
	glTexImage2D(GL_TEXTURE_CUBE_MAP_NEGATIVE_Y_ARB,
					0, 3, size, size, 0,
					GL_RGB, GL_UNSIGNED_INT, pTexture);

	glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_Z_ARB,
					0, 3, size, size, 0,
					GL_RGB, GL_UNSIGNED_INT, pTexture);
	
	glTexImage2D(GL_TEXTURE_CUBE_MAP_NEGATIVE_Z_ARB,
					0, 3, size, size, 0,
					GL_RGB, GL_UNSIGNED_INT, pTexture);

	glTexParameteri(GL_TEXTURE_CUBE_MAP_ARB, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_CUBE_MAP_ARB, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_CUBE_MAP_ARB, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_CUBE_MAP_ARB, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_CUBE_MAP_ARB, GL_TEXTURE_WRAP_R, GL_CLAMP_TO_EDGE);
	
	delete [] pTexture;
}


GLuint TInfrastructure::getWindowTexture(int num) {
	return windowTexture[num];
}

void TInfrastructure::copyScreenToBuffer(int index) {
	glActiveTextureARB(GL_TEXTURE1_ARB );
	glDisable(GL_TEXTURE_CUBE_MAP_ARB );
	glDisable(GL_TEXTURE_2D);

	glActiveTextureARB( GL_TEXTURE0_ARB );
	glDisable( GL_TEXTURE_CUBE_MAP_ARB );
	glDisable(GL_REGISTER_COMBINERS_NV);
	
	glEnable( GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, windowTexture[index]);
	glCopyTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, 0, 0, WINDOW_TEXTURE_SIZE, WINDOW_TEXTURE_SIZE, 0);
	glDisable(GL_TEXTURE_2D);
}

void TInfrastructure::copyScreenToCubeMap(GLenum target) {
	glActiveTextureARB(GL_TEXTURE1_ARB );
	glDisable(GL_TEXTURE_CUBE_MAP_ARB );
	glDisable(GL_TEXTURE_2D);

	glActiveTextureARB( GL_TEXTURE0_ARB );
	glDisable( GL_TEXTURE_CUBE_MAP_ARB );
	glDisable(GL_REGISTER_COMBINERS_NV);
	glDisable(GL_TEXTURE_2D);
	
	glEnable( GL_TEXTURE_CUBE_MAP_ARB);
	int size = 0;
	switch(cubeMapSelect) {
		case 0:
			glBindTexture(GL_TEXTURE_CUBE_MAP_ARB, CubeMap32Ref);
			size = 32;
		break;
		
		case 1:
			glBindTexture(GL_TEXTURE_CUBE_MAP_ARB, CubeMap64Ref);
			size = 64;
		break;
		
		case 2:
			glBindTexture(GL_TEXTURE_CUBE_MAP_ARB, CubeMap128Ref);
			size = 128;
		break;
		
		case 3:
			glBindTexture(GL_TEXTURE_CUBE_MAP_ARB, CubeMap256Ref);
			size = 256;
		break;
	}
	
	
	
	glCopyTexImage2D(target, 0, GL_RGB, 0, 0, size, size, 0);
	glDisable( GL_TEXTURE_CUBE_MAP_ARB);
}

int TInfrastructure::GetCurrentCubeMap() {
	int retval = 0;
	switch(cubeMapSelect) {
		case 0:
			retval = CubeMap32Ref;
		break;
		
		case 1:
			retval = CubeMap64Ref;
		break;
		
		case 2:
			retval = CubeMap128Ref;
		break;
		
		case 3:
			retval = CubeMap256Ref;
		break;
	}
	
	return retval;
}

void TInfrastructure::timerDelay(int delay) {
	int curTimeVal = SDL_GetTicks();
	int timeDiff = curTimeVal - timerVal;
	
	if(timeDiff < delay) {
		while(timeDiff < delay) {
			curTimeVal = SDL_GetTicks();
			timeDiff = curTimeVal - timerVal;
		}
	}
	timerVal = SDL_GetTicks();
}

bool TInfrastructure::timerDelay2(int delay) {
	int curTimeVal = SDL_GetTicks();
	int timeDiff = curTimeVal - timerVal2;
	
	if(timeDiff < delay) return false;
	timerVal2 = SDL_GetTicks();
	return true;
}

void TInfrastructure::LoadConfigKeys(char *filename) {
	FILE *file;
	char buf[256], *gotbuf;
	char name[256], texName[256];
	int status, type, matches = 0;
	char moveFront, moveBack, moveRight, moveLeft, moveJump, renderLevel;
	float moveRot;
	
	file = fopen(filename, "r");
	if (file == NULL) {
		cout << "Could not load config file: " << filename << "." << endl;
		return -1;
 	 }
	 
	 	while (!feof(file)) {
			gotbuf = is_fgets(buf, sizeof(buf), file);
			if (gotbuf) {
				switch(buf[0]) {
					case '#':
					break;
					
					case 'C':
						/* CONFIG moveFront moveBack moveLeft moveRight moveJump movRot renderLevel*/
						matches = sscanf(buf, "CONFIG %g %d\n", &moveRot, &renderLevel);
						if(matches == 2) {
							configKeys.moveFront = SDLK_w;
							configKeys.moveBack = SDLK_s;
							configKeys.moveLeft = SDLK_a;
							configKeys.moveRight = SDLK_d;
							configKeys.moveJump = SDLK_SPACE;
							configKeys.moveXVel = HUMAN_MOVE_X;
							configKeys.moveYVel = HUMAN_MOVE_Y;
							configKeys.moveJVel = HUMAN_MOVE_J;
							configKeys.moveXrot = moveRot;
							configKeys.moveYrot = moveRot;
							configKeys.renderLevel = renderLevel;
						} else {fclose(file); cout << "Bad Parse for CONFIG!" << endl; return -1;}
					break;
				}
			}
		}
		
		return true;

}


void TInfrastructure::SaveConfigKeys(char *filename) {
	FILE *file;
	file = fopen(filename, "w");

	if (file == NULL) {
		cout << "Could not save " << filename << "." << endl;
  }
	fprintf(file, "#Config file for game\n");
	fprintf(file, "CONFIG %g %d\n", configKeys.moveXrot, configKeys.renderLevel);
	
}

int TInfrastructure::LoadSound(char *filename) {
	Mix_Chunk *newSound = Mix_LoadWAV(filename);
	soundList.push_back(newSound);
	cout << "Loading Sound: " << filename << endl;
	
	return soundList.size() - 1;
}

void TInfrastructure::PlaySound(int index) {
	if(index < soundList.size())
		Mix_PlayChannel(-1,soundList[index],0);
	else
		cout << "Invalid sound index!" << endl;
}

int TInfrastructure::LoadMusic(char *filename) {
	Mix_Music *newMusic = Mix_LoadMUS(filename);
	musicList.push_back(newMusic);
	cout << "Loading Music: " << filename << endl;
	
	return musicList.size() - 1;
}

void TInfrastructure::PlayMusic(int index) {
	Mix_PlayMusic(musicList[index], 0);
}


int TInfrastructure::LoadSoundFileList(char *filename) {
	FILE *file;
	char buf[256], *gotbuf;
	char name[256], texName[256];
	int status, type, matches = 0;
	
	file = fopen(filename, "r");
	if (file == NULL) {
		cout << "Could not load config file: " << filename << "." << endl;
		return -1;
 	 }
	 
	 	while (!feof(file)) {
			gotbuf = is_fgets(buf, sizeof(buf), file);
			if (gotbuf) {
				switch(buf[0]) {
					case '#':
					break;
					
					case 'S':
						/* SOUND filename*/
						matches = sscanf(buf, "SOUND %s\n", &name);
						if(matches == 1) {
							LoadSound(name);
						} else {fclose(file); cout << "Bad Parse for SOUND!" << endl; return -1;}
					break;
				
					case 'M':
						/* MUSIC filename*/
						matches = sscanf(buf, "MUSIC %s\n", &name);
						if(matches == 1) {
							LoadMusic(name);
						} else {fclose(file); cout << "Bad Parse for MUSIC!" << endl; return -1;}
					break;
				}
			}
		}
		
		return true;
	
}


