/***************************************************************************
                          tnetwork.h  -  description
                             -------------------
    begin                : Tue Nov 11 2003
    copyright            : (C) 2003 by jychong
    email                : jychong@dromaesaur.stanford.edu
 ***************************************************************************/


#ifndef TNETWORK_H
#define TNETWORK_H
#include "nl.h"
#include <stdint.h>
#include <vector>
#include <iostream>
#include "vectorutils.h"
#include <SDL/SDL.h>
using namespace std;


/**
  *@author jychong
  */

typedef void (*SDL_CallBackFunc)(void *);
typedef void (*SDL_CallBackFunc2)(void *, void *, int);

const int READ_BUFFER_SIZE = 1024;

const int IDLE_MODE = 0;
const int ALLOW_MODE = 1;
const int TRY_CONNECT_MODE = 2;
const int RUNNING_MODE = 3;

const int SERVER_STAT = 100;
const int CLIENT_STAT = 101;

const int CLIENT_CHANNEL = 5;
const int SERVER_CHANNEL = 6;

const int CLIENT_TARGET = 1;
const int LOCAL_TARGET = 2;
const int SERVER_TARGET = 3;

/*
Description of network data packet:
Header:
------------------
uint8_t - type

Data:
------------------
uint8_t[some size] - data

maximum size is 1024 bytes
*/

/* Typical object data
uint8_t - object type
uint8_t - team
uint8_t - selected
int16_t - health
uint8_t - weapon
uint8_t - ammo
uint8_t - action
uint32_t - current frame
NLfloat - x
NLfloat - y
NLfloat - z
int16_t rotate_y
uint16_t index

Total size for data: 3 * 1 + 2 + 3 * 1 + 4 * 4 + 2 = 3 + 2 + 3 + 16 + 2 * 2 = 28 bytes
*/

struct TNetworkObjectPacket {
	int type;
	int team;
	int selected;
	int health;
	int weapon;
	int ammo;
	int action;
	int frame;
	Vector translate;
	float rotateY;
	int index;
};

/* Typical object list description data
uint16_t - total number of objects
uint32_t - current frame
uint8_t - game situation
uint16_t resources

Total size for data:  2 + 4 + 1 + 2 = 9 bytes
*/

struct TNetworkObjectListPacket {
	int num;
	unsigned int frame;
	int situation;
	int resources;
};

/* Typical player input packet
uint8_t - '1' press
uint8_t - '2' press
uint8_t - '3' press
uint8_t - 'w' press
uint8_t - 's' press
uint8_t - 'a' press
uint8_t - 'd' press
uint8_t - spacebar press
uint16_t - mouse X
uint16_t - mouse Y
uint8_t - mouse left press
uint8_t - mouse right press
uint8_t - mode
NLfloat - playerX
NLfloat - playerY
NLfloat - playerZ
NLfloat - playerRotateY
NLfloat - playerRotateX

Total size for data:  11 * 1 + 2 * 2  + 4 * 5 = 11 + 4 + 20 = 35 bytes
*/

struct TNetworkPlayerInputPacket {
	char press_1;
	char press_2;
	char press_3;
	char press_w;
	char press_s;
	char press_a;
	char press_d;
	char press_space;
	int mouseX;
	int mouseY;
	char mouseLeftPress;
	char mouseRightPress;
	char mode;
	Vector playerLoc;
	float playerRotateY;
	float playerRotateX;
};

/* Typical particle packet
uint8_t type
int16_t index
NLfloat x
NLfloat y
NLfloat z
NLfloat velX
NLfloat velY
NLfloat velZ

Total size for data: 1 + 2 + 4 * 6 = 1 + 2 + 24 = 27
*/

struct TNeworkParticlePacket {
	uint8_t type;
	float x;
	float y;
	float z;
	float velX;
	float velY;
	float velZ;
	int16_t index;
};

const uint8_t TEXT_MESSAGE_PACKET = 1;
const uint8_t OBJECT_PACKET = 2;
const uint8_t OBJECT_LIST_PACKET = 3;
const uint8_t PLAYER_INPUT_PACKET = 4;
const uint8_t PARTICLE_PACKET = 5;
const uint8_t START_GAME_PACKET = 100;

class TNetwork {
	protected:
		int localPort, localPort2;
		NLsocket localSock, serverSock;
		NLenum socktype;
		int mode;
		char readBuffer[READ_BUFFER_SIZE];
		void *callBackData;
		NLaddress joinAddress, joinAddress2;
		SDL_CallBackFunc connectCallBack;
		SDL_CallBackFunc2 connectCallBack2;
		NLint connectGroup;
		vector<NLsocket> clients;
		int sendTarget;

	public:
		TNetwork();
		~TNetwork();

		/* network connection/disconnection/parameter adjustment functions*/
		void setLocalPost(int port) { localPort = port; }
		int getMode() { return mode; }
		void allowConnect(SDL_CallBackFunc callFn, void *data);
		void endConnect();
		int joinServer(char *name, SDL_CallBackFunc callFn, void *data);
		void processLogin();
		void processConnect();
		void startNetwork(SDL_CallBackFunc2 callFn, void *data);
		void stopNetwork() { mode = IDLE_MODE; SDL_Delay(100);}
		void processMessages();
		void setNetworkParam(SDL_CallBackFunc2 callFn, void *data) { connectCallBack2 = callFn; callBackData = data; }
		void disConnect();
		int getNumClients() { return clients.size(); }

		/* network data sending functions*/
		void setSendTarget(int target) {sendTarget = target; }
		void sendTextMessageFromServer(char *message);
		void sendTextMessageFromClient(char *message);
		void sendObjectPacket(int objType, int team, int selected, int health, int weapon, int ammo,
					int action, unsigned int frame, Vector translate, Vector rotate, int index);
		void sendStartPacket();
		void sendObjectList(int objNum, unsigned int frame, char situation, int resources);
		void sendPlayerInputPacket(char press_1, char press_2, char press_3, char press_w, char press_s,
					char press_a, char press_d, char press_space, int mouseX, int mouseY,
					char mouseLeft, char mouseRight, char mode, Vector playerLoc, Vector playerRotate);
		void sendParticlePacket(char type, int index, Vector loc, Vector vel);
		
		/* data packet parsing functions*/
		int getPacketType(void *data);
		void getString(void *data, int size, char *output);
		void getObjectPacket(void *data, TNetworkObjectPacket *curPacket);
		void getObjectListPacket(void *data, TNetworkObjectListPacket *curPacket);
		void getPlayerInputPacket(void *data, TNetworkPlayerInputPacket *curPacket);
		void getParticlePacket(void *data, TNeworkParticlePacket *curPacket);
};

#endif
