/***************************************************************************
                          tscene.cpp  -  description
                             -------------------
    begin                : Tue Jun 24 2003
    copyright            : (C) 2003 by Chong Jiayi
    email                : jychong@stanford.edu
 ***************************************************************************/


#include "tscene.h"

static int is_Space(int c)
{
	if (c == ' ' || c == '\t' || c == '\n' || c == '\r') return true;
	else return false;
}

static char * is_fgets(char *buf, int n, FILE *fp)
{
	char *wbuf;
	int c, bcnt = 0;

	if (n <= 0) return (NULL);

	wbuf = buf; n = n - 1;                   

	c = '\t';
	while(is_Space(c)) { c = getc(fp); }

	while (n != 0) {
		if (c == '\n' || c == '\r') break;
		if (c == EOF) 
			if (bcnt == 0) return NULL; else break;
		buf[bcnt] = c; bcnt++;
		n = n - 1;
		c = getc(fp);
	}

	buf[bcnt] = '\0';
	return buf;
}


TScene::TScene(TInfrastructure *theInfrastructure){
	Vector zeroVec, defaultVelCap, defaultAngleVelCap;
	zeroVec.x = zeroVec.y = zeroVec.z = 0.0f;
	defaultVelCap.x = defaultVelCap.y = defaultVelCap.z = 4.0f;
	defaultAngleVelCap.x = defaultAngleVelCap.y = defaultAngleVelCap.z = 2.0f;
	
	curInfrastructure = theInfrastructure;
	viewerLoc.x = viewerLoc.y = viewerLoc.z = 0.0f;
	renderLevel = RENDER_LEVEL2;
	SetViewerBounds(VIEWER_WIDTH_X, VIEWER_WIDTH_Y, VIEWER_WIDTH_Z);
	viewerPhysics.loc = viewerLoc;
	viewerPhysics.vel = zeroVec;
	viewerPhysics.accel = zeroVec;
	viewerPhysics.velCap = defaultVelCap;
	
	viewerPhysics.angle = zeroVec;
	viewerPhysics.angleVel = zeroVec;
	viewerPhysics.angleAccel = zeroVec;
	viewerPhysics.angleVelCap = defaultAngleVelCap;
	
	renderFrameCnt = 0;
	frameCnt = 0;
	blurSwitch = false;
	renderBlurFrame = false;
	portalScissor = false;
	cubeMapSwitch = false;
	dummyObj = new TMd2(theInfrastructure);
	
	SetItemsSpawnY(-30.0);
	
	//initialize jitter offset
	for(int i = 0; i < NUM_JITTER_SAMPLES; i++) {
		float ranf = (float)rand() / (float) RAND_MAX * 16.0 - 8.0;
		float ranf2 = (float)rand() / (float) RAND_MAX * 18.0 - 8.0;
		jitterOffset[i].x = ranf;
		jitterOffset[i].y = ranf2;
	}
	
	callBackFunc = NULL;
	callBackObj = NULL;
	nlMutexInit(&renderMutex);
	
	pillarCnt = 0;
	
}

TScene::~TScene(){
	ObjectList.clear();
	for(int i = 0; i < LoadedTObjects.size(); i++) {
		TObject *curObject = LoadedTObjects[i];
		if(curObject != NULL) {
			if(curObject->GetType() == MD2_TYPE) {
				TMd2 *curMd2 = (TMd2 *)curObject;
				delete curMd2;
			}
			else {
				TPlane *curPlane = (TPlane *)curObject;
//				delete curPlane;
			}
		}
		else { cout << "Loaded TObject is NULL!" << endl; }
	}
}

void TScene::AttachObject(TObject *curObject, Vector translate, Vector scale, Vector rotate, int shadow = true, bool specular = true, 
	bool diffuse = true, bool attenuate = false, int sectorIndex = -1, bool reflect = false, float alphaval = 0.0f, int gameType = -1,
	Vector *rotOffset = NULL, int team = -1) {
	
	SceneObjInfo newSceneObj;
	Vector zeroVec, defaultVelCap, defaultAngleVelCap;
	zeroVec.x = zeroVec.y = zeroVec.z = 0.0f;
	defaultVelCap.x = defaultVelCap.y = defaultVelCap.z = 2.0f;
	defaultAngleVelCap.x = defaultAngleVelCap.y = defaultAngleVelCap.z = 2.0f;
	
	newSceneObj.curObject = curObject;
	newSceneObj.portalCull = true;
	newSceneObj.visible = true;
	newSceneObj.shadow = shadow;
	newSceneObj.diffuse = diffuse;
	newSceneObj.specular = specular;
	newSceneObj.attenuate = attenuate;
	newSceneObj.reflect = reflect;
	newSceneObj.alpha = alphaval;
	newSceneObj.translate = translate;
	newSceneObj.rotate = rotate;
	newSceneObj.scale = scale;
	if(rotOffset == NULL) newSceneObj.rotOffset = rotate;
	else newSceneObj.rotOffset = *rotOffset;
	
	newSceneObj.physicsPacket.loc = translate;
	newSceneObj.physicsPacket.vel = zeroVec;
	newSceneObj.physicsPacket.accel = zeroVec;
	newSceneObj.physicsPacket.velCap = defaultVelCap;
	
	newSceneObj.physicsPacket.angle = rotate;
	newSceneObj.physicsPacket.angleVel = zeroVec;
	newSceneObj.physicsPacket.angleAccel = zeroVec;
	newSceneObj.physicsPacket.angleVelCap = defaultAngleVelCap;
	
	newSceneObj.gamePacket.type = gameType;
	newSceneObj.gamePacket.team = team;
	newSceneObj.gamePacket.health = 965536;
	newSceneObj.gamePacket.selectLevel = -1;
	
	newSceneObj.loadedTObjIndex = LoadedTObjects.size() - 1;
	
	if(sectorIndex == -1) {newSceneObj.sectorIndex = -1; newSceneObj.sectorLocked = -1; }
	else {newSceneObj.sectorIndex = sectorIndex; newSceneObj.sectorLocked = true;}
	
	ObjectList.push_back(newSceneObj);
}

void TScene::SetLighting(Vector lightVec, Vector eyeVec, float brightlevel, float r, float g, float b) {
	currentLightVec = lightVec;
	currentEyeVec = eyeVec;
	brightness = brightlevel;
	lightRed = r;
	lightGreen = g;
	lightBlue = b;
}

//render all objects in the scene. call any frustrum/portal culling functions before this one for optimal performance
void TScene::RenderAll() {

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);
	
	if(blurSwitch && !renderBlurFrame) { //woohoo!
		renderBlurFrame = true;
		glViewport(0, 0, WINDOW_TEXTURE_SIZE, WINDOW_TEXTURE_SIZE);
		RenderAll();
		glViewport(0, 0, curInfrastructure->GetWinWidth(), curInfrastructure->GetWinHeight());
		renderBlurFrame = false;
		curInfrastructure->copyScreenToBuffer(renderFrameCnt % NUM_WINDOW_TEXTURES);
		renderFrameCnt++;
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);
	}
	
	if(!renderBlurFrame) RenderGlowAll();
	
	curPortalSwitch = false;
	reflectSwitch = false;

	if(portalScissor) glEnable(GL_SCISSOR_TEST);
	//do the objects with chromatic aberration first
	if(!cubeMapSwitch) ChromaticRefractRender();
	
	//first pass, decal render
	DecalRender();

	
	//third pass, diffuse/attenuate render
	if(renderLevel >= RENDER_LEVEL1) AttenuateRender();
	if(renderLevel >= RENDER_LEVEL1) DiffuseRender();

	//fourth pass, specular render
	if(renderLevel >= RENDER_LEVEL1) SpecularRender();

	if(portalScissor) glDisable(GL_SCISSOR_TEST);
	
	//fifth pass, shadow render
	if(renderLevel >= RENDER_LEVEL2) ShadowRender();
	
	curPortalSwitch = true;
	glClear(GL_STENCIL_BUFFER_BIT);

	if(renderLevel >= RENDER_LEVEL1) 	ReflectorsRender();

	//first pass, decal render
	DecalRender();

	glClear(GL_STENCIL_BUFFER_BIT);

	//third pass, diffuse/attenuate render
	if(renderLevel >= RENDER_LEVEL1) 	AttenuateRender();
	if(renderLevel >= RENDER_LEVEL1) 	DiffuseRender();

	//fourth pass, specular render
	if(renderLevel >= RENDER_LEVEL1) 	SpecularRender(); 
	
	//fifth pass, shadow render
	if(renderLevel >= RENDER_LEVEL2) 	ShadowRender();
	
	if(renderBlurFrame) return;
	
	
	if(blurSwitch && !renderBlurFrame) { //woohoo!
		BlurRender();
	}
	
	//throw in a timer to synchronize frame rates on different machines
	curInfrastructure->timerDelay(10);
	frameCnt++;
	RenderGameSpiritEffects();
//	PortalWindowRender();    //for debugging purposes only
}

// render 6 faces of the cube map for reflection/refraction rendering
void TScene::RenderCubeMapFaces(int size, TObject *curObject, int sector, int objIndex) {
	GLfloat curViewerMatrix[16];
	GLfloat curInverseViewerMatrix[16];
	bool orgBlurSwitch = blurSwitch;
	Vector orgViewerLoc = viewerLoc, rotateAxis;
	ObjectPhysics orgPhysics = viewerPhysics;
	//save our current viewer matrices first
	for(int i = 0; i < 16; i++) {curViewerMatrix[i] = viewerMatrix[i]; curInverseViewerMatrix[i] = inverseViewerMatrix[i]; }
	
	cubeMapSwitch = true;
	blurSwitch = false;
	glViewport(0, 0, size, size);
	
	glMatrixMode(GL_PROJECTION);
		glPushMatrix();
		glLoadIdentity();
		curInfrastructure->iniCustomProjectionMatrix(90.0, 100, 100);
	glMatrixMode(GL_MODELVIEW);
	
	//get the midpoint of the object
	RestoreSceneObjectsTransform();
	ApplyMd2PhysicsTransform();
	Vector midpnt;
	Vadd(&(curObject->GetMaxBounds()), &(curObject->GetMinBounds()), &midpnt);
	Vscale(&midpnt, 0.5);
		
	//do left face
	rotateAxis.x = 0.0; rotateAxis.y = 90.0; rotateAxis.z = 180.0;
	viewerPhysics.angle.x = 0.0; viewerPhysics.angle.y = 90.0; viewerPhysics.angle.z = 180.0;
	RenderFaceScene(GL_TEXTURE_CUBE_MAP_NEGATIVE_X_ARB, 90.0, &midpnt, &rotateAxis, sector, objIndex, true);
	//do right face
	rotateAxis.x = 0.0; rotateAxis.y = -90.0; rotateAxis.z = 180.0;
	viewerPhysics.angle.x = 0.0; viewerPhysics.angle.y = -90.0; viewerPhysics.angle.z = 180.0;
	RenderFaceScene(GL_TEXTURE_CUBE_MAP_POSITIVE_X_ARB, -90.0, &midpnt, &rotateAxis, sector, objIndex, true);
	//do front face
	rotateAxis.x = 0.0; rotateAxis.y = 0.0; rotateAxis.z = 180.0;
	viewerPhysics.angle.x = 0.0; viewerPhysics.angle.y = 0.0; viewerPhysics.angle.z = 180.0;
	RenderFaceScene(GL_TEXTURE_CUBE_MAP_NEGATIVE_Z_ARB, 0.0, &midpnt, &rotateAxis, sector, objIndex, false);
	//do back face
	rotateAxis.x = 0.0; rotateAxis.y = 180.0; rotateAxis.z = -180.0;
	viewerPhysics.angle.x = 0.0; viewerPhysics.angle.y = 180.0; viewerPhysics.angle.z = -180.0;
	RenderFaceScene(GL_TEXTURE_CUBE_MAP_POSITIVE_Z_ARB, 180.0, &midpnt, &rotateAxis, sector, objIndex, true);
	//do top face
	rotateAxis.x = -90.0; rotateAxis.y = 0.0; rotateAxis.z = 0.0;
	viewerPhysics.angle.x = -90.0; viewerPhysics.angle.y = 0.0; viewerPhysics.angle.z = 0.0;
	RenderFaceScene(GL_TEXTURE_CUBE_MAP_NEGATIVE_Y_ARB, 90.0, &midpnt, &rotateAxis, sector, objIndex, true);
	//do bottom face
	rotateAxis.x = 90.0; rotateAxis.y = 0.0; rotateAxis.z = 0.0;
	viewerPhysics.angle.x = 90.0; viewerPhysics.angle.y = 0.0; viewerPhysics.angle.z = 0.0;
	RenderFaceScene(GL_TEXTURE_CUBE_MAP_POSITIVE_Y_ARB, -90.0, &midpnt, &rotateAxis, sector, objIndex, true);
	
	//now restore the viewport scene
	glViewport(0, 0, curInfrastructure->GetWinWidth(), curInfrastructure->GetWinHeight());
	//restore our current viewer matrices 
	for(int i = 0; i < 16; i++) {viewerMatrix[i] = curViewerMatrix[i]; inverseViewerMatrix[i] = curInverseViewerMatrix[i]; }
	
	RestoreSceneObjectsTransform();
	ApplyMd2PhysicsTransform();
	cubeMapSwitch = false;
	viewerLoc = orgViewerLoc;
	viewerPhysics = orgPhysics;
	blurSwitch = orgBlurSwitch;
	ApplyViewerTransform();
	
}

void TScene::RenderFaceScene(GLenum target, float angle,Vector *midpnt, Vector *axis, int sector, int objIndex, bool flipHoriz) {
//	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);
	
	RestoreSceneObjectsTransform();
	ApplyMd2PhysicsTransform();
//	RecalcSectorPos();
	
	GLfloat mat[16][16];
	glMatrixMode(GL_MODELVIEW);
		glPushMatrix();
		glLoadIdentity();
		if(flipHoriz) glScalef(-1.0, 1.0, 1.0);
		else glScalef(1.0, 1.0, 1.0);
		glGetFloatv(GL_MODELVIEW_MATRIX, &mat[0][0]);
	glPopMatrix();
	
	PrepareViewerTransform();
		glLoadIdentity();
		glTranslatef(midpnt->x, midpnt->y, midpnt->z);
		if(fabs(axis->x) > 0) glRotatef(-axis->x, 1.0, 0.0, 0.0);
		else glRotatef(0.0, 1.0, 0.0, 0.0);
		if(fabs(axis->y) > 0) glRotatef(-axis->y, 0.0, 1.0, 0.0);
		else glRotatef(0.0, 0.0, 1.0, 0.0);
		if(fabs(axis->z) > 0) glRotatef(-axis->z, 0.0, 0.0, 1.0);
		else glRotatef(0.0, 0.0, 0.0, 1.0);
	EndViewerTransform();
	ApplyViewerTransform();
	viewerLoc = *midpnt;
	PrepareSceneForPortalCulling();
	PortalCull();
	FrustrumCull();
	ObjectList[objIndex].visible = false;


	DecalSectorRender(sector, &mat[0][0]);
	AttenuateSectorRender(sector, &mat[0][0]);
	DiffuseSectorRender(sector, &mat[0][0]);
	ReflectorsRender();  
	if(flipHoriz) RenderSkyBox(-1400.0);
	else RenderSkyBox(1400.0);  

	TObject *curObject = ObjectList[objIndex].curObject;
	curObject->copyScreenToCubeMap(target);
//	glFlush();
//	glFinish();
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);
//	glClear(GL_DEPTH_BUFFER_BIT);

}

void TScene::BlurRender() {
	if(renderFrameCnt >= NUM_WINDOW_TEXTURES) {
		glActiveTextureARB(GL_TEXTURE1_ARB );
		glDisable(GL_TEXTURE_CUBE_MAP_ARB );
		glDisable(GL_TEXTURE_2D);

		glActiveTextureARB( GL_TEXTURE0_ARB );
		glDisable( GL_TEXTURE_CUBE_MAP_ARB );
		glDisable(GL_CULL_FACE );
		glDisable(GL_REGISTER_COMBINERS_NV);
		glEnable( GL_TEXTURE_2D);
		glDisable( GL_DEPTH_TEST );
		
		glMatrixMode( GL_PROJECTION );
			glPushMatrix( );
			glLoadIdentity( );
			glOrtho(0, curInfrastructure->GetWinWidth(), 0, curInfrastructure->GetWinHeight(), -1, 1);
		
		glMatrixMode( GL_MODELVIEW );
			glPushMatrix( );
			glLoadIdentity( );

		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA,GL_ONE);
					
		for(int i = 0; i < NUM_WINDOW_TEXTURES; i++) {
			glBindTexture(GL_TEXTURE_2D, curInfrastructure->getWindowTexture( (renderFrameCnt + i) % NUM_WINDOW_TEXTURES) );
			glColor4f(1.0, 1.0, 1.0, 0.3 - 1.0 / (float)NUM_WINDOW_TEXTURES * (float)((renderFrameCnt + i) % NUM_WINDOW_TEXTURES) * 0.3 );
			glBegin(GL_QUADS);
				glTexCoord2f(0.0, 0.0);
				glVertex2d(0, 0);
				
				glTexCoord2f(1.0, 0.0);
				glVertex2d(curInfrastructure->GetWinWidth() - 1, 0);
				
				glTexCoord2f(1.0, 1.0);
				glVertex2d(curInfrastructure->GetWinWidth() - 1, curInfrastructure->GetWinHeight() - 1);
				
				glTexCoord2f(0.0, 1.0);
				glVertex2d(0, curInfrastructure->GetWinHeight() - 1);
			glEnd();
		}
		
		glMatrixMode( GL_PROJECTION );
		glPopMatrix( );
		glMatrixMode( GL_MODELVIEW );
		glPopMatrix( );
		glEnable( GL_DEPTH_TEST );
		glEnable(GL_CULL_FACE );
		glDisable( GL_TEXTURE_2D );
		glDisable(GL_BLEND);
	}
}


void TScene::DecalRender() {
	for(int i = 0; i < ObjectList.size(); i++) {
		if( (ObjectList[i].visible && ObjectList[i].sectorIndex != curSectorIndex && curPortalSwitch == false)// && ObjectList[i].reflect == reflectSwitch)
			|| (ObjectList[i].visible && ObjectList[i].sectorIndex == curSectorIndex && curPortalSwitch == true && ObjectList[i].reflect == reflectSwitch)
			) { //visible, so render it
			TObject *curObject = ObjectList[i].curObject;
			curObject->EnablePlainDecal();
			curObject->SelectSingleDecalTexture();
			curObject->SetupVertexArraysForRendering(DECAL_MODE);
			
			if(!ObjectList[i].attenuate) {
				if(ObjectList[i].reflect && ObjectList[i].sectorIndex == curSectorIndex) {
					glEnable(GL_BLEND);
					glColor4f(1.0f, 1.0f, 1.0f, ObjectList[i].alpha);  /* set alpha value for reflection blending */
				}
				else {
					glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
				}
			}
			else {
				if(ObjectList[i].reflect && ObjectList[i].sectorIndex == curSectorIndex) {
					glEnable(GL_BLEND);
					glColor4f(1.0f, 1.0f, 1.0f, ObjectList[i].alpha);  /* set alpha value for reflection blending */
				}
				else {
					glColor4f(0.0f, 0.0f, 0.0f, 1.0f);
				}
			} 
			
			if(portalScissor) SetObjectPortalScissor(i);
			curObject->Render();
		}
	}
}

void TScene::ShadowRender() {
	curInfrastructure->PrepareStencilShadowing();
	for(int i = 0; i < ObjectList.size(); i++) {
		
		if( (ObjectList[i].visible && ObjectList[i].sectorIndex != curSectorIndex && curPortalSwitch == false)// && ObjectList[i].reflect == reflectSwitch)
			|| ((ObjectList[i].sectorIndex == curSectorIndex) && curPortalSwitch == true)
			) { //visible, so render it
			TObject *curObject = ObjectList[i].curObject;
			if(ObjectList[i].shadow >= true) {
				//code to prevent shadow leakage into other portals... basically a quick hack
				Vector *pBound1 = &PortalList[ObjectList[i].sectorIndex].DstBound1;
				Vector *pBound2 = &PortalList[ObjectList[i].sectorIndex].DstBound2;
				Vector pVec;
				Vsub(pBound1, pBound2, &pVec);
				GLfloat pLength = Vlength(&pVec);
				
				Vector pMidpnt;
				Vadd(pBound1, pBound2, &pMidpnt);
				Vscale(&pMidpnt, 0.5);
				Vector objMidpnt, objDirection;
				Vadd(&curObject->GetMaxBounds(), &curObject->GetMinBounds(), &objMidpnt);
				Vscale(&objMidpnt, 0.5);
				Vsub(&pMidpnt, &objMidpnt, &objDirection);
				GLfloat relativeLength = Vlength(&objDirection);
				
				Vector oldLight = *curObject->GetLight(), newLight;
				newLight.x = (objMidpnt.x - oldLight.x) / pLength * relativeLength + oldLight.x;
				newLight.y = (objMidpnt.y - oldLight.y) / pLength * relativeLength + oldLight.y;
				newLight.z = (objMidpnt.z - oldLight.z) / pLength * relativeLength + oldLight.z;
				
				//now render the shadow volume
				curObject->SetLightPos(newLight.x, newLight.y, newLight.z);
				for(int j = 0; j < LightList.size(); j++) {
					if(LightList[j].sectorIndex == ObjectList[i].sectorIndex) {
						castScissorsShadow(curObject, &newLight, isZpass(&newLight, curObject), 
										LightList[j].brightness);
						break;
					}
				}
				curObject->SetLightPos(oldLight.x, oldLight.y, oldLight.z);
			}
		}
	}

	curInfrastructure->RenderShadowRect(0.0f, 0.0f, 0.0f, 0.5f);
	curInfrastructure->EndStencilShadowing();
}

void TScene::AttenuateRender() {
	for(int i = 0; i < ObjectList.size(); i++) {
		if( (ObjectList[i].visible && ObjectList[i].attenuate && ObjectList[i].sectorIndex != curSectorIndex && curPortalSwitch == false)// && ObjectList[i].reflect == reflectSwitch)
			|| (ObjectList[i].visible && ObjectList[i].attenuate && ObjectList[i].sectorIndex == curSectorIndex && curPortalSwitch == true && ObjectList[i].reflect == reflectSwitch)
			) {
			bool rendered = false;
			TObject *curObject = ObjectList[i].curObject;
			if(portalScissor) SetObjectPortalScissor(i);
				
			if(ObjectList[i].reflect && ObjectList[i].sectorIndex == curSectorIndex && reflectSwitch == true) {
				curObject->EnablePlainDecal();
				curObject->SelectSingleDecalTexture();
				glEnable(GL_BLEND);
				glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
				curObject->SetupVertexArraysForRendering(DECAL_MODE);
				glColor4f(1.0f, 1.0f, 1.0f, ObjectList[i].alpha);  /* set alpha value for reflection blending */
				curObject->Render();
				continue;
			} 
			
			for(int j = 0; j < LightList.size(); j++) {
				if(LightList[j].sectorIndex == ObjectList[i].sectorIndex) {
					lightRed = LightList[j].r;
					lightGreen = LightList[j].g;
					lightBlue = LightList[j].b;
					brightness = LightList[j].brightness;
					
					if(brightness > 0) {
						curObject->EnableAttenuatePass2(lightRed, lightBlue, lightGreen);
							
						curObject->SetLightBrightness(brightness);
						curObject->SetLightPos(LightList[j].dstLightVec.x, LightList[j].dstLightVec.y, LightList[j].dstLightVec.z);
						curObject->SetupLightVectorsForRendering();
						curObject->SetupVertexArraysForRendering(ATTENUATE_MODE);
						curObject->Render();
						
						rendered = true;
					}
				}
			}
			
			if(rendered) {
				curObject->EnablePlainDecal();
				curObject->SelectSingleDecalTexture();
				curObject->SetupVertexArraysForRendering(DECAL_MODE);
				glEnable(GL_BLEND);
				
				glColor4f(1.0, 1.0, 1.0, 1.0);
				glBlendFunc(GL_DST_COLOR, GL_ZERO);
				curObject->Render();
				
			}
		}
	}
}

void TScene::RenderDecalGlow() {
	for(int i = 0; i < ObjectList.size(); i++) {
		if(ObjectList[i].visible) {
			TObject *curObject = ObjectList[i].curObject;
			curObject->EnablePlainDecal();
			curObject->SelectSingleDecalTexture();
			curObject->SetupVertexArraysForRendering(DECAL_MODE);
			if(ObjectList[i].shadow == GLOW_TRUE) {
				glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
			}
			else {
				glColor4f(0.0f, 0.0f, 0.0f, 0.0f);
			}
			if(portalScissor) SetObjectPortalScissor(i);
			curObject->Render();
		}
	}
}

void TScene::RenderDiffuseGlow() {
	for(int i = 0; i < ObjectList.size(); i++) {
		if(ObjectList[i].visible && ObjectList[i].shadow == GLOW_TRUE) {
			TObject *curObject = ObjectList[i].curObject;
			if(portalScissor) SetObjectPortalScissor(i);
			float curDiffuseVal = 0.8; 
			
			curObject->EnableDiffusePass(0.2f, 0.2f, 0.2f, 0.0f, curDiffuseVal, curDiffuseVal, curDiffuseVal, 0.0f);
			curObject->SelectCubeAndBumpTexture();
			curObject->SetupVertexArraysForRendering(DIFFUSE_BUMP_MODE);
			curObject->Render();
		}
	}
	
}

void TScene::RenderSpecularGlow() {
	for(int i = 0; i < ObjectList.size(); i++) {
		if(ObjectList[i].visible && ObjectList[i].shadow == GLOW_TRUE) {
			TObject *curObject = ObjectList[i].curObject;
			if(portalScissor) SetObjectPortalScissor(i);
			float curDiffuseVal = GetObjectDiffuseVal(curObject);
			
			curObject->SelectCubeAndBumpTexture();
			curObject->EnableSpecularPass();
			curObject->SetupVertexArraysForRendering(DIFFUSE_BUMP_MODE);
			curObject->Render();
		}
	}
}

bool TScene::GlowExist() {
	if(renderLevel < RENDER_LEVEL2) return false;
	for(int i = 0; i < ObjectList.size(); i++) {
		if(ObjectList[i].shadow == GLOW_TRUE && ObjectList[i].visible) return true;
	}
	
	return false;
}


void TScene::RenderAttenuateGlow() {
	
}

void TScene::RenderGlowAll() {
	
	if(GlowExist() ) {
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);
		glViewport(0, 0, WINDOW_TEXTURE_SIZE, WINDOW_TEXTURE_SIZE);
		RenderDecalGlow();
		RenderDiffuseGlow();
		RenderSpecularGlow();
		glViewport(0, 0, curInfrastructure->GetWinWidth(), curInfrastructure->GetWinHeight());
		curInfrastructure->copyScreenToBuffer(NUM_WINDOW_TEXTURES);
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);
	}
}

void TScene::GlowRender() {
	float factor = 0.0;
	if(GlowExist() ) {
		glActiveTextureARB(GL_TEXTURE1_ARB );
		glDisable(GL_TEXTURE_CUBE_MAP_ARB );
		glDisable(GL_TEXTURE_2D);

		glActiveTextureARB( GL_TEXTURE0_ARB );
		glDisable( GL_TEXTURE_CUBE_MAP_ARB );
		glDisable(GL_CULL_FACE );
		glDisable(GL_REGISTER_COMBINERS_NV);
		glEnable( GL_TEXTURE_2D);
		glDisable( GL_DEPTH_TEST );
		
		glMatrixMode( GL_PROJECTION );
			glPushMatrix( );
			glLoadIdentity( );
			glOrtho(0, curInfrastructure->GetWinWidth(), 0, curInfrastructure->GetWinHeight(), -1, 1);
		
		glMatrixMode( GL_MODELVIEW );
			glPushMatrix( );
			glLoadIdentity( );

		glEnable(GL_BLEND);
		glBlendFunc(GL_ONE, GL_ONE);
		//supersample and average down			
		glBindTexture(GL_TEXTURE_2D, curInfrastructure->getWindowTexture(NUM_WINDOW_TEXTURES));
		int num_samples = 0;
		if(renderLevel > RENDER_LEVEL2) num_samples = 16;
		else num_samples = 9;
		glColor4f(1.0 / (float)num_samples, 1.0 / (float)num_samples, 1.0 / (float)num_samples, 1.0 / (float)num_samples);
		for(int i = 0; i < num_samples; i++) {
			glBegin(GL_QUADS);
				glTexCoord2f(0.0, 0.0);
				glVertex2f(0.0 - factor + jitterOffset[i].x, 0.0 - factor + jitterOffset[i].y);
				
				glTexCoord2f(1.0, 0.0);
				glVertex2f(curInfrastructure->GetWinWidth() - 1.0 + factor + jitterOffset[i].x, 0.0 - factor + jitterOffset[i].y);
				
				glTexCoord2f(1.0, 1.0);
				glVertex2f(curInfrastructure->GetWinWidth() - 1.0 + factor + jitterOffset[i].x, curInfrastructure->GetWinHeight() - 1.0 + factor + jitterOffset[i].y);
				
				glTexCoord2f(0.0, 1.0);
				glVertex2f(0.0 - factor + jitterOffset[i].x, curInfrastructure->GetWinHeight() - 1 + factor + jitterOffset[i].y);
			glEnd();
		}
		glMatrixMode( GL_PROJECTION );
		glPopMatrix( );
		glMatrixMode( GL_MODELVIEW );
		glPopMatrix( );
		glEnable( GL_DEPTH_TEST );
		glEnable(GL_CULL_FACE );
		glDisable( GL_TEXTURE_2D );
		glDisable(GL_BLEND);
				
	}
}



float TScene::GetObjectDiffuseVal(TObject *curObject) {
	float curLength = 0.0, curDiffuseVal = 0.0;
	Vector curVec, evalVec;
	for(int j = 0; j < LightList.size(); j++) {
		evalVec = *curObject->GetDstBounds1();
		Vsub(&LightList[j].dstLightVec, &evalVec, &curVec);
		curLength = Vlength(&curVec);
		if(curLength <= LightList[j].brightness) curDiffuseVal = curDiffuseVal + 0.1;
		if(curDiffuseVal > 0.4) { curDiffuseVal = 0.45; break;}
			
		evalVec = *curObject->GetDstBounds2();
		Vsub(&LightList[j].dstLightVec, &evalVec, &curVec);
		curLength = Vlength(&curVec);
		if(curLength <= LightList[j].brightness) curDiffuseVal = curDiffuseVal + 0.1;
		if(curDiffuseVal > 0.4) { curDiffuseVal = 0.45; break;}
			
		evalVec = *curObject->GetDstBounds3();
		Vsub(&LightList[j].dstLightVec, &evalVec, &curVec);
		curLength = Vlength(&curVec);
		if(curLength <= LightList[j].brightness) curDiffuseVal = curDiffuseVal + 0.1;
		if(curDiffuseVal > 0.4) { curDiffuseVal = 0.45; break;}
			
		evalVec = *curObject->GetDstBounds4();
		Vsub(&LightList[j].dstLightVec, &evalVec, &curVec);
		curLength = Vlength(&curVec);
		if(curLength <= LightList[j].brightness) curDiffuseVal = curDiffuseVal + 0.1;
		if(curDiffuseVal > 0.4) { curDiffuseVal = 0.45; break;}
			
		evalVec = *curObject->GetDstBounds5();
		Vsub(&LightList[j].dstLightVec, &evalVec, &curVec);
		curLength = Vlength(&curVec);
		if(curLength <= LightList[j].brightness) curDiffuseVal = curDiffuseVal + 0.1;
		if(curDiffuseVal > 0.4) { curDiffuseVal = 0.45; break;}
			
		evalVec = *curObject->GetDstBounds6();
		Vsub(&LightList[j].dstLightVec, &evalVec, &curVec);
		curLength = Vlength(&curVec);
		if(curLength <= LightList[j].brightness) curDiffuseVal = curDiffuseVal + 0.1;
		if(curDiffuseVal > 0.4) { curDiffuseVal = 0.45; break;}
			
		evalVec = *curObject->GetDstBounds7();
		Vsub(&LightList[j].dstLightVec, &evalVec, &curVec);
		curLength = Vlength(&curVec);
		if(curLength <= LightList[j].brightness) curDiffuseVal = curDiffuseVal + 0.1;
		if(curDiffuseVal > 0.4) { curDiffuseVal = 0.45; break;}
	
		evalVec = *curObject->GetDstBounds8();
		Vsub(&LightList[j].dstLightVec, &evalVec, &curVec);
		curLength = Vlength(&curVec);
		if(curLength <= LightList[j].brightness) curDiffuseVal = curDiffuseVal + 0.1;
		if(curDiffuseVal > 0.4) { curDiffuseVal = 0.45; break;}
	}
	
	return curDiffuseVal;
}

void TScene::DiffuseRender() {

	for(int i = 0; i < ObjectList.size(); i++) {
		if( (ObjectList[i].visible && ObjectList[i].sectorIndex != curSectorIndex && curPortalSwitch == false && ObjectList[i].diffuse)// && ObjectList[i].reflect == reflectSwitch)
			|| (ObjectList[i].visible && ObjectList[i].sectorIndex == curSectorIndex && curPortalSwitch == true && ObjectList[i].diffuse && ObjectList[i].reflect == reflectSwitch)
			) {

			TObject *curObject = ObjectList[i].curObject;
			if(portalScissor) SetObjectPortalScissor(i);
			float curDiffuseVal = GetObjectDiffuseVal(curObject);
			
			curObject->EnableDiffusePass(0.2f, 0.2f, 0.2f, 0.0f, curDiffuseVal, curDiffuseVal, curDiffuseVal, 0.0f);
			curObject->SelectCubeAndBumpTexture();
			curObject->SetupVertexArraysForRendering(DIFFUSE_BUMP_MODE);
			curObject->Render();
		}
	}
	
}

void TScene::SpecularRender() {
	for(int i = 0; i < ObjectList.size(); i++) {
		if( (ObjectList[i].visible && ObjectList[i].sectorIndex != curSectorIndex && curPortalSwitch == false && ObjectList[i].specular)// && ObjectList[i].reflect == reflectSwitch)
			|| (ObjectList[i].visible && ObjectList[i].sectorIndex == curSectorIndex && curPortalSwitch == true && ObjectList[i].specular && ObjectList[i].reflect == reflectSwitch)
			) {

			TObject *curObject = ObjectList[i].curObject;
			if(portalScissor) SetObjectPortalScissor(i);
			float curDiffuseVal = GetObjectDiffuseVal(curObject);
			
			if(curDiffuseVal > 0.1) {
				curObject->SelectCubeAndBumpTexture();
				curObject->EnableSpecularPass();
				curObject->SetupVertexArraysForRendering(SPECULAR_BUMP_MODE);
				curObject->Render();
			}
		}
	}
}

void TScene::makeSceneCopy() {
}


void TScene::ChromaticRefractRender() {
	TObject *curObject = NULL;
	for(int i = 0; i < ObjectList.size(); i++) {
		if(ObjectList[i].visible && ObjectList[i].shadow == CHROM_REFRACT_TRUE) {
			curObject = ObjectList[i].curObject;
			
			Vector *v2[8], final[8];
			v2[0] = ( curObject->GetDstBounds1() );
			v2[1] = ( curObject->GetDstBounds2() );
			v2[2] = ( curObject->GetDstBounds3() );
			v2[3] = ( curObject->GetDstBounds4() );
			v2[4] = ( curObject->GetDstBounds5() );
			v2[5] = ( curObject->GetDstBounds6() );
			v2[6] = ( curObject->GetDstBounds7() );
			v2[7] = ( curObject->GetDstBounds8() );
			
			GLint viewport[4];
			GLdouble modelMatrix[16];
			GLdouble projMatrix[16];
			GLdouble screenX[8], screenY[8], screenZ;
			double xMax = -99999.0, yMax = -99999.0, xMin = 99999.0, yMin = 99999.0;
			
			glGetIntegerv(GL_VIEWPORT, viewport);
			glMatrixMode(GL_MODELVIEW);
				glPushMatrix();
				glLoadIdentity();
				glGetDoublev(GL_MODELVIEW_MATRIX, modelMatrix);
				glGetDoublev(GL_PROJECTION_MATRIX, projMatrix);
			glPopMatrix();
			//project all coordinates into screenspace
			for(int j = 0; j < 8; j++) { 
				gluProject((GLdouble)v2[j]->x, (GLdouble)v2[j]->y, (GLdouble)v2[j]->z, modelMatrix, projMatrix,
						viewport, &screenX[j], &screenY[j], &screenZ);
			}
			
			for(j = 0; j < 8; j++) {
				if(screenX[j] > xMax) xMax = screenX[j];
				if(screenY[j] > yMax) yMax = screenY[j];
				if(screenX[j] < xMin) xMin = screenX[j];
				if(screenY[j] < yMin) yMin = screenY[j];
			}

			float ratio = fabs((xMax - xMin) * (yMax - yMin));
			
				if(ratio >= 65536.0) {
					if(renderLevel == RENDER_LEVEL3)	{
						curObject->setCubeMapRenderSize(2);
						RenderCubeMapFaces(128, curObject, ObjectList[i].sectorIndex, i);
					}
					else {
						curObject->setCubeMapRenderSize(1);
						RenderCubeMapFaces(64, curObject, ObjectList[i].sectorIndex, i);
					}
				}
				else if(ratio > 4096.0 && ratio < 65536.0 ) {
					curObject->setCubeMapRenderSize(1);
					RenderCubeMapFaces(64, curObject, ObjectList[i].sectorIndex, i);
				}
				else {
					curObject->setCubeMapRenderSize(0);
					RenderCubeMapFaces(32, curObject, ObjectList[i].sectorIndex, i);
				}
			}
	}
	
	PrepareSceneForPortalCulling();
	PortalCull();
	FrustrumCull();

	glMatrixMode(GL_PROJECTION);
		glPopMatrix();
		glLoadIdentity();
		curInfrastructure->iniProjectionMatrix();
	glMatrixMode(GL_MODELVIEW);
	

	for(int i = 0; i < ObjectList.size(); i++) {
		if(ObjectList[i].visible && ObjectList[i].shadow == CHROM_REFRACT_TRUE) {
			glEnable(GL_VERTEX_PROGRAM_NV);
			ObjectList[i].visible = false;
			curObject = ObjectList[i].curObject;
			curObject->EnableRefractPass1(inverseViewerMatrix);
			curObject->SetupVertexArraysForRendering(REFRACT_MODE_1);
			curObject->Render();
			curObject->EnableRefractPass2(inverseViewerMatrix);
			curObject->SetupVertexArraysForRendering(REFRACT_MODE_2);
			curObject->Render();
			glDisable(GL_VERTEX_PROGRAM_NV);
		}
	}
}

void TScene::DecalSectorRender(int sectorIndex, GLfloat *mat) {
	for(int i = 0; i < ObjectList.size(); i++) {
		if( (ObjectList[i].sectorIndex == sectorIndex && ObjectList[i].reflect == false && !ObjectList[i].attenuate)	)
		{ //visible, so render it
			TObject *curObject = ObjectList[i].curObject;
			
			if(curObject->GetType() == MD2_TYPE && !ObjectList[i].visible) {
				TMd2 *curMd2 = (TMd2 *)curObject;
				curMd2->PlayAnimation();
			}
			
			curObject->EnablePlainDecal();
			curObject->SelectSingleDecalTexture();
			curObject->SetupVertexArraysForRendering(DECAL_MODE);
			curObject->RenderWithMatrix(mat);
		}
	}
}

void TScene::DiffuseSectorRender(int sectorIndex, GLfloat *mat) {
	for(int i = 0; i < ObjectList.size(); i++) {
		if((ObjectList[i].sectorIndex == sectorIndex && ObjectList[i].diffuse && ObjectList[i].reflect == false))
		 {
			TObject *curObject = ObjectList[i].curObject;
			
			
			curObject->EnableDiffusePass(0.2f, 0.2f, 0.2f, 0.0f, 0.45f, 0.45f, 0.45f, 0.0f);
			curObject->SelectCubeAndBumpTexture();
			curObject->SetupVertexArraysForRendering(DIFFUSE_BUMP_MODE);
			curObject->RenderWithMatrix(mat);
		}
	}

}

void TScene::SpecularSectorRender(int sectorIndex, GLfloat *mat) {
	for(int i = 0; i < ObjectList.size(); i++) {
		if((ObjectList[i].sectorIndex == sectorIndex && ObjectList[i].specular && ObjectList[i].reflect == false)
			) {

			TObject *curObject = ObjectList[i].curObject;
			
			
			curObject->SelectCubeAndBumpTexture();
			curObject->EnableSpecularPass();
			curObject->SetupVertexArraysForRendering(SPECULAR_BUMP_MODE);
			curObject->RenderWithMatrix(mat);
		}
	}
}

void TScene::AttenuateSectorRender(int sectorIndex, GLfloat *mat) {
	for(int i = 0; i < ObjectList.size(); i++) {
		if( (ObjectList[i].attenuate && ObjectList[i].sectorIndex == sectorIndex && ObjectList[i].reflect == false))
		 {
			bool rendered = false;
			TObject *curObject = ObjectList[i].curObject;
			
			for(int j = 0; j < LightList.size(); j++) {
				if(LightList[j].sectorIndex == ObjectList[i].sectorIndex) {
					lightRed = LightList[j].r;
					lightGreen = LightList[j].g;
					lightBlue = LightList[j].b;
					brightness = LightList[j].brightness;
					
					if(brightness > 0) {
						curObject->EnableAttenuatePass2(lightRed, lightBlue, lightGreen);
							
						curObject->SetLightBrightness(brightness);
						curObject->SetLightPos(LightList[j].dstLightVec.x, LightList[j].dstLightVec.y, LightList[j].dstLightVec.z);
						curObject->SetupLightVectorsForRendering();
						curObject->SetupVertexArraysForRendering(ATTENUATE_MODE);
						curObject->RenderWithMatrix(mat);
						
						rendered = true;
					}
				}
			}
			
			if(rendered) {
				curObject->EnablePlainDecal();
				curObject->SelectSingleDecalTexture();
				curObject->SetupVertexArraysForRendering(DECAL_MODE);
				glEnable(GL_BLEND);
				
				glColor4f(1.0, 1.0, 1.0, 1.0);
				glBlendFunc(GL_DST_COLOR, GL_ZERO);
				curObject->RenderWithMatrix(mat);
				
			}

		}
	}
}


void TScene::ReflectorsRender() {
	bool renderReflect = false;
	GLfloat reflectY = 0.0f;
	Vector *curReflectVec;

	//Don't update color or depth.
	glDisable(GL_DEPTH_TEST);
	glColorMask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE);

	//Draw 1 into the stencil buffer.
	glEnable(GL_STENCIL_TEST);
	glStencilOp(GL_REPLACE, GL_REPLACE, GL_REPLACE);
	glStencilFunc(GL_ALWAYS, 1, 0xffffffff);
	
	for(int i = 0; i < ObjectList.size(); i++) {
		if(ObjectList[i].sectorIndex == curSectorIndex && ObjectList[i].reflect) { // && ObjectList[i].visible) {
			TObject *curObject = ObjectList[i].curObject;
			curObject->SetupVertexArraysForRendering(PLAIN_MODE);
			curObject->Render();
			curReflectVec = curObject->GetDstBounds1();
			renderReflect = true;
		} 
	}


	//Re-enable update of color and depth.
	glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
	glEnable(GL_DEPTH_TEST);

	//Now, only render where stencil is set to 1.
	glStencilFunc(GL_EQUAL, 1, 0xffffffff);  /* draw if stencil ==1 */
	glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);

	//flip the objects in the sector and render them as reflections
	if(renderReflect) {
	
		GLfloat restoreRotMat[16][16];
		Vector newReflectVec;
		glMatrixMode(GL_MODELVIEW);
			glPushMatrix();
			glLoadIdentity();
			glRotatef(viewerPhysics.angle.x, 1.0, 0.0, 0.0);
			glGetFloatv(GL_MODELVIEW_MATRIX, &restoreRotMat[0][0]);
		glPopMatrix();
		transformVector(&newReflectVec, curReflectVec, &restoreRotMat[0][0]);
		reflectY = newReflectVec.y;
	}
	
		GLfloat mat[16][16];
		glMatrixMode(GL_MODELVIEW);
			glPushMatrix();
			glLoadIdentity();
			glRotatef(-viewerPhysics.angle.x, 1.0, 0.0, 0.0);
			glTranslatef(0.0f, reflectY, 0.0);
			glScalef(1.0f, -1.0f, 1.0f);
			glTranslatef(0.0, -reflectY, 0.0);
			glRotatef(viewerPhysics.angle.x, 1.0, 0.0, 0.0);
			glGetFloatv(GL_MODELVIEW_MATRIX, &mat[0][0]);
		glPopMatrix();

		glFrontFace(GL_CCW);

		//first pass, decal render
		DecalSectorRender(curSectorIndex, &mat[0][0]);

		//second pass, diffuse/attenuate render
		AttenuateSectorRender(curSectorIndex, &mat[0][0]);
		DiffuseSectorRender(curSectorIndex, &mat[0][0]);

		//third pass, specular render
		SpecularSectorRender(curSectorIndex, &mat[0][0]);

		glDisable(GL_STENCIL_TEST);
		glFrontFace(GL_CW);

	//	glColorMask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE);

		reflectSwitch = true;

		//render the mirror again so as to correct the depth values in the z-buffer
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		//first pass, decal render
		DecalRender();
		//second pass, diffuse/attenuate render
		AttenuateRender();
		DiffuseRender();
		//third pass, specular render
		SpecularRender();
		glDisable(GL_BLEND);

	//	glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
		reflectSwitch = false;
	

}


void TScene::ExtractPlanes() {
	PMatrix4x4 curMatrix;
	glGetFloatv(GL_PROJECTION_MATRIX, (GLfloat *)&curMatrix);

	//left clipping plane
	viewPlane[0].A = curMatrix._41 + curMatrix._11;
	viewPlane[0].B = curMatrix._42 + curMatrix._12;
	viewPlane[0].C = curMatrix._43 + curMatrix._13;
	viewPlane[0].D = curMatrix._44 + curMatrix._14;

	//right clipping plane
	viewPlane[1].A = curMatrix._41 - curMatrix._11;
	viewPlane[1].B = curMatrix._42 - curMatrix._12;
	viewPlane[1].C = curMatrix._43 - curMatrix._13;
	viewPlane[1].D = curMatrix._44 - curMatrix._14;

	//top clipping plane
	viewPlane[2].A = curMatrix._41 - curMatrix._21;
	viewPlane[2].B = curMatrix._42 - curMatrix._22;
	viewPlane[2].C = curMatrix._43 - curMatrix._23;
	viewPlane[2].D = curMatrix._44 - curMatrix._24;

	//bottom clipping plane
	viewPlane[3].A = curMatrix._41 + curMatrix._21;
	viewPlane[3].B = curMatrix._42 + curMatrix._22;
	viewPlane[3].C = curMatrix._43 + curMatrix._23;
	viewPlane[3].D = curMatrix._44 + curMatrix._24;

	//near clipping plane
	viewPlane[4].A = curMatrix._41 + curMatrix._31;
	viewPlane[4].B = curMatrix._42 + curMatrix._32;
	viewPlane[4].C = curMatrix._43 + curMatrix._33;
	viewPlane[4].D = curMatrix._44 + curMatrix._34;

	//far clipping plane
	viewPlane[5].A = curMatrix._41 - curMatrix._31;
	viewPlane[5].B = curMatrix._42 - curMatrix._32;
	viewPlane[5].C = curMatrix._43 - curMatrix._33;
	viewPlane[5].D = curMatrix._44 - curMatrix._34;

	for(int i = 0; i < 6; i++) normalizePlane(&viewPlane[i]);
	
	//now fill in the boundary coordinates for the near plane
	int index1, index2, index3;
	
	ExtractNearCoords(0, NEAR_CLIPPING, TOP_CLIPPING, RIGHT_CLIPPING);
	ExtractNearCoords(1, NEAR_CLIPPING, TOP_CLIPPING, LEFT_CLIPPING);
	ExtractNearCoords(2, NEAR_CLIPPING, BOTTOM_CLIPPING, LEFT_CLIPPING);
	ExtractNearCoords(3, NEAR_CLIPPING, BOTTOM_CLIPPING, RIGHT_CLIPPING);
}

void TScene::ExtractNearCoords(int cIndex, int index1, int index2, int index3) {
	/* p0 = ( -d1 * (n2 x n3) -d2 * (n3 x n1) -d3 * (n1 x n2) ) / (n1 . (n2 x n3)) */
	
	Vector N2CrossN3, N3CrossN1, N1CrossN2;
	float factor = 0.0;
		
	Vcross((Vector *)&viewPlane[index2], (Vector *)&viewPlane[index3], &N2CrossN3);
	Vcross((Vector *)&viewPlane[index1], (Vector *)&viewPlane[index2], &N1CrossN2);
	Vcross((Vector *)&viewPlane[index3], (Vector *)&viewPlane[index1], &N3CrossN1);
		
	factor = 1.0 / (float)Vdot((Vector *)&viewPlane[index1], &N2CrossN3);
	Vscale(&N2CrossN3, -viewPlane[index1].D);
	Vscale(&N3CrossN1, -viewPlane[index2].D);
	Vscale(&N1CrossN2, -viewPlane[index3].D);
		
	nearPlaneCoords3D[cIndex] = N2CrossN3;
	Vadd(&nearPlaneCoords3D[cIndex], &N3CrossN1, &nearPlaneCoords3D[cIndex]);
	Vadd(&nearPlaneCoords3D[cIndex], &N1CrossN2, &nearPlaneCoords3D[cIndex]);
	Vscale(&nearPlaneCoords3D[cIndex], factor);
}

//shadow optimization routine, determine if the object falls in the light's pyramid. If it doesn't, we can employ zpass for shadowing
bool TScene::isZpass(Vector *curLight, TObject *curObject) {
	bool retval = true;
	Plane orgPlane[6];
	//save the planes first
	for(int i = 0; i < 6; i++) orgPlane[i] = viewPlane[i];
	//form the light planes
	Plane lightPlane[4];
	//top
	makePlane(&nearPlaneCoords3D[0], &nearPlaneCoords3D[1], curLight, &lightPlane[0] );
	//right
	makePlane(&nearPlaneCoords3D[0], &nearPlaneCoords3D[3], curLight, &lightPlane[1] );
	//left
	makePlane(&nearPlaneCoords3D[2], &nearPlaneCoords3D[1], curLight, &lightPlane[2] );
	//bottom
	makePlane(&nearPlaneCoords3D[2], &nearPlaneCoords3D[3], curLight, &lightPlane[3] );
	
	viewPlane[TOP_CLIPPING] = lightPlane[0];
	viewPlane[RIGHT_CLIPPING] = lightPlane[1];
	viewPlane[LEFT_CLIPPING] = lightPlane[2];
	viewPlane[BOTTOM_CLIPPING] = lightPlane[3];
	
	if(ContainsSphere(curObject) != OUT_FLAG) {
		retval = false;
	}
	//restore the planes
	for(i = 0; i < 6; i++) viewPlane[i] = orgPlane[i];
	
	return retval;
}


//another shadow optimization routine, scissors optmization for shadow volumes
void TScene::castScissorsShadow(TObject *curObject, Vector *curLight, bool zPass, float brightness) {
	float brightnessSquare = brightness * brightness * 2.25;
	float radius = (float)sqrt(brightnessSquare + brightnessSquare);
	bool skipScissors = false;
	/* line sphere intersection
	a u2 + b u + c = 0

	 where:

	a = (x2 - x1) ^2 + (y2 - y1) ^2 + (z2 - z1) ^2 
	b = 2[ (x2 - x1) (x1 - x3) + (y2 - y1) (y1 - y3) + (z2 - z1) (z1 - z3) ] 
	c = x3 ^2 + y3 ^2 + z3 ^2 + x1 ^2 + y1 ^2 + z1 ^2 - 2[x3 x1 + y3 y1 + z3 z1] - r ^ 2 	*/
	float a, b, c, t, det;
	int noCnt = 0, trueCnt = 0;
	double xMax = -99999.0, yMax = -99999.0, xMin = 99999.0, yMin = 99999.0;
	Vector v1, *v2[8], v3, final[8];
	Vector V2SubV1, V1SubV3;
	v3 = *curLight; // centre of sphere
	v1 = *curLight;
	v2[0] = ( curObject->GetDstBounds1() );
	v2[1] = ( curObject->GetDstBounds2() );
	v2[2] = ( curObject->GetDstBounds3() );
	v2[3] = ( curObject->GetDstBounds4() );
	v2[4] = ( curObject->GetDstBounds5() );
	v2[5] = ( curObject->GetDstBounds6() );
	v2[6] = ( curObject->GetDstBounds7() );
	v2[7] = ( curObject->GetDstBounds8() );

	//now find all the points of intersection
	for(int i = 0; i < 8; i++) {
		Vsub(v2[i], &v1, &V2SubV1);
		Vsub(&v1, &v3, &V1SubV3);
		a = V2SubV1.x * V2SubV1.x + V2SubV1.y * V2SubV1.y + V2SubV1.z * V2SubV1.z;
		b = 2.0 * ( V2SubV1.x * V1SubV3.x + V2SubV1.y * V1SubV3.y + V2SubV1.z * V1SubV3.z);
		c = (v3.x * v3.x + v3.y * v3.y + v3.z * v3.z) + (v1.x * v1.x + v1.y * v1.y + v1.z * v1.z) -
			2.0 * (v3.x * v1.x + v3.y * v1.y + v3.z * v1.z) - radius * radius;
		
		det = b * b - 4.0 * a * c;
		if(det <= 0.0)  {noCnt++; }
		else {
			float sqrtDet = (float)sqrt(det);
			float curval1 = (-b + sqrtDet) / (2.0 * a);
			float curval2 = (-b - sqrtDet) / (2.0 * a);
			
			if(curval1 >= 0 && curval2 < 0) {
				Vscale(&V2SubV1, curval1);
				Vadd(&v1, &V2SubV1, &final[trueCnt]);
				if(final[trueCnt].z > -1.0f) { skipScissors = true; break;}
				trueCnt++;
			}
			else if(curval1 < 0 && curval2 >= 0) {
				Vscale(&V2SubV1, curval2);
				Vadd(&v1, &V2SubV1, &final[trueCnt]);
				if(final[trueCnt].z > -1.0f) { skipScissors = true; break;}
				trueCnt++;
			}
			else {
				noCnt++;
			}
			
		}
		
	}
	
	for(i = 0; i < 8; i++) { if(v2[i]->z > 1.0f) { skipScissors = true; break; } }
	if(trueCnt == 0) skipScissors = true;
	
	if(!skipScissors) {
		GLint viewport[4];
		GLdouble modelMatrix[16];
		GLdouble projMatrix[16];
		GLdouble screenX[16], screenY[16], screenZ;
		glGetIntegerv(GL_VIEWPORT, viewport);
		glMatrixMode(GL_MODELVIEW);
			glPushMatrix();
			glLoadIdentity();
			glGetDoublev(GL_MODELVIEW_MATRIX, modelMatrix);
			glGetDoublev(GL_PROJECTION_MATRIX, projMatrix);
		glPopMatrix();
		//project all coordinates into screenspace
		for(i = 0; i < trueCnt; i++) { 
			gluProject((GLdouble)final[i].x, (GLdouble)final[i].y, (GLdouble)final[i].z, modelMatrix, projMatrix,
					viewport, &screenX[i], &screenY[i], &screenZ);
		}
	
		for(i = trueCnt; i < trueCnt + 8; i++) {
			gluProject((GLdouble)v2[i - trueCnt]->x, (GLdouble)v2[i - trueCnt]->y, (GLdouble)v2[i- trueCnt]->z, modelMatrix, projMatrix,
					viewport, &screenX[i], &screenY[i], &screenZ);
		}
		
		for(i = 0; i < trueCnt + 8; i++) {
			if(screenX[i] > xMax) xMax = screenX[i];
			if(screenY[i] > yMax) yMax = screenY[i];
			if(screenX[i] < xMin) xMin = screenX[i];
			if(screenY[i] < yMin) yMin = screenY[i];
		}
	}

	if(!skipScissors) {
		glScissor(xMin, yMin, (GLuint)(xMax - xMin), (GLuint)(yMax - yMin) );
		glEnable(GL_SCISSOR_TEST);
	}
	curObject->CastShadowDepthPass(zPass);
	if(!skipScissors) glDisable(GL_SCISSOR_TEST);
}

int TScene::ContainsSphere(TObject *curObject) {
	Vector center, diag;
	float radius;
	center.x = (curObject->GetDstBounds1()->x + curObject->GetDstBounds2()->x) * 0.5f;
	center.y = (curObject->GetDstBounds1()->y + curObject->GetDstBounds2()->y) * 0.5f;
	center.z = (curObject->GetDstBounds1()->z + curObject->GetDstBounds2()->z) * 0.5f;

	diag.x = (curObject->GetDstBounds1()->x - curObject->GetDstBounds2()->x);
	diag.y = (curObject->GetDstBounds1()->y - curObject->GetDstBounds2()->y);
	diag.z = (curObject->GetDstBounds1()->z - curObject->GetDstBounds2()->z);

	radius = Vlength(&diag) * 0.5f;
	
	for(int i = 0; i < 6; i++) {
		//find distance to plane
		Vector pNormal;
		pNormal.x = viewPlane[i].A;
		pNormal.y = viewPlane[i].B;
		pNormal.z = viewPlane[i].C;

		float distance = Vdot(&center, &pNormal) + viewPlane[i].D;
		if(distance < -radius) return OUT_FLAG;

		if(fabs(distance) < radius) return INTERSECT;
	}
	return IN_FLAG;
}

int TScene::ContainsBox(TObject *curObject) {
	Vector vertices[8];
	int totalPntIn = 0;
	int outCnt = 0;
	//retrieve all eight vertices from the bounding box
	vertices[0].x = curObject->GetDstBounds1()->x;
	vertices[0].y = curObject->GetDstBounds1()->y;
	vertices[0].z = curObject->GetDstBounds1()->z;

	vertices[1].x = curObject->GetDstBounds2()->x;
	vertices[1].y = curObject->GetDstBounds2()->y;
	vertices[1].z = curObject->GetDstBounds2()->z;

	vertices[2].x = curObject->GetDstBounds3()->x;
	vertices[2].y = curObject->GetDstBounds3()->y;
	vertices[2].z = curObject->GetDstBounds3()->z;

	vertices[3].x = curObject->GetDstBounds4()->x;
	vertices[3].y = curObject->GetDstBounds4()->y;
	vertices[3].z = curObject->GetDstBounds4()->z;

	vertices[4].x = curObject->GetDstBounds5()->x;
	vertices[4].y = curObject->GetDstBounds5()->y;
	vertices[4].z = curObject->GetDstBounds5()->z;

	vertices[5].x = curObject->GetDstBounds6()->x;
	vertices[5].y = curObject->GetDstBounds6()->y;
	vertices[5].z = curObject->GetDstBounds6()->z;

	vertices[6].x = curObject->GetDstBounds7()->x;
	vertices[6].y = curObject->GetDstBounds7()->y;
	vertices[6].z = curObject->GetDstBounds7()->z;

	vertices[7].x = curObject->GetDstBounds8()->x;
	vertices[7].y = curObject->GetDstBounds8()->y;
	vertices[7].z = curObject->GetDstBounds8()->z;

	for(int i = 0; i < 6; i++) {
		int inCnt = 8;
		int pntIn = 1;
		
		for(int j = 0; j < 8; j++) {
			if(vecPlaneOrientation(&vertices[j], &(this->viewPlane[i])) < 0) {
				pntIn = 0;
				inCnt--;
			}
		}

		//see if all points are out of the plane
		if(inCnt == 0) {
			return OUT_FLAG;
		}

		totalPntIn = totalPntIn + pntIn;
	}

	if(totalPntIn == 6) return IN_FLAG;

	return INTERSECT;
}

void TScene::PrepareSceneForCulling() {
	for(int i = 0; i < ObjectList.size(); i++) ObjectList[i].visible = true;
	for(i = 0; i < PortalList.size(); i++) PortalList[i].processed = false;
}

void TScene::PrepareSceneForPortalCulling() {
	for(int i = 0; i < ObjectList.size(); i++) ObjectList[i].visible = false;
	for(i = 0; i < PortalList.size(); i++) PortalList[i].processed = false;
}

void TScene::FrustrumCull() {
	ExtractPlanes();
	int visCnt = 0;
	for(int i = 0; i < ObjectList.size(); i++) {
		if(ContainsSphere(ObjectList[i].curObject) != OUT_FLAG) {
			if(ContainsBox(ObjectList[i].curObject) != OUT_FLAG) {
				//ObjectList[i].visible = true;       
				visCnt++;
			}
			else ObjectList[i].visible = false;
		}
		else { ObjectList[i].visible = false;}
	}

//	cout << "Objects Visible: " << visCnt << endl;

}

void TScene::PrepareViewerTransform() {
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
}

void TScene::EndViewerTransform() {
	glMatrixMode(GL_MODELVIEW);
	glGetFloatv(GL_MODELVIEW_MATRIX, this->viewerMatrix);
	invertMatrixf(this->inverseViewerMatrix, this->viewerMatrix);
	glPopMatrix();

	
	UpdateViewerBounds();

}

void TScene::ApplyViewerTransform() {
	Vector newLight, newEye;
	transformVector(&newLight, &this->currentLightVec, this->inverseViewerMatrix);
	transformVector(&newEye, &this->currentEyeVec, this->inverseViewerMatrix);

	this->currentLightVec = newLight;
	this->currentEyeVec = newEye;         

	for(int j = 0; j < LightList.size(); j++) {
		transformVector(&LightList[j].dstLightVec, &LightList[j].LightVec, this->inverseViewerMatrix);
		transformVector(&LightList[j].dstEyeVec, &LightList[j].EyeVec, this->inverseViewerMatrix);
	}

	for(int i = 0; i < ObjectList.size(); i++) {
		TObject *curObject = ObjectList[i].curObject;
		curObject->MatrixMult(this->inverseViewerMatrix);
	}

	//transform the Portal Sectors into view
	for(i = 0; i < PortalList.size(); i++) {
		transformVector(&PortalList[i].DstBound1, &PortalList[i].Bound1, this->inverseViewerMatrix);
		transformVector(&PortalList[i].DstBound2, &PortalList[i].Bound2, this->inverseViewerMatrix);
		transformVector(&PortalList[i].DstBound3, &PortalList[i].Bound3, this->inverseViewerMatrix);
		transformVector(&PortalList[i].DstBound4, &PortalList[i].Bound4, this->inverseViewerMatrix);
	}

	//transform the Portal Windows into view
	for(i = 0; i < PortalWindowList.size(); i++) {
		transformVector(&PortalWindowList[i].DstBounds1, &PortalWindowList[i].Bounds1, this->inverseViewerMatrix);
		transformVector(&PortalWindowList[i].DstBounds2, &PortalWindowList[i].Bounds2, this->inverseViewerMatrix);
		transformVector(&PortalWindowList[i].DstBounds3, &PortalWindowList[i].Bounds3, this->inverseViewerMatrix);
		transformVector(&PortalWindowList[i].DstBounds4, &PortalWindowList[i].Bounds4, this->inverseViewerMatrix);
	}
}

void TScene::DoLightingCalc() {
	for(int i = 0; i < ObjectList.size(); i++) {
		TObject *curObject = ObjectList[i].curObject;
		for(int j = 0; j < LightList.size(); j++) {
			if(LightList[j].sectorIndex == ObjectList[i].sectorIndex) {
				currentLightVec = LightList[j].dstLightVec;
				currentEyeVec = LightList[j].dstEyeVec;
				break;
			}
		}
		curObject->SetLightPos(currentLightVec.x, currentLightVec.y, currentLightVec.z);
//		curObject->SetEyePos(currentEyeVec.x, currentEyeVec.y, currentEyeVec.z);
		curObject->SetEyePos(viewerLoc.x, viewerLoc.y, viewerLoc.z);
		curObject->SetupLightVectorsForRendering();
	}
}

//adds a sector for portal rendering and returns the index of the created portal
int TScene::AddSector(Vector Bound1, Vector Bound2) {
	PortalObj newPortal;
	newPortal.Bound1 = Bound1;
	newPortal.Bound2 = Bound2;

	newPortal.Bound3.x = Bound2.x;
	newPortal.Bound3.y = Bound1.y;
	newPortal.Bound3.z = Bound1.z;

	newPortal.Bound4.x = Bound1.x;
	newPortal.Bound4.y = Bound2.y;
	newPortal.Bound4.z = Bound2.z;

	PortalList.push_back(newPortal);

	return (PortalList.size() - 1);
}

int TScene::AddWindow(Vector Bound1, Vector Bound2) {
	PortalWindow newWindow;
	newWindow.Bounds1 = Bound1;
	newWindow.Bounds2 = Bound2;

	if(Bound2.x != Bound1.x) {
		newWindow.Bounds3.x = Bound2.x;
		newWindow.Bounds3.y = Bound1.y;
		newWindow.Bounds3.z = Bound1.z;

		newWindow.Bounds4.x = Bound1.x;
		newWindow.Bounds4.y = Bound2.y;
		newWindow.Bounds4.z = Bound2.z;
	}
	else {
		newWindow.Bounds3.x = Bound1.x;
		newWindow.Bounds3.y = Bound1.y;
		newWindow.Bounds3.z = Bound2.z;

		newWindow.Bounds4.x = Bound2.x;
		newWindow.Bounds4.y = Bound2.y;
		newWindow.Bounds4.z = Bound1.z;
	}

	PortalWindowList.push_back(newWindow);

	return (PortalWindowList.size() - 1);
}

int TScene::AddLight(Vector lightVec, Vector eyeVec, float brightlevel, float r, float g, float b, int sectorIndex) {
	LightPacket newLight;
	newLight.EyeVec = eyeVec;
	newLight.LightVec = lightVec;
	newLight.brightness = brightlevel;
	newLight.r = r;
	newLight.g = g;
	newLight.b = b;
	newLight.sectorIndex = sectorIndex;

	LightList.push_back(newLight);

	return (LightList.size() - 1);
}

void TScene::SetLight(Vector lightVec, Vector eyeVec, float brightlevel, float r, float g, float b, int sectorIndex, int index) {
	LightList[index].LightVec = eyeVec;
	LightList[index].EyeVec = eyeVec;
	LightList[index].brightness = brightlevel;
	LightList[index].r = r;
	LightList[index].g = g;
	LightList[index].b = b;
	LightList[index].sectorIndex = sectorIndex;
}



//links 2 sectors together
void TScene::LinkSector(int index1, int index2, int windowIndex) {
	PortalList[index1].link.push_back(index2); //&PortalList[index2]);
	PortalList[index2].link.push_back(index1); //&PortalList[index1]);

	PortalWindowList[windowIndex].index1 = index1;
	PortalWindowList[windowIndex].index2 = index2;
}

/* loads a scene from a file description(case sensitive)
	File Format:
	#define the sectors
	SECTOR Bound1X, Bound1Y, Bound1Z Bound2X, Bound2Y, Bound2Z 
	....
	#define portal windows
	WINDOW Bound1X, Bound1Y, Bound1Z Bound2X, Bound2Y, Bound2Z
	....
	#define portal links
	LINK 0 1 windowIndex
	LINK 1 2 windowIndex
	....
	#define light sources (small letter 'l')
	light locX, locY, locZ eyeX, eyeY, eyeZ brightness r, g, b sectorIndex
	
	#define objects in scene, either .Md2 object, a plane object
	PLANE locX, locY, locZ scaleX, scaleY, scaleZ rotX, rotY, rotZ decalTexture bumpTexture shadowTrue diffuseTrue specularTrue attenuateTrue scale tileFactor sectorIndex
	MD2 filename locX, locY, locZ scaleX, scaleY, scaleZ rotX, rotY, rotZ decalTexture bumpTexture shadowTrue diffuseTrue specularTrue attenuateTrue scale animation smoothness flipNormal
	REFLECTOR locX, locY, locZ scaleX, scaleY, scaleZ rotX, rotY, rotZ decalTexture bumpTexture alphaVal diffuseTrue specularTrue attenuateTrue scale tileFactor sectorIndex
	GAMEPILLAR filename locX, locY, locZ scaleX, scaleY, scaleZ rotX, rotY, rotZ decalTexture bumpTexture shadowTrue diffuseTrue specularTrue attenuateTrue scale animation smoothness flipNormal
*/

int TScene::LoadSceneFile(char *filename) {
	FILE *file;
	char buf[256], *gotbuf;
	char name[256], modelName[256], bumpName[256], decalName[256];
	int matches, specTrue, diffuseTrue, shadowTrue, attenuateTrue, animation, smoothness, windowIndex, sectorIndex, flipnormal;
	float x, y, z, scale, rotate, bscale, tileFactor, brightness, red, green, blue, alphaval;
	Vector vec1, vec2, vec3, vec4;

	file = fopen(filename, "r");
	if (file == NULL) {
		cout << "Could not load scene file: " << filename << "." << endl;
		return LOAD_ERROR;
  }

	while (!feof(file)) {
		gotbuf = is_fgets(buf, sizeof(buf), file);
		if (gotbuf) {
			switch(buf[0]) {
				case '#':
				break;

				case 'S':
					matches = sscanf(buf, "SECTOR %f, %f, %f %f, %f, %f\n",&vec1.x, &vec1.y, &vec1.z, &vec2.x, &vec2.y, &vec2.z);
					if (matches == 6) {
						AddSector(vec1, vec2);
						if(callBackObj != NULL && callBackFunc != NULL) callBackFunc(callBackObj, "Loading Sector...");
					} else {fclose(file); cout << "Bad Parse for SECTOR!" << endl; return BAD_PARSE;}
				break;


				case 'W':
					matches = sscanf(buf, "WINDOW %f, %f, %f %f, %f, %f\n",&vec1.x, &vec1.y, &vec1.z, &vec2.x, &vec2.y, &vec2.z);
					if(callBackObj != NULL && callBackFunc != NULL) callBackFunc(callBackObj, "Loading Window...");
					if (matches == 6) {
						AddWindow(vec1, vec2);
					} else {fclose(file); cout << "Bad Parse for WINDOW!" << endl; return BAD_PARSE;}
				break;

				case 'l':
					matches = sscanf(buf, "light %f, %f, %f %f, %f, %f %f %f, %f, %f %d\n",&vec1.x, &vec1.y, &vec1.z, &vec2.x, &vec2.y, &vec2.z,
										&brightness, &red, &green, &blue, &sectorIndex);
						if(callBackObj != NULL && callBackFunc != NULL) callBackFunc(callBackObj, "Loading Lightsource...");
					if (matches == 11) {
						AddLight(vec1, vec2, brightness, red, green, blue, sectorIndex);
					} else {fclose(file); cout << "Bad Parse for light!" << endl; return BAD_PARSE;}
				break;

				
				case 'L':
					int index1, index2;
					matches = sscanf(buf, "LINK %d %d %d\n",&index1, &index2, &windowIndex);
					if(callBackObj != NULL && callBackFunc != NULL) callBackFunc(callBackObj, "Loading Link...");
					if (matches == 3) {
						LinkSector(index1, index2, windowIndex);
					} else {fclose(file); cout << "Bad Parse for LINK!" << endl;  return BAD_PARSE;}
				break;

				case 'P':
					matches = sscanf(buf, "PLANE %f, %f, %f %f, %f, %f %f, %f, %f %s %s %d %d %d %d %f %f %d",
										&vec1.x, &vec1.y, &vec1.z, &vec2.x, &vec2.y, &vec2.z, &vec3.x, &vec3.y, &vec3.z,
										decalName, bumpName, &shadowTrue, &diffuseTrue, &specTrue, &attenuateTrue, &bscale, &tileFactor, &sectorIndex);
					if(callBackObj != NULL && callBackFunc != NULL) callBackFunc(callBackObj, "Loading Plane...");
					if(matches == 18) {
						TPlane *newPlane = new TPlane(curInfrastructure);
						newPlane->SetTileFactor(tileFactor);
						newPlane->SetSizeFactor(vec2.x, vec2.y, vec2.z);
						newPlane->LoadGeom(NULL);
						newPlane->LoadTexture(decalName, bumpName, true, bscale);
						newPlane->PrepareTransform();
							glTranslatef(vec1.x, vec1.y, vec1.z);
							glRotatef(vec3.z, 0.0, 0.0, 1.0f);
							glRotatef(vec3.y, 0.0, 1.0f, 0.0);
							glRotatef(vec3.x, 1.0f, 0.0, 0.0);
						newPlane->EndTransform();
						newPlane->ApplyTransformPermanent();

						newPlane->PrepareTransform();
							glLoadIdentity();
						newPlane->EndTransform();
						
						newPlane->SaveMatrix();
						LoadedTObjects.push_back(newPlane);
						AttachObject(LoadedTObjects[LoadedTObjects.size() - 1], vec1, vec2, vec3, shadowTrue, specTrue, diffuseTrue, attenuateTrue, sectorIndex, false, 0.0f);
					}	else {fclose(file); ; cout << "Bad Parse for PLANE!" << endl; return BAD_PARSE;}
				break;

				case 'R':
					matches = sscanf(buf, "REFLECTOR %f, %f, %f %f, %f, %f %f, %f, %f %s %s %f %d %d %d %f %f %d",
										&vec1.x, &vec1.y, &vec1.z, &vec2.x, &vec2.y, &vec2.z, &vec3.x, &vec3.y, &vec3.z,
										decalName, bumpName, &alphaval, &diffuseTrue, &specTrue, &attenuateTrue, &bscale, &tileFactor, &sectorIndex);
					if(callBackObj != NULL && callBackFunc != NULL) callBackFunc(callBackObj, "Loading Reflector...");
					if(matches == 18) {
						TPlane *newPlane = new TPlane(curInfrastructure);
						newPlane->SetTileFactor(tileFactor);
						newPlane->SetSizeFactor(vec2.x, vec2.y, vec2.z);
						newPlane->LoadGeom(NULL);
						newPlane->LoadTexture(decalName, bumpName, true, bscale);
						newPlane->PrepareTransform();
							glTranslatef(vec1.x, vec1.y, vec1.z);
							glRotatef(vec3.z, 0.0, 0.0, 1.0f);
							glRotatef(vec3.y, 0.0, 1.0f, 0.0);
							glRotatef(vec3.x, 1.0f, 0.0, 0.0);
						newPlane->EndTransform();
						newPlane->ApplyTransformPermanent();

						newPlane->PrepareTransform();
							glLoadIdentity();
						newPlane->EndTransform();

						newPlane->SaveMatrix();
						LoadedTObjects.push_back(newPlane);
						AttachObject(LoadedTObjects[LoadedTObjects.size() - 1], vec1, vec2, vec3, shadowTrue, specTrue, diffuseTrue, attenuateTrue, sectorIndex, true, alphaval);
					}	else {fclose(file); ; cout << "Bad Parse for REFLECTOR!" << endl; return BAD_PARSE;}
				break;
				
				case 'M':
					matches = sscanf(buf, "MD2 %s %f, %f, %f %f, %f, %f %f, %f, %f %s %s %d %d %d %d %f %d %d %d", name,
										&vec1.x, &vec1.y, &vec1.z, &vec2.x, &vec2.y, &vec2.z, &vec3.x, &vec3.y, &vec3.z,
										decalName, bumpName, &shadowTrue, &diffuseTrue, &specTrue, &attenuateTrue, &bscale, &animation, &smoothness, &flipnormal);
					if(callBackObj != NULL && callBackFunc != NULL) callBackFunc(callBackObj, "Loading Model...");
					if(matches == 20) {
						TMd2 *newMd2 = new TMd2(curInfrastructure);
						newMd2->SetLoadFlippedNormals(flipnormal);

						if(LoadedMd2(name) != NULL) newMd2->Copy(*(LoadedMd2(name))); 
						else newMd2->LoadGeom(name);
						newMd2->LoadTexture(decalName, bumpName, true, bscale);
						newMd2->PrepareTransform();
							glLoadIdentity();
							glTranslatef(vec1.x, vec1.y, vec1.z);
							glRotatef(vec3.z, 0.0, 0.0, 1.0f);
							glRotatef(vec3.y, 0.0, 1.0f, 0.0);
							glRotatef(vec3.x, 1.0f, 0.0, 0.0);
							glScalef(vec2.x, vec2.y, vec2.z);
						newMd2->EndTransform();
						newMd2->SetCurrentAnimation(animation, smoothness);
						newMd2->PlayAnimation();
						newMd2->SaveMatrix();
						newMd2->CalcBounds();
						LoadedTObjects.push_back(newMd2);
						AttachObject(LoadedTObjects[LoadedTObjects.size() - 1], vec1, vec2, vec3, shadowTrue, specTrue, diffuseTrue, attenuateTrue);
					}	else {fclose(file); ; cout << "Bad Parse for MD2! Number of matches: " << matches << endl; return BAD_PARSE;}
				break;
			
				case 'G':
					matches = sscanf(buf, "GAMEPILLAR %s %f, %f, %f %f, %f, %f %f, %f, %f %s %s %d %d %d %d %f %d %d %d", name,
										&vec1.x, &vec1.y, &vec1.z, &vec2.x, &vec2.y, &vec2.z, &vec3.x, &vec3.y, &vec3.z,
										decalName, bumpName, &shadowTrue, &diffuseTrue, &specTrue, &attenuateTrue, &bscale, &animation, &smoothness, &flipnormal);
					if(callBackObj != NULL && callBackFunc != NULL) callBackFunc(callBackObj, "Loading Model...");
					if(matches == 20) {
						pillarCnt++;
						TMd2 *newMd2 = new TMd2(curInfrastructure);
						newMd2->SetLoadFlippedNormals(flipnormal);

						if(LoadedMd2(name) != NULL) newMd2->Copy(*(LoadedMd2(name))); 
						else newMd2->LoadGeom(name);
						newMd2->LoadTexture(decalName, bumpName, true, bscale);
						newMd2->PrepareTransform();
							glLoadIdentity();
							glTranslatef(vec1.x, vec1.y, vec1.z);
							glRotatef(vec3.z, 0.0, 0.0, 1.0f);
							glRotatef(vec3.y, 0.0, 1.0f, 0.0);
							glRotatef(vec3.x, 1.0f, 0.0, 0.0);
							glScalef(vec2.x, vec2.y, vec2.z);
						newMd2->EndTransform();
						newMd2->SetCurrentAnimation(animation, smoothness);
						newMd2->PlayAnimation();
						newMd2->SaveMatrix();
						newMd2->CalcBounds();
						LoadedTObjects.push_back(newMd2);
						AttachObject(LoadedTObjects[LoadedTObjects.size() - 1], vec1, vec2, vec3, shadowTrue, specTrue, diffuseTrue, attenuateTrue);
						ObjectList[ObjectList.size() - 1].gamePacket.type = GAME_PILLAR_TYPE;
						ObjectList[ObjectList.size() - 1].gamePacket.health = 1000;
						ObjectList[ObjectList.size() - 1].gamePacket.team = 1;
						
						
					}	else {fclose(file); ; cout << "Bad Parse for GAMEPILLAR! Number of matches: " << matches << endl; return BAD_PARSE;}
				break;
			}
		}
	}
	
	spawnPt.y = getMinSpawnY();
	cout << "Min Spawn Y: " << spawnPt.y << endl;
	
	return true;

}

int TScene::SaveSceneFile(char *filename) {
	FILE *file;
	file = fopen(filename, "w");

	if (file == NULL) {
		cout << "Could not save " << filename << "." << endl;
		return SAVE_ERROR;
  }
	fprintf(file, "#Scene File Description for Troodon Engine\n");

	//write out the sectors
	fprintf(file, "#Define the Sectors\n");
	for(int i = 0; i < PortalList.size(); i++) {
		fprintf(file, "SECTOR %f, %f, %f %f, %f, %f\n", PortalList[i].Bound1.x, PortalList[i].Bound1.y, PortalList[i].Bound1.z,
					 PortalList[i].Bound2.x,  PortalList[i].Bound2.y,  PortalList[i].Bound2.z);
	}

	fprintf(file, "\n#Define the Windows\n");
	//write out the windows
	for(i = 0; i < PortalWindowList.size(); i++) {
		fprintf(file, "WINDOW %f, %f, %f %f, %f, %f\n", PortalWindowList[i].Bounds1.x, PortalWindowList[i].Bounds1.y, PortalWindowList[i].Bounds1.z,
					 PortalWindowList[i].Bounds2.x,  PortalWindowList[i].Bounds2.y,  PortalWindowList[i].Bounds2.z);
	}

	fprintf(file, "\n#Define the Links\n");
	//write out the links
	for(i = 0; i < PortalWindowList.size(); i++) {
		fprintf(file, "LINK %d %d %d\n", PortalWindowList[i].index1, PortalWindowList[i].index2, i);
	}

	//write out lights
	fprintf(file, "\n#Define the lights\n");
	for(i = 0; i < LightList.size(); i++) {
		fprintf(file, "light %f, %f, %f %f, %f, %f %f %f, %f, %f %d\n",LightList[i].LightVec.x, LightList[i].LightVec.y, LightList[i].LightVec.z, LightList[i].EyeVec.x, LightList[i].EyeVec.y, LightList[i].EyeVec.z,
					LightList[i].brightness, LightList[i].r, LightList[i].g, LightList[i].b, LightList[i].sectorIndex);
	}

	//write out the objects
	fprintf(file, "\n#Define the Objects\n");
	for(i = 0; i < ObjectList.size(); i++) {
		if(ObjectList[i].curObject->GetType() == MD2_TYPE) {
			TMd2 *curMd2 = (TMd2 *)ObjectList[i].curObject;
			
			fprintf(file, "MD2 %s %f, %f, %f %f, %f, %f %f, %f, %f %s %s %d %d %d %d %f %d %d %d\n", curMd2->GetFileName().c_str(),
								ObjectList[i].translate.x, ObjectList[i].translate.y, ObjectList[i].translate.z, ObjectList[i].scale.x, ObjectList[i].scale.y, ObjectList[i].scale.z, ObjectList[i].rotate.x, ObjectList[i].rotate.y, ObjectList[i].rotate.z,
								ObjectList[i].curObject->GetDecalName().c_str(), ObjectList[i].curObject->GetBumpName().c_str(), ObjectList[i].shadow, ObjectList[i].diffuse, ObjectList[i].specular, ObjectList[i].attenuate, curMd2->GetBumpScale(), curMd2->GetAnimationIndex(), curMd2->GetSmoothness(), curMd2->GetFlipNormals());
		}
		else if(ObjectList[i].curObject->GetType() == PLANE_TYPE) {
			TPlane *curPlane = (TPlane *)ObjectList[i].curObject;
			if(!ObjectList[i].reflect) {
				fprintf(file, "PLANE %f, %f, %f %f, %f, %f %f, %f, %f %s %s %d %d %d %d %f %f %d\n", 
									ObjectList[i].translate.x, ObjectList[i].translate.y, ObjectList[i].translate.z, ObjectList[i].scale.x, ObjectList[i].scale.y, ObjectList[i].scale.z, ObjectList[i].rotate.x, ObjectList[i].rotate.y, ObjectList[i].rotate.z,
									ObjectList[i].curObject->GetDecalName().c_str(), ObjectList[i].curObject->GetBumpName().c_str(), ObjectList[i].shadow, ObjectList[i].diffuse, ObjectList[i].specular, ObjectList[i].attenuate, ObjectList[i].curObject->GetBumpScale(),
									curPlane->GetTileFactor(), ObjectList[i].sectorIndex);
			}
			else {
				fprintf(file, "REFLECTOR %f, %f, %f %f, %f, %f %f, %f, %f %s %s %f %d %d %d %f %f %d\n",
									ObjectList[i].translate.x, ObjectList[i].translate.y, ObjectList[i].translate.z, ObjectList[i].scale.x, ObjectList[i].scale.y, ObjectList[i].scale.z, ObjectList[i].rotate.x, ObjectList[i].rotate.y, ObjectList[i].rotate.z,
									ObjectList[i].curObject->GetDecalName().c_str(), ObjectList[i].curObject->GetBumpName().c_str(), ObjectList[i].alpha, ObjectList[i].diffuse, ObjectList[i].specular, ObjectList[i].attenuate, ObjectList[i].curObject->GetBumpScale(),
									curPlane->GetTileFactor(), ObjectList[i].sectorIndex);
			}
		}
	}
	                                                                                                                                                        
	return true;
}


TMd2* TScene::LoadedMd2(char *filename) {
	string cmpName = filename;
	for(int i = 0; i < LoadedTObjects.size(); i++) {
		if(LoadedTObjects[i] != NULL) {
			if(LoadedTObjects[i]->GetType() == MD2_TYPE) {
				TMd2 *curMd2 = (TMd2 *)LoadedTObjects[i];
				if(cmpName == curMd2->GetFileName()) return curMd2;
			}
		}
	}

	return NULL;
}

void TScene::RestoreSceneObjectsTransform() {
	for(int i = 0; i < LoadedTObjects.size(); i++) {
		if(LoadedTObjects[i] != NULL) LoadedTObjects[i]->RestoreMatrix();
	}
}

void TScene::ApplyMd2PhysicsTransform() {
	for(int i = 0; i < ObjectList.size(); i++) {
		TObject *curObject = ObjectList[i].curObject;
		if(curObject->GetType() == MD2_TYPE) {
			curObject->PrepareTransform();
				glLoadIdentity();
				glTranslatef(ObjectList[i].translate.x, ObjectList[i].translate.y, ObjectList[i].translate.z);
				glRotatef(ObjectList[i].rotate.z, 0.0, 0.0, 1.0f);
				glRotatef(ObjectList[i].rotate.y, 0.0, 1.0f, 0.0);
				glRotatef(ObjectList[i].rotate.x, 1.0f, 0.0, 0.0);
				glScalef(ObjectList[i].scale.x, ObjectList[i].scale.y, ObjectList[i].scale.z);
			curObject->EndTransform();
		}
	}
}


int TScene::WithinBounds(Vector *srcBound1, Vector *srcBound2, Vector *dstBound1, Vector *dstBound2) {
	Vector MaxBound, MinBound;

	if(dstBound1->x > dstBound2->x) {MaxBound.x = dstBound1->x; MinBound.x = dstBound2->x; }
	else { MaxBound.x = dstBound2->x; MinBound.x = dstBound1->x; }

	if(dstBound1->y > dstBound2->y) {MaxBound.y = dstBound1->y; MinBound.y = dstBound2->y; }
	else { MaxBound.y = dstBound2->y; MinBound.y = dstBound1->y; }

	if(dstBound1->z > dstBound2->z) {MaxBound.z = dstBound1->z; MinBound.z = dstBound2->z; }
	else { MaxBound.z = dstBound2->z; MinBound.z = dstBound1->z; }

	if(srcBound1->x >= MinBound.x && srcBound1->x <= MaxBound.x &&
	srcBound1->y >= MinBound.y && srcBound1->y <= MaxBound.y &&
	srcBound1->z >= MinBound.z && srcBound1->z <= MaxBound.z) return true;

	if(srcBound2->x >= MinBound.x && srcBound2->x <= MaxBound.x &&
	srcBound2->y >= MinBound.y && srcBound2->y <= MaxBound.y &&
	srcBound2->z >= MinBound.z && srcBound2->z <= MaxBound.z) return true;

	return false;
}

void TScene::RecalcSectorPos() {
	for(int i = 0; i < ObjectList.size(); i++) {
		if(ObjectList[i].sectorLocked == -1) ObjectList[i].sectorIndex = -1;
	}
	
	for(i = 0; i < PortalList.size(); i++) {
		for(int j = 0; j < ObjectList.size(); j++) {
			if(ObjectList[j].sectorIndex == -1) {
				if(WithinBounds(&ObjectList[j].curObject->GetMaxBounds(), &ObjectList[j].curObject->GetMinBounds(), &PortalList[i].Bound1, &PortalList[i].Bound2)) {
					ObjectList[j].sectorIndex = i;
				}
			}
		}
	}

	//currentLightVec.x = viewerLoc.x;
	//currentLightVec.z = viewerLoc.z;
}

void TScene::MoveViewerDelta(GLfloat dispX, GLfloat dispY, GLfloat dispZ, GLfloat rotX, GLfloat rotY, GLfloat rotZ) {
	PrepareViewerTransform();
		glRotatef(rotX, 1.0f, 0.0f, 0.0f);
		glRotatef(rotY, 0.0f, 1.0f, 0.0f);
		glRotatef(rotZ, 0.0f, 0.0f, 1.0f);
	EndViewerTransform();

	Vector defaultVec, defaultVec2;
	defaultVec.x = dispX;
	defaultVec.y = dispY;
	defaultVec.z = dispZ;
	transformVector(&defaultVec2, &defaultVec, this->viewerMatrix);

	Vadd(&viewerLoc, &defaultVec2, &viewerLoc);

	PrepareViewerTransform();
		glLoadIdentity();
		glTranslatef(viewerLoc.x, viewerLoc.y, viewerLoc.z);
		glRotatef(rotX, 1.0f, 0.0f, 0.0f);
		glRotatef(rotY, 0.0f, 1.0f, 0.0f);
		glRotatef(rotZ, 0.0f, 0.0f, 1.0f);
	EndViewerTransform();
}

void TScene::MoveViewerDelta2(GLfloat dispX, GLfloat dispY, GLfloat dispZ, GLfloat rotX, GLfloat rotY, GLfloat rotZ) {

	Vector defaultVec2;
	defaultVec2.x = dispX;
	defaultVec2.y = dispY;
	defaultVec2.z = dispZ;

	Vadd(&viewerLoc, &defaultVec2, &viewerLoc);

	PrepareViewerTransform();
		glLoadIdentity();
		glTranslatef(viewerLoc.x, viewerLoc.y, viewerLoc.z);
		glRotatef(rotX, 1.0f, 0.0f, 0.0f);
		glRotatef(rotY, 0.0f, 1.0f, 0.0f);
		glRotatef(rotZ, 0.0f, 0.0f, 1.0f);
	EndViewerTransform();
}

Vector TScene::GetViewerMoveDelta(GLfloat dispX, GLfloat dispY, GLfloat dispZ, GLfloat rotX, GLfloat rotY, GLfloat rotZ) {
	GLfloat curMatrix[16][16];
	
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
		glRotatef(rotY, 0.0f, 1.0f, 0.0f);
		glRotatef(rotX, 1.0f, 0.0f, 0.0f);
		glRotatef(rotZ, 0.0f, 0.0f, 1.0f);
		glGetFloatv(GL_MODELVIEW_MATRIX, &curMatrix[0][0]);
	glPopMatrix();
	
	Vector defaultVec, defaultVec2;
	defaultVec.x = dispX;
	defaultVec.y = dispY;
	defaultVec.z = dispZ;
	transformVector(&defaultVec2, &defaultVec, &curMatrix[0][0]);
	
	return defaultVec2;
}


int TScene::WithinRect(GLdouble srcX, GLdouble srcY, GLdouble dstX1, GLdouble dstY1, GLdouble dstX2, GLdouble dstY2) {
	GLdouble minX, minY, maxX, maxY;

	if(dstX2 > dstX1) { maxX = dstX2; minX = dstX1; } else { maxX = dstX1, minX = dstX2; }
	if(dstY2 > dstY1) { maxY = dstY2; minY = dstY1; } else { maxY = dstY1, minX = dstY2; }

	if(srcX >= minX && srcX <= maxX && srcY >= minY && srcY <= maxY) return true;

	return false;
}

int TScene::RectIntersectRect(GLdouble srcX1, GLdouble srcY1, GLdouble srcX2, GLdouble srcY2, GLdouble dstX1, GLdouble dstY1, GLdouble dstX2, GLdouble dstY2) {
	GLdouble newClipX1 = srcX1, newClipY1 = srcY1, newClipX2 = srcX2, newClipY2 = srcY2;
	ClipRect2D(&newClipX1, &newClipY1, &newClipX2, &newClipY2, dstX1, dstY1, dstX2, dstY2);

	if(newClipX1 == newClipX2 || newClipY1 == newClipY2) return false;
	
	return true;
}

void TScene::ClipRect2D(GLdouble *x1, GLdouble *y1, GLdouble *x2, GLdouble *y2, GLdouble winX1, GLdouble winY1, GLdouble winX2, GLdouble winY2) {
	GLdouble minX, minY, maxX, maxY;

	if(winX2 > winX1) { maxX = winX2; minX = winX1; } else { maxX = winX1, minX = winX2; }
	if(winY2 > winY1) { maxY = winY2; minY = winY1; } else { maxY = winY1, minX = winY2; }

	if(*x1 < minX) {*x1 = minX;} if(*x1 > maxX) {*x1 = maxX; }
	if(*y1 < minY) {*y1 = minY;} if(*y1 > maxY) {*y1 = maxY; }

	if(*x2 < minX) {*x2 = minX;} if(*x2 > maxX) {*x2 = maxX; }
	if(*y2 < minY) {*y2 = minY;} if(*y2 > maxY) {*y2 = maxY; }
}

void TScene::ClipLine2D(Vector *Pnt1, Vector *Pnt2) {
	GLfloat newX = 0.0, newY = 0.0;
	if(Pnt1->z > -1.0f) {
		newX = (Pnt2->x - Pnt1->x) / fabs(Pnt2->z- Pnt1->z) * (-1.0f - Pnt1->z)  + Pnt1->x;
		newY = (Pnt2->y - Pnt1->y) / fabs(Pnt2->z - Pnt1->z) * (-1.0f - Pnt1->z) + Pnt1->y;
		Pnt1->x = newX;
		Pnt1->y = newY;
		Pnt1->z = -1.0;
	}

	return;

	if(Pnt2->z > -1.0f) {
		newX = (Pnt1->x - Pnt2->x) / fabs(Pnt2->z - Pnt1->z) * (-1.0f - Pnt2->z) + Pnt2->x;
		newY = (Pnt1->y - Pnt2->y) / fabs(Pnt2->z - Pnt1->z) * (-1.0f - Pnt2->z) + Pnt2->y;
		Pnt2->x = newX;
		Pnt2->y = newY;
		Pnt2->z = -1.0;
	}
	
}



//does portal culling by projecting the bounding volume into 2 space and then recursively culling the neighbouring sectors
void TScene::PortalCull() {
	//first determine which sector the viewer is in
	int curIndex = -1;
	for(int i = 0; i < PortalList.size(); i++) {
		if(WithinBounds(&viewerLoc, &viewerLoc, &PortalList[i].Bound1, &PortalList[i].Bound2)) {
			curIndex = i;
			break;
		}
	}
	
	for(i = 0; i < ObjectList.size(); i++) {
		ObjectList[i].winX1 = -1; ObjectList[i].winY1 = -1; ObjectList[i].winX2 = -1; ObjectList[i].winY2 = -1;
		if(ObjectList[i].sectorIndex == curIndex) ObjectList[i].visible = true;
	}

	if(curIndex == -1) {curSectorIndex = -1; return;}  //no culling since we are out of every portal
	curSectorIndex = curIndex;

	//now do portal culling
	SectorCull(curIndex, 0, 0, this->curInfrastructure->GetWinWidth(), this->curInfrastructure->GetWinHeight() );
	
}

void TScene::GetProjectionCoords(TObject *curObject, GLdouble *minX, GLdouble *minY, GLdouble *maxX, GLdouble *maxY, GLdouble *modelMatrix, GLdouble *projMatrix, GLint *viewport) {
	GLdouble screenX[8];
	GLdouble screenY[8], screenZ;

	Vector *curVec = curObject->GetDstBounds1();
	gluProject((GLdouble)curVec->x, (GLdouble)curVec->y, (GLdouble)curVec->z, modelMatrix, projMatrix,
						viewport, &screenX[0], &screenY[0], &screenZ);
	
	curVec = curObject->GetDstBounds2();
	gluProject((GLdouble)curVec->x, (GLdouble)curVec->y, (GLdouble)curVec->z, modelMatrix, projMatrix,
						viewport, &screenX[1], &screenY[1], &screenZ);

	curVec = curObject->GetDstBounds3();
	gluProject((GLdouble)curVec->x, (GLdouble)curVec->y, (GLdouble)curVec->z, modelMatrix, projMatrix,
						viewport, &screenX[2], &screenY[2], &screenZ);

	curVec = curObject->GetDstBounds4();
	gluProject((GLdouble)curVec->x, (GLdouble)curVec->y, (GLdouble)curVec->z, modelMatrix, projMatrix,
						viewport, &screenX[3], &screenY[3], &screenZ);

	curVec = curObject->GetDstBounds5();
	gluProject((GLdouble)curVec->x, (GLdouble)curVec->y, (GLdouble)curVec->z, modelMatrix, projMatrix,
						viewport, &screenX[4], &screenY[4], &screenZ);

	curVec = curObject->GetDstBounds6();
	gluProject((GLdouble)curVec->x, (GLdouble)curVec->y, (GLdouble)curVec->z, modelMatrix, projMatrix,
						viewport, &screenX[5], &screenY[5], &screenZ);

	curVec = curObject->GetDstBounds7();
	gluProject((GLdouble)curVec->x, (GLdouble)curVec->y, (GLdouble)curVec->z, modelMatrix, projMatrix,
						viewport, &screenX[6], &screenY[6], &screenZ);

	curVec = curObject->GetDstBounds8();
	gluProject((GLdouble)curVec->x, (GLdouble)curVec->y, (GLdouble)curVec->z, modelMatrix, projMatrix,
						viewport, &screenX[7], &screenY[7], &screenZ);

	*minX = screenX[0];
	*minY = screenY[0];
	*maxX = screenX[0];
	*maxY = screenY[0];

	for(int i = 0; i < 8; i++) {
		if(screenX[i] > *maxX) *maxX = screenX[i];
		if(screenY[i] > *maxY) *maxY = screenY[i];

		if(screenX[i] < *minX) *minX = screenX[i];
		if(screenY[i] < *minY) *minY = screenY[i];
	}
}

void TScene::GetWindowProjectionCoords(PortalWindow *curWindow, GLdouble *minX, GLdouble *minY, GLdouble *maxX, GLdouble *maxY, GLdouble *modelMatrix, GLdouble *projMatrix, GLint *viewport) {
	GLdouble screenX[4];
	GLdouble screenY[4], screenZ;

	Vector *curVec = &curWindow->DstBounds1;
	gluProject((GLdouble)curVec->x, (GLdouble)curVec->y, (GLdouble)curVec->z, modelMatrix, projMatrix,
						viewport, &screenX[0], &screenY[0], &screenZ);

	curVec = &curWindow->DstBounds2;
	gluProject((GLdouble)curVec->x, (GLdouble)curVec->y, (GLdouble)curVec->z, modelMatrix, projMatrix,
						viewport, &screenX[1], &screenY[1], &screenZ);

	curVec = &curWindow->DstBounds3;
	gluProject((GLdouble)curVec->x, (GLdouble)curVec->y, (GLdouble)curVec->z, modelMatrix, projMatrix,
						viewport, &screenX[2], &screenY[2], &screenZ);

	curVec = &curWindow->DstBounds4;
	gluProject((GLdouble)curVec->x, (GLdouble)curVec->y, (GLdouble)curVec->z, modelMatrix, projMatrix,
						viewport, &screenX[3], &screenY[3], &screenZ);

	*minX = screenX[0];
	*minY = screenY[0];
	*maxX = screenX[0];
	*maxY = screenY[0];

	for(int i = 0; i < 4; i++) {
		if(screenX[i] > *maxX) *maxX = screenX[i];
		if(screenY[i] > *maxY) *maxY = screenY[i];

		if(screenX[i] < *minX) *minX = screenX[i];
		if(screenY[i] < *minY) *minY = screenY[i];
	}
}



void TScene::SectorCull(int index, GLdouble winX1, GLdouble winY1, GLdouble winX2, GLdouble winY2) {
	GLint viewport[4];
	GLdouble modelMatrix[16];
	GLdouble projMatrix[16];
	GLdouble screenX1, screenY1, screenZ1, screenX2, screenY2, screenZ2;
	GLdouble newWinX1, newWinY1, newWinZ1, newWinX2, newWinY2, newWinZ2;
	glGetIntegerv(GL_VIEWPORT, viewport);
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
	glGetDoublev(GL_MODELVIEW_MATRIX, modelMatrix);
	glGetDoublev(GL_PROJECTION_MATRIX, projMatrix);
	glPopMatrix();
		
	PortalObj *curPortal = &PortalList[index];
	if(curPortal->processed == true) return;
	//cull objects in this sector first
	if(curSectorIndex != index) {
		for(int i = 0; i < ObjectList.size(); i++) {
			if(ObjectList[i].sectorIndex == index) { //see if object is visible within this sector
				bool tossClip = false;
				Vector *bnd1 = ObjectList[i].curObject->GetDstBounds1();
				Vector *bnd2 = ObjectList[i].curObject->GetDstBounds2();
				Vector *bnd3 = ObjectList[i].curObject->GetDstBounds3();
				Vector *bnd4 = ObjectList[i].curObject->GetDstBounds4();
				Vector *bnd5 = ObjectList[i].curObject->GetDstBounds5();
				Vector *bnd6 = ObjectList[i].curObject->GetDstBounds6();
				Vector *bnd7 = ObjectList[i].curObject->GetDstBounds7();
				Vector *bnd8 = ObjectList[i].curObject->GetDstBounds8();
				
				ObjectList[i].winX1 = (int)winX1; ObjectList[i].winY1 = (int)winY1; ObjectList[i].winX2 = (int)winX2; ObjectList[i].winY2 = (int)winY2;
				
				if(bnd1->z > -1.0f || bnd2->z > -1.0f || bnd3->z > -1.0f || bnd4->z  > -1.0f || bnd5->z  > -1.0f || bnd6->z  > -1.0f || bnd7->z  > -1.0f 
					|| bnd8->z > -1.0f) tossClip =true;

				if(!tossClip) {
					GetProjectionCoords(ObjectList[i].curObject, &screenX1, &screenY1, &screenX2, &screenY2, modelMatrix, projMatrix, viewport);
					if(RectIntersectRect(screenX1, screenY1, screenX2, screenY2, winX1, winY1, winX2, winY2)) ObjectList[i].visible = true;
//					else { ObjectList[i].visible = false; }
				}
				else { ObjectList[i].visible = true; }
			
			}
		}
	}


	//recursively cull the neighbouring sectors
	curPortal->processed = true;
	for(int i = 0; i < curPortal->link.size(); i++) { // go through every sector that is linked to this current sector
		for(int j = 0; j < PortalWindowList.size(); j++) {
			int index1 = PortalWindowList[j].index1;
			int index2 = PortalWindowList[j].index2;

			if( (curPortal->link[i] == index1 && index == index2) || (curPortal->link[i] == index2 && index == index1) ) { //matching window
				Vector *wbnd1 = &PortalWindowList[j].DstBounds1;
				Vector *wbnd2 =  &PortalWindowList[j].DstBounds2;
				Vector *wbnd3 =  &PortalWindowList[j].DstBounds3;
				Vector *wbnd4 =  &PortalWindowList[j].DstBounds4;
				bool tossClip = false;
				if(wbnd1->z > -1.0f || wbnd2->z > -1.0f || wbnd3->z > -1.0f || wbnd4->z > -1.0f) tossClip = true;
					GetWindowProjectionCoords(&PortalWindowList[j], &newWinX1, &newWinY1, &newWinX2, &newWinY2,
																			modelMatrix, projMatrix, viewport);
					ClipRect2D(&newWinX1, &newWinY1, &newWinX2, &newWinY2, winX1, winY1, winX2, winY2);
				if(!tossClip) SectorCull(curPortal->link[i], newWinX1, newWinY1, newWinX2, newWinY2);
				else SectorCull(curPortal->link[i], winX1, winY1, winX2, winY2);
			}
		}
	}
	curPortal->processed = false;
}

//test to see if 2 OBB's intersect with one another using the separating axis theorem
bool TScene::OBBIntersect(TObject *obj1, TObject *obj2) {
	Vector A_base[3], B_base[3];
	float a_ext[3], b_ext[3];
	Vector a_midpnt, b_midpnt;
	
	//find the midpoint of both boxes
	if(obj1 != NULL) {
		Vector MinBounds = obj1->GetMinBounds(), MaxBounds = obj1->GetMaxBounds();
		a_midpnt.x = (MinBounds.x + MaxBounds.x) * 0.5f;
		a_midpnt.y = (MinBounds.y + MaxBounds.y) * 0.5f;
		a_midpnt.z = (MinBounds.z + MaxBounds.z) * 0.5f;
	}
	else {
		a_midpnt.x = (dstViewerBounds[0].x + dstViewerBounds[6].x) * 0.5f;
		a_midpnt.y = (dstViewerBounds[0].y + dstViewerBounds[6].y) * 0.5f;
		a_midpnt.z = (dstViewerBounds[0].z + dstViewerBounds[6].z) * 0.5f;
	}
	
	Vector MinBounds = obj2->GetMinBounds(), MaxBounds = obj2->GetMaxBounds();
	Vector PlaneNormal;
	if(obj2->GetType() == PLANE_TYPE) {
		TPlane *curPlane = (TPlane *)obj2;
		PlaneNormal = curPlane->GetNormal();
		Vscale(&PlaneNormal, 8.0f);
		Vadd(&MaxBounds, &PlaneNormal, &MaxBounds);
		Vscale(&PlaneNormal, -1.0f);
		Vadd(&MinBounds, &PlaneNormal, &MinBounds);
		Vscale(&PlaneNormal, -1.0f);
	}
	
	b_midpnt.x = (MinBounds.x + MaxBounds.x) * 0.5f;
	b_midpnt.y = (MinBounds.y + MaxBounds.y) * 0.5f;
	b_midpnt.z = (MinBounds.z + MaxBounds.z) * 0.5f;
	
	//find the orthonormal basis of object A and B and extents of A and B
	if(obj1 != NULL) {
		Vsub(obj1->GetDstBounds3(), obj1->GetDstBounds1(), &A_base[0]);
		Vsub(obj1->GetDstBounds8(), obj1->GetDstBounds1(), &A_base[1]);
		Vsub(obj1->GetDstBounds5(), obj1->GetDstBounds1(), &A_base[2]);
	}
	else {
		Vsub(&dstViewerBounds[1], &dstViewerBounds[0], &A_base[0]);
		Vsub(&dstViewerBounds[3], &dstViewerBounds[0], &A_base[2]);
		Vsub(&dstViewerBounds[4], &dstViewerBounds[0], &A_base[1]);
	}
	
	Vsub(obj2->GetDstBounds3(), obj2->GetDstBounds1(), &B_base[0]);
	Vsub(obj2->GetDstBounds8(), obj2->GetDstBounds1(), &B_base[1]);
	Vsub(obj2->GetDstBounds5(), obj2->GetDstBounds1(), &B_base[2]);

	if(obj2->GetType() == PLANE_TYPE) {
		for(int p = 0; p < 3; p++) {
			if(Vlength(&B_base[p]) < 1) {
				B_base[p] = PlaneNormal;
				if(fabs(B_base[p].x ) < 1) B_base[p].x = 0.0f;
				if(fabs(B_base[p].y ) < 1) B_base[p].y = 0.0f;
				if(fabs(B_base[p].z ) < 1) B_base[p].z = 0.0f;
				break;
			}
		}
		
		/*Vector subVec;
		Vsub(&B_base[0], &B_base[1], &subVec);
		
		if(Vlength(&subVec) < 1) {
			B_base[0] = PlaneNormal;
		}
	
		Vsub(&B_base[1], &B_base[2], &subVec);
		if(Vlength(&subVec) < 1) {
			B_base[1] = PlaneNormal;
		}*/
		/*printVector(&B_base[0], "Base0:");
		printVector(&B_base[1], "Base1:");
		printVector(&B_base[2], "Base2:");
		printVector(&b_midpnt, "MidPoint:");
		printVector(&MinBounds, "MinPoint:");
		printVector(&MaxBounds, "MaxPoint:");
		printVector(&a_midpnt, "A's MidPoint:");
		printVector(&A_base[0], "A Base0:");
		printVector(&A_base[1], "A Base1:");
		printVector(&A_base[2], "A Base2:");*/
	}
	
	
	a_ext[0] = Vlength(&A_base[0]) * 0.5; 
	a_ext[1] = Vlength(&A_base[1]) * 0.5; 
	a_ext[2] = Vlength(&A_base[2]) * 0.5; 
	
	b_ext[0] = Vlength(&B_base[0]) * 0.5; 
	b_ext[1] = Vlength(&B_base[1]) * 0.5; 
	b_ext[2] = Vlength(&B_base[2]) * 0.5; 
	
	Vnormal(&A_base[0]);
	Vnormal(&A_base[1]);
	Vnormal(&A_base[2]);
	Vnormal(&B_base[0]);
	Vnormal(&B_base[1]);
	Vnormal(&B_base[2]);
	
	//translation in parent frame
	Vector v;
	float T[3];
	float R[3][3];
	Vsub(&b_midpnt, &a_midpnt, &v);
	float ra, rb, t;
	T[0] = Vdot(&v, &A_base[0]);
	T[1] = Vdot(&v, &A_base[1]);
	T[2] = Vdot(&v, &A_base[2]);
	
	//calculate rotation matrix R
	for(int m = 0; m < 3; m++) {
		for(int n = 0; n < 3; n++) {
			R[m][n] = Vdot(&A_base[m], &B_base[n]);
		}
	}
	
	//now use the separating axis theorem to do the intersection test for 15 potential separating axes
	//if a separating axis cannot be found, then the 2 boxes overlap
	
	//A's basis vectors
	for(int i = 0; i < 3; i++) {
 		ra = a_ext[i];
		rb = b_ext[0] * fabs(R[i][0]) + b_ext[1] * fabs(R[i][1]) + b_ext[2] * fabs(R[i][2]);
		t = fabs(T[i]);
		if(t > ra + rb) return false;
	}
	
	//B's basis vectors
	for(int k = 0; k < 3; k++) {
		rb = b_ext[k];
		ra = a_ext[0] * fabs(R[0][k]) + a_ext[1] * fabs(R[1][k]) + a_ext[2] * fabs(R[2][k]);
		t = fabs(T[0] *R[0][k] + T[1] * R[1][k] + T[2] * R[2][k]);
		if(t > ra + rb) return false;
	}
	
	//9 cross-product comparisons
	
	//L = A0 x B0
	ra = a_ext[1] * fabs(R[2][0]) + a_ext[2] * fabs(R[1][0]);
	rb = b_ext[1] * fabs(R[0][2]) + b_ext[2] * fabs(R[0][1]);
	t = fabs(T[2] * R[1][0] - T[1] * R[2][0]);
	
	if(t > ra + rb) return false;
	
	//L = A0 x B1
	ra = a_ext[1] * fabs(R[2][1]) + a_ext[2] * fabs(R[1][1]);
	rb = b_ext[0] * fabs(R[0][2]) + b_ext[2] * fabs(R[0][0]);
	t = fabs(T[2] * R[1][1] - T[1] * R[2][1]);
	
	if(t > ra + rb) return false;
	
	//L = A0 x B2
	ra = a_ext[1] * fabs(R[2][2]) + a_ext[2] * fabs(R[1][2]);
	rb = b_ext[0] * fabs(R[0][1]) + b_ext[1] * fabs(R[0][0]);
	t = fabs(T[2] * R[1][2] - T[1] * R[2][2]);
	
	if(t > ra + rb) return false;
	
	//L = A1 x B0
	ra = a_ext[0] * fabs(R[2][0]) + a_ext[2] * fabs(R[0][0]);
	rb = b_ext[1] * fabs(R[1][2]) + b_ext[2] * fabs(R[1][1]);
	t = fabs(T[0] * R[2][0] - T[2] * R[0][0]);
	
	if(t > ra + rb) return false;
	
	//L = A1 x B1
	ra = a_ext[0] * fabs(R[2][1]) + a_ext[2] * fabs(R[0][1]);
	rb = b_ext[0] * fabs(R[1][2]) + b_ext[2] * fabs(R[1][0]);
	t = fabs(T[0] * R[2][1] - T[2] * R[0][1]);
	
	if(t > ra + rb) return false;
	
	//L = A1 x B2
	ra = a_ext[0] * fabs(R[2][2]) + a_ext[2] * fabs(R[0][2]);
	rb = b_ext[0] * fabs(R[1][1]) + b_ext[1] * fabs(R[1][0]);
	t = fabs(T[0] * R[2][2] - T[2] * R[0][2]);
	
	if(t > ra + rb) return false;
	
	//L = A2 x B0
	ra = a_ext[0] * fabs(R[1][0]) + a_ext[1] * fabs(R[0][0]);
	rb = b_ext[1] * fabs(R[2][2]) + b_ext[2] * fabs(R[2][1]);
	t = fabs(T[1] * R[0][0] - T[0] * R[1][0]);
	
	if(t > ra + rb) return false;
	
	//L = A2 x B1
	ra = a_ext[0] * fabs(R[1][1]) + a_ext[1] * fabs(R[0][1]);
	rb = b_ext[0] * fabs(R[2][2]) + b_ext[2] * fabs(R[2][0]);
	t = fabs(T[1] * R[0][1] - T[0] * R[1][1]);
	
	if(t > ra + rb) return false;
	
	//L = A2 x B2
	ra = a_ext[0] * fabs(R[1][2]) + a_ext[1] * fabs(R[0][2]);
	rb = b_ext[0] * fabs(R[2][1]) + b_ext[1] * fabs(R[2][0]);
	t = fabs(T[1] * R[0][2] - T[0] * R[1][2]);
	
	if(t > ra + rb) return false;
	
	return true;
	
	
}

void TScene::SetViewerBounds(GLfloat width, GLfloat height, GLfloat depth) {
	viewerBounds[0].x = -width * 0.5;
	viewerBounds[0].y = -height * 0.5; 
	viewerBounds[0].z = -depth * 0.5;
	
	viewerBounds[1].x = width * 0.5;
	viewerBounds[1].y = -height * 0.5;
	viewerBounds[1].z = -depth * 0.5;

	viewerBounds[2].x = width * 0.5;
	viewerBounds[2].y = -height * 0.5;
	viewerBounds[2].z = depth * 0.5;

	viewerBounds[3].x = -width * 0.5;
	viewerBounds[3].y = -height * 0.5; 
	viewerBounds[3].z = depth * 0.5;

	viewerBounds[4].x = -width * 0.5;
	viewerBounds[4].y = height * 0.5; 
	viewerBounds[4].z = -depth * 0.5;
	
	viewerBounds[5].x = width * 0.5;
	viewerBounds[5].y = height * 0.5; 
	viewerBounds[5].z = -depth * 0.5;

	viewerBounds[6].x = width * 0.5;
	viewerBounds[6].y = height * 0.5; 
	viewerBounds[6].z = depth * 0.5;

	viewerBounds[7].x = -width * 0.5;
	viewerBounds[7].y = height * 0.5; 
	viewerBounds[7].z = depth * 0.5;
}


void TScene::UpdateViewerBounds() {
	for(int i = 0; i < 8; i++) {
//		transformVector(&dstViewerBounds[i], &viewerBounds[i], this->viewerMatrix);
		Vadd(&viewerBounds[i], &viewerLoc, &dstViewerBounds[i]);
	}
}

//attempt to move the viewer taking into account the objects that are in the scene. 
void TScene::TryMoveViewerDelta(GLfloat dispX, GLfloat dispY, GLfloat dispZ, GLfloat rotX, GLfloat rotY, GLfloat rotZ) {
	Vector orgLoc = viewerLoc;
	GLfloat orgViewerMatrix[16];
	memcpy(orgViewerMatrix, viewerMatrix, 16 * sizeof(GLfloat));
	
	MoveViewerDelta(dispX, dispY, dispZ, rotX, rotY, rotZ);
	
	//loop through every object in the scene and test for intersection 
	bool flag = false;
	for(int i = 0; i < ObjectList.size(); i++) {
		//if(ObjectList[i].curObject->GetType() == MD2_TYPE) {
			if(OBBIntersect(NULL, ObjectList[i].curObject)) {flag = true; break;}
		//}
		//else {
		//}
	}
	
	if(flag) {
		PrepareViewerTransform();
		glLoadMatrixf(orgViewerMatrix);
		EndViewerTransform();
		viewerLoc = orgLoc;
	}
}

void TScene::ApplyViewerAccel(GLfloat accelX, GLfloat accelY, GLfloat accelZ, GLfloat rotX, GLfloat rotY, GLfloat rotZ) {
	viewerPhysics.accel.x = accelX;
	viewerPhysics.accel.y = accelY;
	viewerPhysics.accel.z = accelZ;
	
	Vadd(&viewerPhysics.accel, &viewerPhysics.vel, &viewerPhysics.vel);
	
	if(viewerPhysics.vel.x > viewerPhysics.velCap.x) viewerPhysics.vel.x = viewerPhysics.velCap.x;
	if(viewerPhysics.vel.x < -fabs(viewerPhysics.velCap.x)) viewerPhysics.vel.x = -fabs(viewerPhysics.velCap.x);
	if(viewerPhysics.vel.y > viewerPhysics.velCap.y) viewerPhysics.vel.y = viewerPhysics.velCap.y;
	if(viewerPhysics.vel.y < -fabs(viewerPhysics.velCap.y)) viewerPhysics.vel.y = -fabs(viewerPhysics.velCap.y);
	if(viewerPhysics.vel.z > viewerPhysics.velCap.z) viewerPhysics.vel.z = viewerPhysics.velCap.z;
	if(viewerPhysics.vel.z < -fabs(viewerPhysics.velCap.z)) viewerPhysics.vel.z = -fabs(viewerPhysics.velCap.z);
	
	
	viewerPhysics.angle.x = rotX;
	viewerPhysics.angle.y = rotY;
	viewerPhysics.angle.z = rotZ;
}

void TScene::TryMoveViewerPhysics() {
	Vector orgLoc = viewerLoc;
	GLfloat orgViewerMatrix[16];
	memcpy(orgViewerMatrix, viewerMatrix, 16 * sizeof(GLfloat));
	
	MoveViewerDelta(viewerPhysics.vel.x, viewerPhysics.vel.y, viewerPhysics.vel.z, 
				viewerPhysics.angle.x, viewerPhysics.angle.y, viewerPhysics.angle.z);
	 
	viewerPhysics.resultVel = GetViewerMoveDelta(viewerPhysics.vel.x, viewerPhysics.vel.y, viewerPhysics.vel.z, 
				viewerPhysics.angle.x, viewerPhysics.angle.y, viewerPhysics.angle.z);
	
	//loop through every object in the scene and test for intersection 
	bool flag = false;
	for(int i = 0; i < ObjectList.size(); i++) {
		if(ObjectList[i].curObject->GetType() == MD2_TYPE && ObjectList[i].sectorIndex == curSectorIndex) {
			if(OBBIntersect(NULL, ObjectList[i].curObject)) {
				flag = true;
				Vector pNormal;
				Vsub(&GetViewerMidPnt(), &ObjectList[i].curObject->GetBndMidPnt(), &pNormal);
				Vnormal(&pNormal);
				
				float dotProj = Vdot(&viewerPhysics.resultVel, &pNormal);
				Vscale(&pNormal, -dotProj);
				Vector resultVec;
				Vadd(&viewerPhysics.resultVel, &pNormal, &resultVec);
				viewerPhysics.resultVel = resultVec;
			}
		}
	}
	
	for(i = 0; i < ObjectList.size(); i++) {
		if(ObjectList[i].curObject->GetType() == PLANE_TYPE) {
			if(OBBIntersect(NULL, ObjectList[i].curObject)) {
				flag = true; 
				TPlane *curPlane = (TPlane *)ObjectList[i].curObject;
				Vector pNormal = curPlane->GetNormal();
				
				float dotProj = (Vdot(&viewerPhysics.resultVel, &pNormal));
				Vscale(&pNormal, -dotProj);
				Vector resultVec;
				Vadd(&viewerPhysics.resultVel, &pNormal, &resultVec);
				viewerPhysics.resultVel = resultVec;
			}
		}
	}
	
	
	if(flag) {
		viewerLoc = orgLoc;
		PrepareViewerTransform();
		glLoadMatrixf(orgViewerMatrix);
		EndViewerTransform();
		MoveViewerDelta2(viewerPhysics.resultVel.x, viewerPhysics.resultVel.y, viewerPhysics.resultVel.z, 
				viewerPhysics.angle.x, viewerPhysics.angle.y, viewerPhysics.angle.z);
	}
	
}

Vector TScene::GetViewerMidPnt() {
	Vector midpnt;
	midpnt.x = (dstViewerBounds[0].x + dstViewerBounds[6].x) * 0.5f;
	midpnt.y = (dstViewerBounds[0].y + dstViewerBounds[6].y) * 0.5f;
	midpnt.z = (dstViewerBounds[0].z + dstViewerBounds[6].z) * 0.5f;
	
	return midpnt;
}

void TScene::StopViewerPhysics(bool xTrue, bool yTrue, bool zTrue) {
	if(xTrue) viewerPhysics.vel.x = 0.0f;
	if(yTrue) viewerPhysics.vel.y = 0.0f;
	if(zTrue) viewerPhysics.vel.z = 0.0f;
	viewerPhysics.resultVel = viewerPhysics.vel;
	viewerPhysics.accel = viewerPhysics.vel;
}

void TScene::RenderSkyBox(float scale) {
	GLfloat orientateMatrix[16];
	GLfloat inverseOrientateMatrix[16];
	
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
		glLoadIdentity();
		glRotatef(viewerPhysics.angle.y, 0.0f, 1.0f, 0.0f);
		glRotatef(viewerPhysics.angle.x, 1.0f, 0.0f, 0.0f);
		glRotatef(viewerPhysics.angle.z, 0.0f, 0.0f, 1.0f);
		
		glGetFloatv(GL_MODELVIEW_MATRIX, orientateMatrix);
		invertMatrixf(inverseOrientateMatrix, orientateMatrix);
	glPopMatrix();
	

	glDisable(GL_CULL_FACE );

	glActiveTextureARB( GL_TEXTURE1_ARB );
	glDisable( GL_TEXTURE_CUBE_MAP_ARB );
	glDisable( GL_TEXTURE_2D );

	glActiveTextureARB( GL_TEXTURE0_ARB );
	glDisable( GL_TEXTURE_CUBE_MAP_ARB );

	glDisable(GL_REGISTER_COMBINERS_NV);
	glEnable(GL_TEXTURE_2D);
	glDepthFunc(GL_LEQUAL);

	glDisable(GL_LIGHTING);
	
	glDisable(GL_STENCIL_TEST);
	glDisable(GL_BLEND);
	glDepthMask(GL_FALSE);
	glColor4f(1.0, 1.0, 1.0, 1.0);
	
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
		glLoadIdentity();
		glMultMatrixf(inverseOrientateMatrix);
		if(scale > 0) glScalef(scale, scale, scale);
		else glScalef(scale, -scale, -scale);
		
		glBindTexture(GL_TEXTURE_2D, curInfrastructure->getSkyBoxTexture(SKYBOX_BACK));
		glBegin(GL_QUADS);
			glTexCoord2f(1.0, 0.0);
			glVertex3f(-1.0, -1.0, 1.0);
			glTexCoord2f(0.0, 0.0);
			glVertex3f(1.0, -1.0, 1.0);
			glTexCoord2f(0.0, 1.0);
			glVertex3f(1.0, 1.0, 1.0);
			glTexCoord2f(1.0, 1.0);
			glVertex3f(-1.0, 1.0, 1.0);
		glEnd();
	
		glBindTexture(GL_TEXTURE_2D, curInfrastructure->getSkyBoxTexture(SKYBOX_FRONT));
		glBegin(GL_QUADS);
			glTexCoord2f(0.0, 1.0);
			glVertex3f(-1.0, 1.0, -1.0);
			glTexCoord2f(1.0, 1.0);
			glVertex3f(1.0, 1.0, -1.0);
			glTexCoord2f(1.0, 0.0);
			glVertex3f(1.0, -1.0, -1.0);
			glTexCoord2f(0.0, 0.0);
			glVertex3f(-1.0, -1.0, -1.0);
		glEnd();
	
		glBindTexture(GL_TEXTURE_2D, curInfrastructure->getSkyBoxTexture(SKYBOX_UP));
		glBegin(GL_QUADS);
			glTexCoord2f(1.0, 0.0);
			glVertex3f(-1.0, 1.0, -1.0);
			glTexCoord2f(1.0, 1.0);
			glVertex3f(1.0, 1.0, -1.0);
			glTexCoord2f(0.0, 1.0);
			glVertex3f(1.0, 1.0, 1.0);
			glTexCoord2f(0.0, 0.0);
			glVertex3f(-1.0, 1.0, 1.0);
		glEnd();
	
		glBindTexture(GL_TEXTURE_2D, curInfrastructure->getSkyBoxTexture(SKYBOX_DOWN));
		glBegin(GL_QUADS);
			glTexCoord2f(0.0, 0.0);
			glVertex3f(-1.0, -1.0, -1.0);
			glTexCoord2f(1.0, 0.0);
			glVertex3f(1.0, -1.0, -1.0);
			glTexCoord2f(1.0, 1.0);
			glVertex3f(1.0, -1.0, 1.0);
			glTexCoord2f(0.0, 1.0);
			glVertex3f(-1.0, -1.0, 1.0);
		glEnd();

		glBindTexture(GL_TEXTURE_2D, curInfrastructure->getSkyBoxTexture(SKYBOX_RIGHT));
		glBegin(GL_QUADS);
			glTexCoord2f(1.0, 0.0);
			glVertex3f(-1.0, -1.0, -1.0);
			glTexCoord2f(1.0, 1.0);
			glVertex3f(-1.0, 1.0, -1.0);
			glTexCoord2f(0.0, 1.0);
			glVertex3f(-1.0, 1.0, 1.0);
			glTexCoord2f(0.0, 0.0);
			glVertex3f(-1.0, -1.0, 1.0);
		glEnd();
		
		glBindTexture(GL_TEXTURE_2D, curInfrastructure->getSkyBoxTexture(SKYBOX_LEFT));
		glBegin(GL_QUADS);
			glTexCoord2f(0.0, 0.0);
			glVertex3f(1.0, -1.0, -1.0);
			glTexCoord2f(0.0, 1.0);
			glVertex3f(1.0, 1.0, -1.0);
			glTexCoord2f(1.0, 1.0);
			glVertex3f(1.0, 1.0, 1.0);
			glTexCoord2f(1.0, 0.0);
			glVertex3f(1.0, -1.0, 1.0);
		glEnd();
	
	glPopMatrix();
			
	glDepthMask(GL_TRUE);
	glEnable(GL_CULL_FACE );
	
}

void TScene::SetObjectPortalScissor(int index) {
	if(renderBlurFrame) return;
	if(ObjectList[index].sectorIndex != curSectorIndex && ObjectList[index].winX1 != -1 && ObjectList[index].winY1 != -1
		&& ObjectList[index].winX2 != -1 && ObjectList[index].winY2 != -1) {
		glScissor(ObjectList[index].winX1, ObjectList[index].winY1, 
				abs(ObjectList[index].winX2 - ObjectList[index].winX1), abs(ObjectList[index].winY2 - ObjectList[index].winY1));
	}
	else {
		glScissor(0, 0, curInfrastructure->GetWinWidth(), curInfrastructure->GetWinHeight() );
	}
}

void TScene::PortalWindowRender() {
	glActiveTextureARB( GL_TEXTURE1_ARB );
	glDisable( GL_TEXTURE_CUBE_MAP_ARB );
	glDisable( GL_TEXTURE_2D );

	glActiveTextureARB( GL_TEXTURE0_ARB );
	glDisable( GL_TEXTURE_CUBE_MAP_ARB );

	glDisable(GL_REGISTER_COMBINERS_NV);
	glDisable(GL_CULL_FACE);
	glDisable(GL_DEPTH_TEST);
	glDepthFunc(GL_LEQUAL);

	glDisable(GL_LIGHTING);
	
	glDisable(GL_STENCIL_TEST);
	glDepthMask(GL_FALSE);
	
	for(int i = 0; i < PortalWindowList.size(); i++) {
		glColor3f(0.0, 1.0, 0.0);
		glBegin(GL_QUADS);
			glVertex3f(PortalWindowList[i].DstBounds1.x, PortalWindowList[i].DstBounds1.y,  PortalWindowList[i].DstBounds1.z);
			glVertex3f(PortalWindowList[i].DstBounds3.x, PortalWindowList[i].DstBounds3.y,  PortalWindowList[i].DstBounds3.z);
			glVertex3f(PortalWindowList[i].DstBounds2.x, PortalWindowList[i].DstBounds2.y,  PortalWindowList[i].DstBounds2.z);
			glVertex3f(PortalWindowList[i].DstBounds4.x, PortalWindowList[i].DstBounds4.y,  PortalWindowList[i].DstBounds4.z);
		glEnd();
	}

	glColor3f(1.0, 1.0, 1.0);
	glDepthMask(GL_TRUE);
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_CULL_FACE);
}

void TScene::AddGameUnitDefine(char * name, char * md2_Filename, int type, int attackType, int damage, int buildtime, int health,
						float moveVel, float turnVel, int cost, float size, float range) {
	GameType newUnit;
	newUnit.name = name;
	newUnit.md2_Filename = md2_Filename;
	newUnit.type = type;
	newUnit.attackType = attackType;
	newUnit.damage = damage;
	newUnit.buildtime = buildtime;
	newUnit.health = health;
	newUnit.transVel = moveVel;
	newUnit.rotVel = turnVel;
	newUnit.cost = cost;
	newUnit.size = size;
	newUnit.range = range;
	
	gameObjectTypeList.push_back(newUnit);	
}

int TScene::LoadUnitInfoFile(char *filename) {
	FILE *file;
	char buf[256], *gotbuf;
	char name[256], md2_Filename[256];
	char modelName[256], bumpName[256], decalName[256];
	int specTrue, diffuseTrue, shadowTrue, attenuateTrue, animation, smoothness, windowIndex, sectorIndex, flipnormal;
	float x, y, z, scale, rotate, bscale, tileFactor, brightness, red, green, blue, alphaval, size, range;
	Vector vec1, vec2, vec3, vec4;	
	int type, attackType, damage, buildtime, health, cost, matches = 0;
	Vector moveVec, turnVec;
	
	file = fopen(filename, "r");
	if (file == NULL) {
		cout << "Could not Unit Definition load " << filename << "." << endl;
		return -1;
 	 }
	 	cout << "Loading Unit Definition File: " << filename << endl;
		
	 	while (!feof(file)) {
			gotbuf = is_fgets(buf, sizeof(buf), file);
			if (gotbuf) {
				switch(buf[0]) {
					case '#':
					break;
					
					case 'M':
					matches = sscanf(buf, "MD2 %s %f, %f, %f %f, %f, %f %f, %f, %f %s %s %d %d %d %d %f %d %d %d", name,
										&vec1.x, &vec1.y, &vec1.z, &vec2.x, &vec2.y, &vec2.z, &vec3.x, &vec3.y, &vec3.z,
										decalName, bumpName, &shadowTrue, &diffuseTrue, &specTrue, &attenuateTrue, &bscale, &animation, &smoothness, &flipnormal);
					if(matches == 20) {
						TMd2 *newMd2 = new TMd2(curInfrastructure);
						newMd2->SetLoadFlippedNormals(flipnormal);

						if(LoadedMd2(name) != NULL) newMd2->Copy(*(LoadedMd2(name))); 
						else newMd2->LoadGeom(name);
						newMd2->LoadTexture(decalName, bumpName, true, bscale);
						newMd2->PrepareTransform();
							glTranslatef(vec1.x, vec1.y, vec1.z);
							glRotatef(vec3.z, 0.0, 0.0, 1.0f);
							glRotatef(vec3.y, 0.0, 1.0f, 0.0);
							glRotatef(vec3.x, 1.0f, 0.0, 0.0);
							glScalef(vec2.x, vec2.y, vec2.z);
						newMd2->EndTransform();
						newMd2->SetCurrentAnimation(animation, smoothness);
						newMd2->PlayAnimation();
						newMd2->SaveMatrix();
						newMd2->CalcBounds();
						LoadedTObjects.push_back(newMd2);
					}	else {fclose(file); ; cout << "Bad Parse for MD2! Number of matches: " << matches << endl; return BAD_PARSE;}
					
					break;
					
					case 'U':
						/* UNIT name md2_filename type attackType damage buildtime health moveVel rotVel cost size range*/
						matches = sscanf(buf, "UNIT %s %s %d %d %d %d %d %g %g %d %g %g\n",
									name, md2_Filename, &type, &attackType, &damage, &buildtime, &health, &moveVec.x, 
									&turnVec.x, &cost, &size, &range);
						if (matches == 12) {
							AddGameUnitDefine(name, md2_Filename, type, attackType, damage, buildtime, health, moveVec.x, turnVec.x, cost, size, range);
						} else {fclose(file); cout << "Bad Parse for UNIT!" << endl; return -1;}
					break;
				}
			}
		}
		
		return true;
}

GameType TScene::GetUnitInfo(int type) {
	for(int i = 0; i < gameObjectTypeList.size(); i++) {
		if(gameObjectTypeList[i].type == type) return gameObjectTypeList[i];
	}
}

int TScene::LoadItemsFile(char *filename) {
	FILE *file;
	char buf[256], *gotbuf;
	char name[256], md2_Filename[256];
	char modelName[256], bumpName[256], decalName[256];
	int specTrue, diffuseTrue, shadowTrue, attenuateTrue, animation, smoothness, windowIndex, sectorIndex, flipnormal;
	float x, y, z, scale, rotate, bscale, tileFactor, brightness, red, green, blue, alphaval, size, range;
	Vector vec1, vec2, vec3, vec4;	
	int type, attackType, damage, buildtime, health, cost, matches = 0;
	Vector moveVec, turnVec;
	
	file = fopen(filename, "r");
	if (file == NULL) {
		cout << "Could not Load Items Definition: " << filename << "." << endl;
		return -1;
 	 }
	 	cout << "Loading Items Definition File: " << filename << endl;
		
	 	while (!feof(file)) {
			gotbuf = is_fgets(buf, sizeof(buf), file);
			if (gotbuf) {
				switch(buf[0]) {
					case '#':
					break;
					
					case 'M':
					// MD2 filename locX, locY, locZ scaleX, scaleY, scaleZ rotX, rotY, rotZ decalTexture bumpTexture shadowTrue diffuseTrue specularTrue attenuateTrue scale animation smoothness flipNormal type
					matches = sscanf(buf, "MD2 %s %f, %f, %f %f, %f, %f %f, %f, %f %s %s %d %d %d %d %f %d %d %d %d", name, &vec1.x, &vec1.y, &vec1.z, &vec2.x, &vec2.y, &vec2.z, &vec3.x, &vec3.y, &vec3.z,
										decalName, bumpName, &shadowTrue, &diffuseTrue, &specTrue, &attenuateTrue, &bscale, &animation, &smoothness, &flipnormal, &type);
					if(matches == 21) {
						TMd2 *newMd2 = new TMd2(curInfrastructure);
						newMd2->SetLoadFlippedNormals(flipnormal);

						if(LoadedMd2(name) != NULL) newMd2->Copy(*(LoadedMd2(name))); 
						else newMd2->LoadGeom(name);
						newMd2->LoadTexture(decalName, bumpName, true, bscale);
						newMd2->PrepareTransform();
							glTranslatef(vec1.x, vec1.y, vec1.z);
							glRotatef(vec3.z, 0.0, 0.0, 1.0f);
							glRotatef(vec3.y, 0.0, 1.0f, 0.0);
							glRotatef(vec3.x, 1.0f, 0.0, 0.0);
							glScalef(vec2.x, vec2.y, vec2.z);
						newMd2->EndTransform();
						newMd2->SetCurrentAnimation(animation, smoothness);
						newMd2->PlayAnimation();
						newMd2->SaveMatrix();
						newMd2->CalcBounds();
						LoadedTObjects.push_back(newMd2);
						Vset(&vec1, OUT_OF_RANGE, OUT_OF_RANGE, OUT_OF_RANGE);
						AttachObject(LoadedTObjects[LoadedTObjects.size() - 1], vec1, vec2, vec3, shadowTrue, specTrue, 
									diffuseTrue, attenuateTrue, -1, false, 0.0, type, NULL, 9000);
						ObjectList[ObjectList.size() - 1].gamePacket.health = 65536 * 100;
					}	
					else {
						fclose(file); 
						cout << "Bad Parse for MD2! Number of matches: " << matches << endl; return BAD_PARSE;
					}
					
					break;
					
					case 'U':
						/* UNIT name md2_filename type attackType damage buildtime health moveVel rotVel cost size range*/
						matches = sscanf(buf, "UNIT %s %s %d %d %d %d %d %g %g %d %g %g\n",
									name, md2_Filename, &type, &attackType, &damage, &buildtime, &health, &moveVec.x, 
									&turnVec.x, &cost, &size, &range);
						if (matches == 12) {
//							AddGameUnitDefine(name, md2_Filename, type, attackType, damage, buildtime, health, moveVec.x, turnVec.x, cost, size, range);
						} else {fclose(file); cout << "Bad Parse for UNIT!" << endl; return -1;}
					break;
				}
			}
		}
		
		for(int m = 0; m < 20; m++) ReSpawnItems();
		
		return true;
}

void TScene::ReSpawnItems() {
	for(int i = 0; i < ObjectList.size(); i++) {
		if(ObjectList[i].gamePacket.type >= 50 && ObjectList[i].gamePacket.type != GAME_PILLAR_TYPE) { //object is an item! we can try to respawn it
			Vector curPos = ObjectList[i].translate;
			if(curPos.x == OUT_OF_RANGE && curPos.y == OUT_OF_RANGE && curPos.z == OUT_OF_RANGE) {
				Vector trySpawn, realSpawn;
				int sectorPick = rand() % PortalList.size();
				float factor1 = (float)rand() / (float) RAND_MAX;
				float factor2 = 1.0 - factor1;
				factor2 = (float)rand() / (float) RAND_MAX * factor2;
				float factor3 = 1.0 - factor1 - factor2;
				factor3 = (float)rand() / (float) RAND_MAX * factor3;
				float factor4 = 1.0 - factor1 - factor2 - factor3;
				
				//now see if we can spawn at this point
				trySpawn.x = PortalList[sectorPick].Bound1.x * factor1 + PortalList[sectorPick].Bound2.x * factor2
									+ PortalList[sectorPick].Bound3.x * factor3 + PortalList[sectorPick].Bound4.x * factor4;
				trySpawn.z = PortalList[sectorPick].Bound1.z * factor1 + PortalList[sectorPick].Bound2.z * factor2
									+ PortalList[sectorPick].Bound3.z * factor3 + PortalList[sectorPick].Bound4.z * factor4;
									
				trySpawn.y = getObjectSpawnY(ObjectList[i].curObject, 0.0, 1.0); //itemsSpawnY;
				
				realSpawn = tryToSpawn(trySpawn, ObjectList[i].curObject, 0.5, i);
				ObjectList[i].translate = realSpawn;
			}
		}
	}
}

Vector TScene::mustSpawnRandom(TObject *curObject, float size, int index) {
	for(int i = 0; i < 1000; i++) {
		int sectorPick = rand() % PortalList.size();
		Vector attemptSpawn;
		Vset(&attemptSpawn, (PortalList[sectorPick].Bound1.x + PortalList[sectorPick].Bound2.x) * 0.5,
			(PortalList[sectorPick].Bound1.y + PortalList[sectorPick].Bound2.y) * 0.5,
			(PortalList[sectorPick].Bound1.z + PortalList[sectorPick].Bound2.z) * 0.5);
		
		//now see if we can spawn at this point
		Vector trySpawn = tryToSpawn(attemptSpawn, curObject, size, index);
		if(trySpawn.x != OUT_OF_RANGE && trySpawn.y != OUT_OF_RANGE && trySpawn.z != OUT_OF_RANGE) {
			return trySpawn;
		}
	}
	
	Vector zeroVec;
	Vset(&zeroVec, 0.0, 0.0, 0.0);
	return zeroVec;
	
}


void TScene::AnimateItems() {
	for(int i = 0; i < ObjectList.size(); i++) {
		if(ObjectList[i].gamePacket.type >= 50) { //object is an item! we can animate it!
			ObjectList[i].rotate.y = ObjectList[i].rotate.y + 2.0;
		}
	}
}

float TScene:: getMinSpawnY() {
	float minY = 999999.0;
	for(int i = 0; i < ObjectList.size(); i++) {
		if(ObjectList[i].curObject->GetType() == PLANE_TYPE) {
			Vector maxBound = ObjectList[i].curObject->GetMaxOrgBounds();
			if(maxBound.y < minY) minY = maxBound.y;
		}
	}
	
	return minY;
}

float TScene::getObjectSpawnY(TObject *curObject, float offset, float scale) {
	Vector maxBound = curObject->GetMaxOrgBounds();
	Vector minBound = curObject->GetMinOrgBounds();
	
	float deltaY  = scale * (maxBound.y - minBound.y);
	return getMinSpawnY() + offset;
	
}

Vector TScene::tryToSpawn(Vector curPt, TObject *curObject, float size, int index) {
	Vector maxBound = curObject->GetMaxOrgBounds();
	Vector minBound = curObject->GetMinOrgBounds();
	Vector diffVec;
	Vsub(&maxBound, &minBound, &diffVec);
	float length = Vlength(&diffVec);
	Vector spawnVector, spawnVector2;
	Vset(&spawnVector, 0.0, 0.0, length);
	spawnVector2 = spawnVector;

	RestoreSceneObjectsTransform();
	ApplyMd2PhysicsTransform();
		
	for(int i = 1; i <= 20; i++) {
		spawnVector = spawnVector2;
		Vscale(&spawnVector, (float)(i - 1) * 0.8);
		for(int j = 0; j < 360; j = j + 10) {
			Vector tmpVec = spawnVector;
			spawnVector.x = tmpVec.x * cos( (float)MATH_PI / 180.0 * (float)j) - tmpVec.z * sin( (float)MATH_PI / 180.0 * (float)j);
			spawnVector.z = tmpVec.x * sin( (float)MATH_PI / 180.0 * (float)j) + tmpVec.z * cos( (float)MATH_PI / 180.0 * (float)j);
			spawnVector.y = 0.0;
			
			Vector bnd1 = *(curObject->GetBounds1());
			Vector bnd2 = *(curObject->GetBounds2());
			Vector bnd3 = *(curObject->GetBounds3());
			Vector bnd4 = *(curObject->GetBounds4());
			Vector bnd5 = *(curObject->GetBounds5());
			Vector bnd6 = *(curObject->GetBounds6());
			Vector bnd7 = *(curObject->GetBounds7());
			Vector bnd8 = *(curObject->GetBounds8());
			
			Vscale(&bnd1, size);
			Vscale(&bnd2, size);
			Vscale(&bnd3, size);
			Vscale(&bnd4, size);
			Vscale(&bnd5, size);
			Vscale(&bnd6, size);
			Vscale(&bnd7, size);
			Vscale(&bnd8, size);
			
			Vadd(&bnd1, &spawnVector, &bnd1);
			Vadd(&bnd2, &spawnVector, &bnd2);
			Vadd(&bnd3, &spawnVector, &bnd3);
			Vadd(&bnd4, &spawnVector, &bnd4);
			Vadd(&bnd5, &spawnVector, &bnd5);
			Vadd(&bnd6, &spawnVector, &bnd6);
			Vadd(&bnd7, &spawnVector, &bnd7);
			Vadd(&bnd8, &spawnVector, &bnd8);

			Vadd(&bnd1, &curPt, &bnd1);
			Vadd(&bnd2, &curPt, &bnd2);
			Vadd(&bnd3, &curPt, &bnd3);
			Vadd(&bnd4, &curPt, &bnd4);
			Vadd(&bnd5, &curPt, &bnd5);
			Vadd(&bnd6, &curPt, &bnd6);
			Vadd(&bnd7, &curPt, &bnd7);
			Vadd(&bnd8, &curPt, &bnd8);
						
			dummyObj->SetDstBounds1(bnd1.x, bnd1.y, bnd1.z);
			dummyObj->SetDstBounds2(bnd2.x, bnd2.y, bnd2.z);
			dummyObj->SetDstBounds3(bnd3.x, bnd3.y, bnd3.z);
			dummyObj->SetDstBounds4(bnd4.x, bnd4.y, bnd4.z);
			dummyObj->SetDstBounds5(bnd5.x, bnd5.y, bnd5.z);
			dummyObj->SetDstBounds6(bnd6.x, bnd6.y, bnd6.z);
			dummyObj->SetDstBounds7(bnd7.x, bnd7.y, bnd7.z);
			dummyObj->SetDstBounds8(bnd8.x, bnd8.y, bnd8.z);
			
			//now do intersectiont tests;
			bool flag = false;
			for(int k = 0; k < ObjectList.size(); k++) {
				if(k == index) continue;
				if(ObjectList[k].curObject->GetType() == PLANE_TYPE) {
					TPlane *curPlane = (TPlane *)ObjectList[k].curObject;
					if( (curPlane->GetNormal()).y > 0.9) continue;
				}
				
				if(OBBIntersect(dummyObj, ObjectList[k].curObject)) { // found one
					flag = true;
					break;
				}
			}
			
			if(!flag) {
				Vector retVec;
				Vadd(&curPt, &spawnVector,&retVec);
				return retVec;
			}
			
		}
	}
	
	Vset(&spawnVector, OUT_OF_RANGE, OUT_OF_RANGE, OUT_OF_RANGE);
	return spawnVector;
}

void TScene::RenderGameSpiritEffects() {
	//go through all objects and see if we need to render spirit effects for a human player
	for(int i = 0; i < ObjectList.size(); i++) {
		if(ObjectList[i].gamePacket.type == PLAYER_TYPE) {
			if(ObjectList[i].gamePacket.health <= 0 && ObjectList[i].gamePacket.deadFrameCnt <= MAX_DEAD_FRAME) {
				ObjectList[i].gamePacket.deadFrameCnt++;
				if(ObjectList[i].gamePacket.deadFrameCnt <= 9) break;
				TMd2 *curMd2 = (TMd2 *)ObjectList[i].curObject;
				curMd2->RestoreMatrix();
				curMd2->PrepareTransform();
					glLoadIdentity();
					glTranslatef(ObjectList[i].translate.x, ObjectList[i].translate.y + (float)(ObjectList[i].gamePacket.deadFrameCnt - 10), ObjectList[i].translate.z);
					glRotatef(ObjectList[i].rotate.z, 0.0, 0.0, 1.0f);
					glRotatef(ObjectList[i].rotate.y + (float)(ObjectList[i].gamePacket.deadFrameCnt - 10), 0.0, 1.0f, 0.0);
					glRotatef(ObjectList[i].rotate.x, 1.0f, 0.0, 0.0);
					glScalef(ObjectList[i].scale.x, ObjectList[i].scale.y, ObjectList[i].scale.z);
				curMd2->EndTransform();
				
				curMd2->MatrixMult(this->inverseViewerMatrix);
				
				curMd2->EnablePlainDecal();
				curMd2->SelectSingleDecalTexture();
				curMd2->SetupVertexArraysForRendering(DECAL_MODE);
				glEnable(GL_BLEND);
					glBlendFunc(GL_SRC_ALPHA,GL_ONE);
					glColor4f(1.0, 1.0, 1.0, 0.5 - (float)(ObjectList[i].gamePacket.deadFrameCnt - 10) / ((MAX_DEAD_FRAME - 10) * 2));
					curMd2->Render();
				glDisable(GL_BLEND);
				
				break;
			}
		}
	}
}




void TScene::cleanUpTObjects(int index) {
	
}












