#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <signal.h>

#include "lrt.h"
#include "scene.h"
#include "camera.h"
#include "mbdofcamera.h"
#include "primitives.h"
#include "image.h"
#include "transport.h"
#include "accel.h"
#include "sampling.h"
#include "caustic.h"

Scene *scene = NULL;
struct shmDB *shm_base;
Caustic *caustic;

void Scene::Render()
{

	Integrator *integrator = NULL;
	if (IlluminationIntegrator == LRT_COLOR)
		integrator = new ColorIntegrator;
	else if (IlluminationIntegrator == LRT_RAYCAST)
		integrator = new RayCastingIntegrator;
	else if (IlluminationIntegrator == LRT_WHITTED)
		integrator = new WhittedIntegrator;
	else if (IlluminationIntegrator == LRT_MONTECARLO)
		integrator = new MCIntegrator;
	else if (IlluminationIntegrator == LRT_RMAN)
		integrator = new RendermanIntegrator;
	else {
		Severe("Unknown integrator \'%s\'", IlluminationIntegrator);
		return;
	}

	camera->FinalizeOptions();
	sampler->FinalizeValues();
	image->FinalizeValues();
	shm_base->imageWidth = image->XResolution;
	shm_base->imageHeight = image->YResolution;

	cerr << "Rendering: " << shm_base->imageWidth << "x" << shm_base->imageHeight;
	Float sample[5];
	while (sampler->GetNextImageSample(sample)) {

		Ray ray;
		if (!camera->GenerateRay(sample, ray))
			continue;

		static int eyeRaysTraced = 0;
		if (eyeRaysTraced == 0)
			StatsRegisterCounter(STATS_BASIC, "Camera", "Eye Rays Traced",
								 &eyeRaysTraced);
		++eyeRaysTraced;
		if (eyeRaysTraced % 10000 == 0)
			cerr << '+';

		HitInfo hitInfo;
		Float alpha, hitDist = INFINITY;
		Spectrum L =
			integrator->Integrate(ray, &hitInfo, &hitDist, &alpha);
			
		Float screenz = camera->WorldToScreen(ray(hitDist)).z;
		if (screenz > 1.) {
			L = 0.;
			alpha = 0.;
		}
		Point Praster(sample[0], sample[1], screenz);
		image->AddSample(Praster, L, alpha);		
	}
	image->Write();
	cerr << endl;
}

bool Scene::Intersect(const Ray & ray, Float mint, Float * maxt,
					  HitInfo * hit) const
{
	if (!accelerator)
		return false;
	return accelerator->IntersectClosest(ray, mint, maxt, hit);
}

Scene::Scene()
{
	int smemid;

	IlluminationIntegrator = LRT_WHITTED;

	smemid = shmget(13482, 2048, IPC_CREAT|0666);
	if (smemid == -1){
		perror("shmget:");
		printf("shared memory get failed\n");
		exit(1);
	}
	shm_base = NULL;
	shm_base = (struct shmDB *)shmat(smemid, NULL, 0); 
	if (!shm_base){
		perror("shmat:");
		printf("shared memory attach failed\n");
		exit(1);
	}
	shm_base->lrtpid = getpid();

	accelerator = NULL;
	camera = new PinholeCamera;
	image = new Image;
	sampler = new JitterSampler;
	caustic = new Caustic;
}

void Scene::AddPrimitives(const vector < Primitive * >&prim)
{
	primitives = prim;
	if (primitives.size() > 0)
		accelerator = new GridAccelerator(primitives);
}

Scene::~Scene()
{
	delete accelerator;
	delete camera;
	delete sampler;
	delete image;
//  for (u_int i = 0; i < primitives.size(); ++i)
//      delete primitives[i];
}

bool Scene::Unoccluded(const Point & p1, const Point & p2) const
{
	if (!accelerator)
		return true;

	static int shadowRayChecks = 0, shadowRayOccluded = 0;
	if (shadowRayChecks == 0)
		StatsRegisterRatio(STATS_DETAILED, "Integration",
						   "Finite Shadow Ray Checks",
						   &shadowRayOccluded, &shadowRayChecks);
	++shadowRayChecks;

	Float tmin = 1e-6;
	Float tmax = 1. - tmin;
	if (accelerator->IntersectClosest(Ray(p1, p2 - p1), tmin, &tmax, NULL)) {
		++shadowRayOccluded;
		return false;
	}

	return true;
}

bool Scene::Unoccluded(const Ray & r) const
{
	if (!accelerator)
		return true;

	static int shadowRayChecks = 0, shadowRayOccluded = 0;
	if (shadowRayChecks == 0)
		StatsRegisterRatio(STATS_DETAILED, "Integration",
						   "Infinite Shadow Ray Checks",
						   &shadowRayOccluded, &shadowRayChecks);
	++shadowRayChecks;

	Float tmin = 1e-6;
	Float tmax = INFINITY;
	Ray ray = r;
	ray.D.Normalize();
	if (accelerator->IntersectClosest(ray, tmin, &tmax, NULL)) {
		++shadowRayOccluded;
		return false;
	}

	return true;
}
