
#include "lrt.h"
#include "color.h"
#include "reflection.h"

#define WAVE_VALUE 70

BRDF::~BRDF()
{
}

Spectrum Lambertian::fr(const Vector & wi) const
{
	Float costheta = Dot(wi, N);
	if (costheta > 0.)
		return R;
	else
		return Spectrum(0.);
}

BlinnGlossy::BlinnGlossy(const Spectrum & reflectance, Float roughness,
						 const Normal & normal, const Vector & w)
{
	R = reflectance;
	roughness /= 8.;			// hack to match BMRT, prman
	invRoughness = 1. / roughness;
	N = normal;
	wo = w;
	costhetao = Dot(wo, N);
}

Spectrum BlinnGlossy::fr(const Vector & wi) const
{
	Vector H = (wi + wo).Hat();
	Float costhetah = Dot(N, H);
	Float costhetai = Dot(N, wi);
	if (costhetah > 0. && costhetai > 0.)
		return R * pow(costhetah, invRoughness) / (costhetai * costhetao);
	else
		return Spectrum(0.);
}

int BRDF::SpecularComponents() const
{
	return 0;
}

Spectrum BRDF::SampleSpecular(int component, Vector * wo) const
{
	Severe("SampleSpecular() shouldn't be called for BRDFs "
		   "without specular components.");
	return Spectrum(0.);
}

SpecularReflection::SpecularReflection(const Spectrum & r,
									   const Normal & N, const Vector & wo)
{
	R = r;
	reflectedDirection = -wo - 2. * Dot(N, -wo) * Vector(N);
}

Spectrum SpecularReflection::SampleSpecular(int component, Vector * wi) const
{
	Assert(component == 0);
	*wi = reflectedDirection;
	return R;
}

SpecularTransmission::SpecularTransmission(const Spectrum & r,
										   const Normal & N,
										   const Vector & wo, Float indexi,
										   Float indext)
{
	int i, lambda;
	R = r;
/* This is here for debugging purposes. To see an image with only a
	single wavelength, uncomment this code and set WAVE_VALUE to a
	value between 0 and 79.
	if (R.monochromatic == 0) {
		R.populateWaves();
	}
	R = Spectrum(R, WAVE_VALUE);
	// R.printWaves();
	R.spectrum_to_rgb();
	R.constrain_rgb();
	R.printRGB();
*/

	if (R.monochromatic == 1) {
	/* If the spectrum is monochromatic, just calculate the direction
	   of transmission for the single value
	*/
		num_split = 1;
		Float c1 = Dot(wo, N);
		lambda = 380 + ((400/SPEC) * R.wave_bucket);
		if (indext != 1.0) {
			indext = (37841.0 + (lambda + (200./SPEC))) / 25620.;
		} 
		if (indexi != 1.0) {
			indexi = (37841. + (lambda + (200./SPEC))) / 25620.;
		}
		Float eta = indexi / indext;
		Float c2 = 1 - (eta * eta * (1. - c1));
		if (c2 < 0.) {
			// total internal reflection
			R = 0.;
			DirT[R.wave_bucket] = wo;
		} else {
			c2 = sqrt(c2);
			DirT[R.wave_bucket] = -eta * wo + (eta * c1 - c2) * Vector(N);
		}
	}
	else {
	/* If the spectrum is not initizalize (monochromatic == 0), do so
		here.
	*/
		if (R.monochromatic == 0) {
			R.populateWaves();
		}
		num_split = 400/(GAP * 5);

		/* Calculate the direction of transmission for each wavelength
			bucket.
		*/
		Float c1 = Dot(wo, N);
		for (i = 0, lambda = 380; lambda < 780; i += GAP, lambda += 5 * GAP) {
			if (indext != 1.0) {
				indext = (37841. + (lambda + (200./SPEC))) / 25620.;
			}
			if (indexi != 1.0) {
				indexi = (37841. + (lambda + (200./SPEC))) / 25620.;
			}
			Float eta = indexi / indext;
			Float c2 = 1 - (eta * eta * (1. - c1));
			if (c2 < 0.) {
				// total internal reflection
				R = 0.;
				DirT[i] = wo;
			} else {
				c2 = sqrt(c2);
				DirT[i] = -eta * wo + (eta * c1 - c2) * Vector(N);
			}
		}
	}
}

Spectrum SpecularTransmission::SampleSpecular(int component, Vector * wi) const
{
	if (R.monochromatic == 1) {
		*wi = DirT[R.wave_bucket];
		return R;
	} 

	/* If the original spectrum was not monochromatic, then we need to split
		out the energy corresponding to the wavelenth of the component.
		The Spectrum constructor below does this, returning a Spectrum with
		the same energy in that wavelenth and 0 elsewhere.
	*/
	*wi = DirT[component * GAP];
	Spectrum s = Spectrum(R, component * GAP);
	s.spectrum_to_rgb();
	s.constrain_rgb();
	return s;
}

ScatteringMixture::~ScatteringMixture()
{
	vector < BRDF * >::const_iterator iter;
	for (iter = funcs.begin(); iter != funcs.end(); ++iter)
		delete *iter;
}

Spectrum ScatteringMixture::fr(const Vector & wi) const
{
	Spectrum ret(0);
	vector < BRDF * >::const_iterator funcIter = funcs.begin();
	vector < Float >::const_iterator weightIter = weights.begin();

	while (funcIter != funcs.end()) {
		ret += (*weightIter) * (*funcIter)->fr(wi);
		funcIter++, weightIter++;
	}
	return ret;
}

Spectrum ScatteringMixture::SampleSpecular(int component, Vector * wi) const
{
	Assert(component < SpecularComponents());
	vector < BRDF * >::const_iterator funcIter = funcs.begin();
	vector < Float >::const_iterator weightIter = weights.begin();

	while (funcIter != funcs.end()) {
		if (component >= (*funcIter)->SpecularComponents())
			component -= (*funcIter)->SpecularComponents();
		else
			return (*weightIter) * (*funcIter)->SampleSpecular(component,
															   wi);
		++funcIter, ++weightIter;
	}

	// We shouldn't get to this point.
	Severe("Major logic error in ScatteringMixture::SampleSpecular()");
	return Spectrum(0);
}

#if 0
Lafortune::Lafortune(int n, const LafortuneLobe l[], const Vector & w,
					 const Normal & norm, const Vector & dpdu)
{
	nLobes = n;
	lobes = new LafortuneLobe[n];
	for (int i = 0; i < nLobes; ++i)
		lobes[i] = l[i];
	wo = w;
	N = norm.Hat();
	dPdu = dpdu.Hat();
	dPdv = Cross((Vector) N, dPdu);
}
#endif

#if 0
Spectrum Lafortune::fr(const Vector & wi) const
{
	Spectrum value(0.);

	Float xd = Dot(dPdu, wi) * Dot(dPdu, wo);
	Float yd = Dot(dPdv, wi) * Dot(dPdv, wo);
	Float zd = Dot(N, wi) * Dot(N, wo);

	for (int i = 0; i < nLobes; ++i) {

		Spectrum dot =
			xd * lobes[i].Cx + yd * lobes[i].Cy + zd * lobes[i].Cz;
		value += dot.Pow(lobes[i].Exponent);

	}
	return value;
}
#endif

BRDF *CreateLambertian(const Spectrum & reflectance, const Normal & normal)
{
	return new Lambertian(reflectance, normal);
}

BRDF *CreateBlinnGlossy(const Spectrum & reflectance, Float roughness,
						const Normal & normal, const Vector & wo)
{
	return new BlinnGlossy(reflectance, roughness, normal, wo);
}

BRDF *CreateSpecularReflection(const Spectrum & r, const Normal & N,
							   const Vector & wo)
{
	return new SpecularReflection(r, N, wo);
}

BRDF *CreateSpecularTransmission(const Spectrum & r, const Normal & N,
								 const Vector & wo, Float indexi,
								 Float indext)
{
	return new SpecularTransmission(r, N, wo, indexi, indext);
}

ScatteringMixture *CreateScatteringMixture()
{
	return new ScatteringMixture;
}

#if 0
BRDF *CreateLafortune(int nLobes, const LafortuneLobe lobes[],
					  const Vector & wi, const Normal & N,
					  const Vector & dPdu)
{
	Assert(1 == 0);
	return NULL;
	// need to implement the sampling methods
//  return new Lafortune(nLobes, lobes, wi, N, dPdu);
}
#endif

Spectrum Lambertian::Sample(Float u[2], Vector * wi, Float * pdf) const
{

	Float u0 = 2 * u[0] - 1.;
	Float u1 = 2 * u[1] - 1.;
	if (u0 * u0 + u1 * u1 > 1.) {
		*pdf = 0.;
		return Spectrum(0.);
	}

	Vector Du, Dv;
	if (N.x == 0. && N.y == 0.) {
		Du = Vector(1, 0, 0);
		Dv = Vector(0, 1, 0);
	} else {
		Du = Cross(Vector(N.y, N.x, 0.), Vector(N)).Hat();
		Dv = Cross(Du, Vector(N)).Hat();
	}

	*wi = u0 * Du + u1 * Dv + sqrt(1. - (u0 * u0 + u1 * u1)) * Vector(N);
	*pdf = Dot(N, *wi) / M_PI;

	return fr(*wi);
}

Spectrum BlinnGlossy::Sample(Float u[2], Vector * wi, Float * pdf) const
{
	// YOUR CODE HERE
	return Spectrum(0.);
}

Spectrum ScatteringMixture::Sample(Float u[2], Vector * wi, Float * pdf) const
{
	int fnum = int (RandomFloat() * funcs.size());
	funcs[fnum]->Sample(u, wi, pdf);
	*pdf /= funcs.size();
	return fr(*wi);
}

Float Lambertian::Pdf(const Vector & wi) const
{
	Float costheta = Dot(wi, N);
	if (costheta > 0.)
		return 1. / (M_PI * costheta);
	else
		return 0.;
}

Float BlinnGlossy::Pdf(const Vector & wi) const
{
	// YOUR CODE HERE
	return 0.;
}

Float ScatteringMixture::Pdf(const Vector & wi) const
{
	Float pdfSum = 0.;
	for (u_int i = 0; i < funcs.size(); ++i)
		pdfSum += funcs[i]->Pdf(wi);
	return pdfSum / funcs.size();
}
