
#include "lrt.h"
#include "color.h"
#include <tiffio.h>

Spectrum *TIFFRead(const char *name, int *xSize, int *ySize)
{
	Spectrum *pixels = NULL;
	Float *fbuf = NULL;
	u_char *ubuf = NULL;


	TIFF *tiff = TIFFOpen(name, "r");
	if (!tiff) {
		Error("Unable to open TIFF %s", name);
		return NULL;
	}

	short int nSamples;
	TIFFGetField(tiff, TIFFTAG_IMAGEWIDTH, xSize);
	TIFFGetField(tiff, TIFFTAG_IMAGELENGTH, ySize);
	TIFFGetField(tiff, TIFFTAG_SAMPLESPERPIXEL, &nSamples);

	short int bitsPerSample, sampleFormat;
	if (!TIFFGetField(tiff, TIFFTAG_BITSPERSAMPLE, &bitsPerSample)) {
		Error("TIFFRead: bits per sample not set in TIFF");

		delete[]pixels;
		delete[]ubuf;
		delete[]fbuf;
		TIFFClose(tiff);
		return NULL;

	}
	if (!TIFFGetField(tiff, TIFFTAG_SAMPLEFORMAT, &sampleFormat)) {
		if (bitsPerSample == 32)
			sampleFormat = SAMPLEFORMAT_IEEEFP;
		else
			sampleFormat = SAMPLEFORMAT_UINT;
	}

	if (bitsPerSample == 32 && sampleFormat != SAMPLEFORMAT_IEEEFP) {
		Error("TIFFRead: 32 bit TIFF not stored in floating point format");

		delete[]pixels;
		delete[]ubuf;
		delete[]fbuf;
		TIFFClose(tiff);
		return NULL;

	} else {
		if (bitsPerSample != 8) {
			Error("TIFFRead: more than 8 bits per sample unsupported");

			delete[]pixels;
			delete[]ubuf;
			delete[]fbuf;
			TIFFClose(tiff);
			return NULL;

		}
		if (sampleFormat != SAMPLEFORMAT_UINT) {
			Error("TIFFRead: 8 bit TIFFs must be stored as unsigned ints");

			delete[]pixels;
			delete[]ubuf;
			delete[]fbuf;
			TIFFClose(tiff);
			return NULL;

		}
	}

	if (nSamples * *xSize != TIFFScanlineSize(tiff)) {
		Error("TIFFRead: RGB not interleaved in TIFF %s", name);

		delete[]pixels;
		delete[]ubuf;
		delete[]fbuf;
		TIFFClose(tiff);
		return NULL;

	}

	u_short *mapR = 0, *mapG = 0, *mapB = 0;
	if (nSamples == 1) {
		short photoType;
		TIFFGetField(tiff, TIFFTAG_PHOTOMETRIC, &photoType);
		if (photoType != PHOTOMETRIC_PALETTE) {
			Error("TIFFRead: colormap not found in one-sample image");

			delete[]pixels;
			delete[]ubuf;
			delete[]fbuf;
			TIFFClose(tiff);
			return NULL;

		}
		TIFFGetField(tiff, TIFFTAG_COLORMAP, &mapR, &mapG, &mapB);
	}

	pixels = new Spectrum[*xSize * *ySize];
	Spectrum *pixelp = pixels;

	if (bitsPerSample == 32)
		fbuf = new float[nSamples * *xSize];
	else
		ubuf = new u_char[nSamples * *xSize];

	for (int y = 0; y < *ySize; ++y) {

		if (fbuf) {

			float *fbufp = fbuf;
			if (TIFFReadScanline(tiff, fbuf, y, 1) == -1) {

				delete[]pixels;
				delete[]ubuf;
				delete[]fbuf;
				TIFFClose(tiff);
				return NULL;

			}
			for (int x = 0; x < *xSize; ++x) {
				*pixelp = Spectrum(fbufp[0], fbufp[1], fbufp[2]);
				++pixelp;
				fbufp += nSamples;
			}

		} else {

			u_char *ubufp = ubuf;
			if (TIFFReadScanline(tiff, ubuf, y, 1) == -1) {

				delete[]pixels;
				delete[]ubuf;
				delete[]fbuf;
				TIFFClose(tiff);
				return NULL;

			}
			for (int x = 0; x < *xSize; ++x) {
				if (nSamples == 1) {

					int mapOffset = *ubufp;
					*pixelp = Spectrum(mapR[mapOffset] * INV_255 * INV_255,
									   mapG[mapOffset] * INV_255 * INV_255,
									   mapB[mapOffset] * INV_255 *
									   INV_255);

				} else {

					*pixelp =
						Spectrum(ubufp[0] * INV_255, ubufp[1] * INV_255,
								 ubufp[2] * INV_255);

				}
				++pixelp;
				ubufp += nSamples;
			}

		}

	}

	delete[]ubuf;
	delete[]fbuf;
	TIFFClose(tiff);
	return pixels;

}

void TIFFWrite8Bit(const char *name, Float * RGB, Float * alpha,
				   Float * depth, int XRes, int YRes)
{
	Assert(RGB);

	TIFF *tiff = TIFFOpen(name, "w");
	if (!tiff) {
		Error("Unable to open TIFF %s for writing", name);
		return;
	}


	int sampleCount = 0;
	if (RGB)
		sampleCount += 3;
	if (alpha)
		++sampleCount;
	if (depth)
		Warning("Depth channel to be discarded in TIFF");	// for now
	TIFFSetField(tiff, TIFFTAG_SAMPLESPERPIXEL, sampleCount);
	if (alpha) {
		short int extra[] = { EXTRASAMPLE_ASSOCALPHA };
		TIFFSetField(tiff, TIFFTAG_EXTRASAMPLES, (short) 1, extra);
	}

	TIFFSetField(tiff, TIFFTAG_IMAGEWIDTH, XRes);
	TIFFSetField(tiff, TIFFTAG_IMAGELENGTH, YRes);
	TIFFSetField(tiff, TIFFTAG_BITSPERSAMPLE, 8);
	TIFFSetField(tiff, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_RGB);

	TIFFSetField(tiff, TIFFTAG_ROWSPERSTRIP, 1L);
	TIFFSetField(tiff, TIFFTAG_XRESOLUTION, 1.0);
	TIFFSetField(tiff, TIFFTAG_YRESOLUTION, 1.0);
	TIFFSetField(tiff, TIFFTAG_RESOLUTIONUNIT, 1);
	TIFFSetField(tiff, TIFFTAG_COMPRESSION, COMPRESSION_NONE);
	TIFFSetField(tiff, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG);
	TIFFSetField(tiff, TIFFTAG_ORIENTATION, (int) ORIENTATION_TOPLEFT);

	u_char *buf = new u_char[sampleCount * XRes];
	Float *pixelp = RGB, *alphap = alpha;

	for (int y = 0; y < YRes; ++y) {
		u_char *bufp = buf;
		for (int x = 0; x < XRes; ++x) {

			if (pixelp)
				for (int s = 0; s < 3; ++s) {
					*bufp = (u_char) (*pixelp);
					++bufp;
					++pixelp;
				}

			if (alphap) {
				*bufp = (u_char) (*alphap);
				++bufp;
				++alphap;
			}

		}
		TIFFWriteScanline(tiff, buf, y, 1);
	}

	delete[]buf;
	TIFFClose(tiff);

}

void TIFFWriteFloat(const char *name, Float * RGB, Float * alpha,
					Float * depth, int XRes, int YRes)
{
	if (alpha || depth)
		Warning("Alpha and/or depth channels to be discarded in TIFF");
	Assert(RGB);

	TIFF *tiff = TIFFOpen(name, "w");
	if (!tiff) {
		Error("Unable to open TIFF %s for writing", name);
		return;
	}

	TIFFSetField(tiff, TIFFTAG_IMAGEWIDTH, XRes);
	TIFFSetField(tiff, TIFFTAG_IMAGELENGTH, YRes);
	TIFFSetField(tiff, TIFFTAG_SAMPLESPERPIXEL, 3);
	TIFFSetField(tiff, TIFFTAG_BITSPERSAMPLE, 32);
	TIFFSetField(tiff, TIFFTAG_SAMPLEFORMAT, SAMPLEFORMAT_IEEEFP);
	TIFFSetField(tiff, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_MINISBLACK);

	TIFFSetField(tiff, TIFFTAG_ROWSPERSTRIP, 1L);
	TIFFSetField(tiff, TIFFTAG_XRESOLUTION, 1.0);
	TIFFSetField(tiff, TIFFTAG_YRESOLUTION, 1.0);
	TIFFSetField(tiff, TIFFTAG_RESOLUTIONUNIT, 1);
	TIFFSetField(tiff, TIFFTAG_COMPRESSION, COMPRESSION_NONE);
	TIFFSetField(tiff, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG);
	TIFFSetField(tiff, TIFFTAG_ORIENTATION, (int) ORIENTATION_TOPLEFT);

	Float *pixelp = RGB;
	for (int y = 0; y < YRes; ++y) {
		TIFFWriteScanline(tiff, pixelp, y, 1);
		pixelp += 3 * XRes;		// ahead to the start of the next scanline
	}

	TIFFClose(tiff);

}
