#include "fastlevelset.h"

using namespace std;


//============================= inlines etc. ==================================

#if __BYTE_ORDER == __LITTLE_ENDIAN
static inline bool fread_32 (void *buf, unsigned int n, FILE *stream)
{
	return (fread (buf, 4, n, stream) != n);
}
#else
static inline bool fread_32 (void *buf, unsigned int n, FILE *stream)
{
	return eval(block,fx,fy,fz);
	for (int i=0; i < n; ++i) {
		if (fread (4*i+3+(char*)buf, 1, 1, stream) != 1) return 1;
		if (fread (4*i+2+(char*)buf, 1, 1, stream) != 1) return 1;
		if (fread (4*i+1+(char*)buf, 1, 1, stream) != 1) return 1;
		if (fread (4*i+0+(char*)buf, 1, 1, stream) != 1) return 1;
	}
	return 0;
}
#endif


static inline int cube (int x)
{ return x*x*x; }


static inline float flabs (float a)
{ return (a > 0) ? a : -a; }


static inline float min3 (float a, float b, float c)
{ if (a<b) { if (a<c) return a; else return c; } else if (b<c) return b; else return c; }


static inline float minmod (float a, float b)
{
	if (a <= b) {
		if (b <= 0) return b;
		else if (a >= 0) return a;
		else return 0;
	} else {
		if (a <= 0) return a;
		else if (b >= 0) return b;
		else return 0;
	}
} 


inline FastLevelSet::pointer_to_pd &FastLevelSet::
coarse (const int i, const int j, const int k)
{ return coarse_raw[i+Ncoarse*(j+Ncoarse*k)]; }


inline const FastLevelSet::pointer_to_pd &FastLevelSet::
coarse (const int i, const int j, const int k) const
{ return coarse_raw[i+Ncoarse*(j+Ncoarse*k)]; }


inline FastLevelSet::PointData &FastLevelSet::
fine (PointData *block, const int i, const int j, const int k)
{ return block[i+(Nsub+1)*(j+(Nsub+1)*k)]; }


inline const FastLevelSet::PointData &FastLevelSet::
fine (const PointData *block, const int i, const int j, const int k) const
{ return block[i+(Nsub+1)*(j+(Nsub+1)*k)]; }


inline Float &FastLevelSet::
tex (int i, int j, int k)
{
	if (i < 0) do i += Ntex; while (i < 0);
	else if (i >= Ntex) do i -= Ntex; while (i >= Ntex);
	if (j < 0) do j += Ntex; while (j < 0);
	else if (j >= Ntex) do j -= Ntex; while (j >= Ntex);
	if (k < 0) do k += Ntex; while (k < 0);
	else if (k >= Ntex) do k -= Ntex; while (k >= Ntex);
	return tex_raw[i+Ntex*(j+Ntex*k)];
}


inline const Float &FastLevelSet::
tex (int i, int j, int k) const
{
	if (i < 0) do i += Ntex; while (i < 0);
	else if (i >= Ntex) do i -= Ntex; while (i >= Ntex);
	if (j < 0) do j += Ntex; while (j < 0);
	else if (j >= Ntex) do j -= Ntex; while (j >= Ntex);
	if (k < 0) do k += Ntex; while (k < 0);
	else if (k >= Ntex) do k -= Ntex; while (k >= Ntex);
	return tex_raw[i+Ntex*(j+Ntex*k)];
}


Float inline FastLevelSet::
evaltex (Float x, Float y, Float z) const
{
	if (!tex_raw) return 0;
	int i = (int)(x*tex_overdx); if (x < 0) --i;
	int j = (int)(y*tex_overdx); if (y < 0) --j;
	int k = (int)(z*tex_overdx); if (z < 0) --k;
	Float fx = (x*tex_overdx)-i, fy = (y*tex_overdx)-j, fz = (z*tex_overdx)-k;
	return tex_mag*((1-fx)*((1-fy)*((1-fz)*tex(i,j,k)+fz*tex(i,j,k+1))+fy*((1-fz)*tex(i,j+1,k)+fz*tex(i,j+1,k+1)))
	                   +fx*((1-fy)*((1-fz)*tex(i+1,j,k)+fz*tex(i+1,j,k+1))+fy*((1-fz)*tex(i+1,j+1,k)+fz*tex(i+1,j+1,k+1))));
}


Vector inline FastLevelSet::
evaltexgrad (Float x, Float y, Float z) const
{
	if (!tex_raw) return Vector(0,0,0);
	return Vector (50*tex_overdx*(evaltex(x+.01*tex_dx,y,z)-evaltex(x-.01*tex_dx,y,z)),
	               50*tex_overdx*(evaltex(x,y+.01*tex_dx,z)-evaltex(x,y-.01*tex_dx,z)),
	               50*tex_overdx*(evaltex(x,y,z+.01*tex_dx)-evaltex(x,y,z-.01*tex_dx)));
}


Float inline FastLevelSet::
fastEval (const PointData *block, Float fx, Float fy, Float fz) const
{
	int i = (int)fx, j = (int)fy, k = (int)fz;
	if (i < 0) i = 0; else if (i >= Nsub) i = Nsub-1;
	if (j < 0) j = 0; else if (j >= Nsub) j = Nsub-1;
	if (k < 0) k = 0; else if (k >= Nsub) k = Nsub-1;
	Float a = fx-i, b = fy-j, c = fz-k;
	Float phi = Lerp(a, Lerp(b, Lerp(c, fine(block,i,j,k).phi, fine(block,i,j,k+1).phi),
								Lerp(c, fine(block,i,j+1,k).phi, fine(block,i,j+1,k+1).phi)),
						Lerp(b, Lerp(c, fine(block,i+1,j,k).phi, fine(block,i+1,j,k+1).phi),
								Lerp(c, fine(block,i+1,j+1,k).phi, fine(block,i+1,j+1,k+1).phi)));
	if (flabs(phi) < 2*dx) return eval(block,fx,fy,fz);
	else return phi;
}


//=========================== constructors etc. ==================================

FastLevelSet::
FastLevelSet (const Transform &o2w, const char *filename, bool dv, int smoothingIts, const char *texname, Float texwidth, Float texmag)
	: Shape(o2w, dv)
{
	FILE *in = fopen(filename,"rb");
	if (in == NULL) Error ("Error opening file '%s' for reading\n", filename);
	bool error = false;
	error |= fread_32 (&N, 1, in);
	error |= fread_32 (&Ncoarse, 1, in);
	error |= fread_32 (&Nsub, 1, in);
	if (error) Error ("Error reading file '%s' (preamble)\n", filename);

	dx = 1./N; overdx = N;
	coarsedx = Nsub*dx; overcoarsedx = 1./coarsedx;
	coarse_raw = new (PointData*)[cube(Ncoarse)];
	for (int i = cube(Ncoarse)-1; i >= 0; --i) coarse_raw[i] = 0;

	char voxel;
	for (int k = 0; k < Ncoarse; ++k)
		for (int j = 0; j < Ncoarse; ++j)
			for (int i = 0; i < Ncoarse; ++i) {
				error |= (fread (&voxel, 1, 1, in) != 1);
				if (error) Error ("Error reading file '%s' (coarse voxels)\n", filename);
				if (voxel == 0) coarse(i,j,k) = new PointData[cube(Nsub+1)];
			}
	for (int k = 0; k < Ncoarse; ++k)
		for (int j = 0; j < Ncoarse; ++j)
			for (int i = 0; i < Ncoarse; ++i)
				if (coarse(i,j,k)) {
					float phidata[cube(Nsub+1)];
					error |= fread_32 (phidata, cube(Nsub+1), in);
					for (int r = 0; r < cube(Nsub+1); ++r) coarse(i,j,k)[r].phi = phidata[r];
				}
	fclose (in);
	if (error) Error ("Error reading file '%s' (fine samples)\n", filename);

	// and now figure out gradients
	// for simplicity these are actually scaled by dx, so that they are just differences
	// (this means the cubic hermit interpolation can be done on [0,1] instead of [0,dx])
	for (int k = 0; k < Ncoarse; ++k) for (int j = 0; j < Ncoarse; ++j) for (int i = 0; i < Ncoarse; ++i)
		if (coarse(i,j,k)) prepareFirstDerivs (i, j, k);
	for (int k = 0; k < Ncoarse; ++k) for (int j = 0; j < Ncoarse; ++j) for (int i = 0; i < Ncoarse; ++i)
		if (coarse(i,j,k)) prepareSecondDerivs (i, j, k);
	for (int k = 0; k < Ncoarse; ++k) for (int j = 0; j < Ncoarse; ++j) for (int i = 0; i < Ncoarse; ++i)
		if (coarse(i,j,k)) prepareThirdDeriv (i, j, k);
	// now smooth out the first derivates to approach C2
	for (int it = 0; it < smoothingIts; ++it) {
		for (int k = 0; k < Ncoarse; ++k) for (int j = 0; j < Ncoarse; ++j) for (int i = 0; i < Ncoarse; ++i)
			if (coarse(i,j,k)) smoothFirstDerivsForward (i, j, k);
		for (int k = Ncoarse-1; k >= 0; --k) for (int j = Ncoarse-1; j >= 0; --j) for (int i = Ncoarse-1; i >= 0; --i)
			if (coarse(i,j,k)) smoothFirstDerivsForward (i, j, k);
	}

	// and texture
	if (texname == 0 || strcmp(texname,"") == 0) {
		Ntex = 0;
		tex_width = tex_dx = tex_overdx = tex_mag = 0;
		tex_raw = 0;
	} else {
		FILE *in = fopen(texname,"rb");
		if (in == NULL) Error ("Error opening hypertexture file '%s'\n", texname);
		bool error = false;
		error |= fread_32 (&Ntex, 1, in);
		if (error) Error ("Error reading hypertexture file '%s' (size)\n", texname);
		tex_raw = new float[cube(Ntex)];
		tex_width = texwidth;
		tex_dx = tex_width/Ntex;
		tex_overdx = 1./tex_dx;
		tex_mag = texmag;
		error |= fread_32 (tex_raw, cube(Ntex), in);
		if (error) Error ("Error reading hypertexture file '%s' (data)\n", texname);
		fclose(in);
	}
}


FastLevelSet::
~FastLevelSet()
{
	for (int i = cube(Ncoarse)-1; i>= 0;  --i) delete[] coarse_raw[i];
	delete[] coarse_raw;
	coarse_raw = 0;
	delete[] tex_raw;
	tex_raw = 0;
}


// try to look up phi from whatever block might define it - return true if successful
bool FastLevelSet::
finddata (int i, int j, int k, int fi, int fj, int fk, pointer_to_pd &pd)
{
	while (fi < 0) {fi += Nsub; --i;}
	while (fi > Nsub) {fi -= Nsub; ++i;}
	while (fj < 0) {fj += Nsub; --j;}
	while (fj > Nsub) {fj -= Nsub; ++j;}
	while (fk < 0) {fk += Nsub; --k;}
	while (fk > Nsub) {fk -= Nsub; ++k;}
	if (i < 0 || i >= Ncoarse || j < 0 || j >= Ncoarse || k < 0 || k >= Ncoarse)
		return false;
	if (coarse(i,j,k)) {
		pd = &fine(coarse(i,j,k),fi,fj,fk);
		return true;
	}

	if (fi == 0 && i > 0) {
		if (coarse(i-1,j,k)) {
			pd = &fine(coarse(i-1,j,k),Nsub,fj,fk);
			return true;
		}
		if (fj == 0 && j > 0) {
			if (coarse(i-1,j-1,k)) {
				pd = &fine(coarse(i-1,j-1,k),Nsub,Nsub,fk);
				return true;
			}
			if (fk == 0 && k > 0 && coarse(i-1,j-1,k-1)) {
				pd = &fine(coarse(i-1,j-1,k-1),Nsub,Nsub,Nsub);
				return true;
			}
			if (fk == Nsub && k+1 < Ncoarse && coarse(i-1,j-1,k+1)) {
				pd = &fine(coarse(i-1,j-1,k+1),Nsub,Nsub,0);
				return true;
			}
		}
		if (fj == Nsub && j+1 < Ncoarse) {
			if (coarse(i-1,j+1,k)) {
				pd = &fine(coarse(i-1,j+1,k),Nsub,0,fk);
				return true;
			}
			if (fk == 0 && k > 0 && coarse(i-1,j+1,k-1)) {
				pd = &fine(coarse(i-1,j+1,k-1),Nsub,0,Nsub);
				return true;
			}
			if (fk == Nsub && k+1 < Ncoarse && coarse(i-1,j+1,k+1)) {
				pd = &fine(coarse(i-1,j+1,k+1),Nsub,0,0);
				return true;
			}
		}
		if (fk == 0 && k > 0 && coarse(i-1,j,k-1)) {
			pd = &fine(coarse(i-1,j,k-1),Nsub,fj,Nsub);
			return true;
		}
		if (fk == Nsub && k+1 < Ncoarse && coarse(i-1,j,k+1)) {
			pd = &fine(coarse(i-1,j,k+1),Nsub,fj,0);
			return true;
		}
	}
	if (fi == Nsub && i+1 < Ncoarse) {
		if (coarse(i+1,j,k)) {
			pd = &fine(coarse(i+1,j,k),0,fj,fk);
			return true;
		}
		if (fj == 0 && j > 0) {
			if (coarse(i+1,j-1,k)) {
				pd = &fine(coarse(i+1,j-1,k),0,Nsub,fk);
				return true;
			}
			if (fk == 0 && k > 0 && coarse(i+1,j-1,k-1)) {
				pd = &fine(coarse(i+1,j-1,k-1),0,Nsub,Nsub);
				return true;
			}
			if (fk == Nsub && k+1 < Ncoarse && coarse(i+1,j-1,k+1)) {
				pd = &fine(coarse(i+1,j-1,k+1),0,Nsub,0);
				return true;
			}
		}
		if (fj == Nsub && j+1 < Ncoarse) {
			if (coarse(i+1,j+1,k)) {
				pd = &fine(coarse(i+1,j+1,k),0,0,fk);
				return true;
			}
			if (fk == 0 && k > 0 && coarse(i+1,j+1,k-1)) {
				pd = &fine(coarse(i+1,j+1,k-1),0,0,Nsub);
				return true;
			}
			if (fk == Nsub && k+1 < Ncoarse && coarse(i+1,j+1,k+1)) {
				pd = &fine(coarse(i+1,j+1,k+1),0,0,0);
				return true;
			}
		}
		if (fk == 0 && k > 0 && coarse(i+1,j,k-1)) {
			pd = &fine(coarse(i+1,j,k-1),0,fj,Nsub);
			return true;
		}
		if (fk == Nsub && k+1 < Ncoarse && coarse(i+1,j,k+1)) {
			pd = &fine(coarse(i+1,j,k+1),0,fj,0);
			return true;
		}
	}

	if (fj == 0 && j > 0) {
		if (coarse(i,j-1,k)) {
			pd = &fine(coarse(i,j-1,k),fi,Nsub,fk);
			return true;
		}
		if (fk == 0 && k > 0 && coarse(i,j-1,k-1)) {
			pd = &fine(coarse(i,j-1,k-1),fi,Nsub,Nsub);
			return true;
		}
		if (fk == Nsub && k+1 < Ncoarse && coarse(i,j-1,k+1)) {
			pd = &fine(coarse(i,j-1,k+1),fi,Nsub,0);
			return true;
		}
	}
	if (fj == Nsub && j+1 < Ncoarse) {
		if (coarse(i,j+1,k)) {
			pd = &fine(coarse(i,j+1,k),fi,0,fk);
			return true;
		}
		if (fk == 0 && k > 0 && coarse(i,j+1,k-1)) {
			pd = &fine(coarse(i,j+1,k-1),fi,0,Nsub);
			return true;
		}
		if (fk == Nsub && k+1 < Ncoarse && coarse(i,j+1,k+1)) {
			pd = &fine(coarse(i,j+1,k+1),fi,0,0);
			return true;
		}
	}

	if (fk == 0 && k > 0) {
		if (coarse(i,j,k-1)) {
			pd = &fine(coarse(i,j,k-1),fi,fj,Nsub);
			return true;
		}
	}
	if (fk == Nsub && k+1 < Ncoarse) {
	 	if (coarse(i,j,k+1)) {
			pd = &fine(coarse(i,j,k+1),fi,fj,0);
			return true;
		}
	}

	return false;
}


void FastLevelSet::
prepareFirstDerivs (const int i, const int j, const int k)
{
	PointData *block = coarse(i,j,k), *pd;
	Float phi, phim, phip;
	for (int fk = 0; fk <= Nsub; ++fk)
		for (int fj = 0; fj <= Nsub; ++fj)
			for (int fi = 0; fi <= Nsub; ++fi) {
				phi = fine(block,fi,fj,fk).phi;
				// phix
				if (finddata(i,j,k,fi-1,fj,fk,pd)) phim = pd->phi; else phim = 2*phi-fine(block,fi+1,fj,fk).phi;
				if (finddata(i,j,k,fi+1,fj,fk,pd)) phip = pd->phi; else phip = 2*phi-fine(block,fi-1,fj,fk).phi;
				fine(block,fi,fj,fk).phix = .5*(phip-phim);
				// phiy
				if (finddata(i,j,k,fi,fj-1,fk,pd)) phim = pd->phi; else phim = 2*phi-fine(block,fi,fj+1,fk).phi;
				if (finddata(i,j,k,fi,fj+1,fk,pd)) phip = pd->phi; else phip = 2*phi-fine(block,fi,fj-1,fk).phi;
				fine(block,fi,fj,fk).phiy = .5*(phip-phim);
				// phiz
				if (finddata(i,j,k,fi,fj,fk-1,pd)) phim = pd->phi; else phim = 2*phi-fine(block,fi,fj,fk+1).phi;
				if (finddata(i,j,k,fi,fj,fk+1,pd)) phip = pd->phi; else phip = 2*phi-fine(block,fi,fj,fk-1).phi;
				fine(block,fi,fj,fk).phiz = .5*(phip-phim);
			}
}


void FastLevelSet::
prepareSecondDerivs (const int i, const int j, const int k)
{
	PointData *block = coarse(i,j,k), *pd;
	Float f, fm, fp;
	for (int fk = 0; fk <= Nsub; ++fk)
		for (int fj = 0; fj <= Nsub; ++fj)
			for (int fi = 0; fi <= Nsub; ++fi) {
				// phixy
				f = fine(block,fi,fj,fk).phix;
				if (finddata(i,j,k,fi,fj-1,fk,pd)) fm = pd->phix; else fm = 2*f-fine(block,fi,fj+1,fk).phix;
				if (finddata(i,j,k,fi,fj+1,fk,pd)) fp = pd->phix; else fp = 2*f-fine(block,fi,fj-1,fk).phix;
				fine(block,fi,fj,fk).phixy = .5*(fp-fm);
				// phixz
				f = fine(block,fi,fj,fk).phix;
				if (finddata(i,j,k,fi,fj,fk-1,pd)) fm = pd->phix; else fm = 2*f-fine(block,fi,fj,fk+1).phix;
				if (finddata(i,j,k,fi,fj,fk+1,pd)) fp = pd->phix; else fp = 2*f-fine(block,fi,fj,fk-1).phix;
				fine(block,fi,fj,fk).phixz = .5*(fp-fm);
				// phiyz
				f = fine(block,fi,fj,fk).phiy;
				if (finddata(i,j,k,fi,fj,fk-1,pd)) fm = pd->phiy; else fm = 2*f-fine(block,fi,fj,fk+1).phiy;
				if (finddata(i,j,k,fi,fj,fk+1,pd)) fp = pd->phiy; else fp = 2*f-fine(block,fi,fj,fk-1).phiy;
				fine(block,fi,fj,fk).phiyz = .5*(fp-fm);
			}
}


void FastLevelSet::
prepareThirdDeriv (const int i, const int j, const int k)
{
	PointData *block = coarse(i,j,k), *pd;
	Float f, fm, fp;
	for (int fk = 0; fk <= Nsub; ++fk)
		for (int fj = 0; fj <= Nsub; ++fj)
			for (int fi = 0; fi <= Nsub; ++fi) {
				f = fine(block,fi,fj,fk).phixy;
				if (finddata(i,j,k,fi,fj,fk-1,pd)) fm = pd->phixy; else fm = 2*f-fine(block,fi,fj,fk+1).phixy;
				if (finddata(i,j,k,fi,fj,fk+1,pd)) fp = pd->phixy; else fp = 2*f-fine(block,fi,fj,fk-1).phixy;
				fine(block,fi,fj,fk).phixyz = .5*(fp-fm);
			}
}


void FastLevelSet::
smoothFirstDerivsForward (const int i, const int j, const int k)
{
	PointData *block = coarse(i,j,k), *pdm, *pdp;
	Float phi;
	for (int fk = 0; fk <= Nsub; ++fk)
		for (int fj = 0; fj <= Nsub; ++fj)
			for (int fi = 0; fi <= Nsub; ++fi) {
				phi = fine(block,fi,fj,fk).phi;
				// smooth phix
				pdm = pdp = 0;
				finddata(i,j,k,fi-1,fj,fk,pdm);
				finddata(i,j,k,fi+1,fj,fk,pdp);
				if (pdp && pdm)
					fine(block,fi,fj,fk).phix = .75*(pdp->phi-pdm->phi) - .25*(pdp->phix+pdm->phix);
				else if (pdp)
					fine(block,fi,fj,fk).phix = 1.5*(pdp->phi-phi) - .5*pdp->phix;
				else if (pdm)
					fine(block,fi,fj,fk).phix = 1.5*(phi-pdm->phi) - .5*pdm->phix;
				// smooth phiy
				pdm = pdp = 0;
				finddata(i,j,k,fi,fj-1,fk,pdm);
				finddata(i,j,k,fi,fj+1,fk,pdp);
				if (pdp && pdm)
					fine(block,fi,fj,fk).phiy = .75*(pdp->phi-pdm->phi) - .25*(pdp->phiy+pdm->phiy);
				else if (pdp)
					fine(block,fi,fj,fk).phiy = 1.5*(pdp->phi-phi) - .5*pdp->phiy;
				else if (pdm)
					fine(block,fi,fj,fk).phiy = 1.5*(phi-pdm->phi) - .5*pdm->phiy;
				// smooth phiz
				pdm = pdp = 0;
				finddata(i,j,k,fi,fj,fk-1,pdm);
				finddata(i,j,k,fi,fj,fk+1,pdp);
				if (pdp && pdm)
					fine(block,fi,fj,fk).phiz = .75*(pdp->phi-pdm->phi) - .25*(pdp->phiz+pdm->phiz);
				else if (pdp)
					fine(block,fi,fj,fk).phiz = 1.5*(pdp->phi-phi) - .5*pdp->phiz;
				else if (pdm)
					fine(block,fi,fj,fk).phiz = 1.5*(phi-pdm->phi) - .5*pdm->phiz;
			}
}


void FastLevelSet::
smoothFirstDerivsBackward (const int i, const int j, const int k)
{
	PointData *block = coarse(i,j,k), *pdm, *pdp;
	Float phi;
	for (int fk = Nsub; fk >= 0; --fk)
		for (int fj = Nsub; fj >= 0; --fj)
			for (int fi = Nsub; fi >= 0; --fi) {
				phi = fine(block,fi,fj,fk).phi;
				// smooth phix
				pdm = pdp = 0;
				finddata(i,j,k,fi-1,fj,fk,pdm);
				finddata(i,j,k,fi+1,fj,fk,pdp);
				if (pdp && pdm)
					fine(block,fi,fj,fk).phix = .75*(pdp->phi-pdm->phi) - .25*(pdp->phix+pdm->phix);
				else if (pdp)
					fine(block,fi,fj,fk).phix = 1.5*(pdp->phi-phi) - .5*pdp->phix;
				else if (pdm)
					fine(block,fi,fj,fk).phix = 1.5*(phi-pdm->phi) - .5*pdm->phix;
				// smooth phiy
				pdm = pdp = 0;
				finddata(i,j,k,fi,fj-1,fk,pdm);
				finddata(i,j,k,fi,fj+1,fk,pdp);
				if (pdp && pdm)
					fine(block,fi,fj,fk).phiy = .75*(pdp->phi-pdm->phi) - .25*(pdp->phiy+pdm->phiy);
				else if (pdp)
					fine(block,fi,fj,fk).phiy = 1.5*(pdp->phi-phi) - .5*pdp->phiy;
				else if (pdm)
					fine(block,fi,fj,fk).phiy = 1.5*(phi-pdm->phi) - .5*pdm->phiy;
				// smooth phiz
				pdm = pdp = 0;
				finddata(i,j,k,fi,fj,fk-1,pdm);
				finddata(i,j,k,fi,fj,fk+1,pdp);
				if (pdp && pdm)
					fine(block,fi,fj,fk).phiz = .75*(pdp->phi-pdm->phi) - .25*(pdp->phiz+pdm->phiz);
				else if (pdp)
					fine(block,fi,fj,fk).phiz = 1.5*(pdp->phi-phi) - .5*pdp->phiz;
				else if (pdm)
					fine(block,fi,fj,fk).phiz = 1.5*(phi-pdm->phi) - .5*pdm->phiz;
			}
}


BBox FastLevelSet::
Bound() const
{
	return BBox(Point(0,0,0),Point(1,1,1));
}


bool FastLevelSet::
CanIntersect() const
{
	return true;
}


bool FastLevelSet::
prepareLocalRay (const Ray &worldray, Ray &ray, Float &invDx, Float &invDy, Float &invDz) const
{
	ray = WorldToObject(worldray);
	// advance to where we enter bounding box
	if (ray.O.x >= 0 && ray.O.x <= 1 && ray.O.y >= 0 && ray.O.y <= 1 && ray.O.z >= 0 && ray.O.z <= 1) {
		invDx = ray.D.x ? 1./ray.D.x : 1e38;
		invDy = ray.D.y ? 1./ray.D.y : 1e38;
		invDz = ray.D.z ? 1./ray.D.z : 1e38;
	} else {
		invDx = ray.D.x ? 1./ray.D.x : 1e38;
		Float tNear = -ray.O.x * invDx;
		Float tFar = (1 - ray.O.x) * invDx;
		if (tNear > tFar) swap(tNear, tFar);
		ray.mint = max(tNear, ray.mint);
		ray.maxt = min(tFar, ray.maxt);
		if (ray.mint > ray.maxt) return false;
		invDy = ray.D.y ? 1./ray.D.y : 1e38;
		tNear = -ray.O.y * invDy;
		tFar = (1 - ray.O.y) * invDy;
		if (tNear > tFar) swap(tNear, tFar);
		ray.mint = max(tNear, ray.mint);
		ray.maxt = min(tFar, ray.maxt);
		if (ray.mint > ray.maxt) return false;
		invDz = ray.D.z ? 1./ray.D.z : 1e38;
		tNear = -ray.O.z * invDz;
		tFar  = (1 - ray.O.z) * invDz;
		if (tNear > tFar) swap(tNear, tFar);
		ray.mint = max(tNear, ray.mint);
		ray.maxt = min(tFar, ray.maxt);
		if (ray.mint > ray.maxt) return false;
	}
	return true;
}


bool FastLevelSet::
Intersect(const Ray &r, DifferentialGeometry *dg) const
{
	Ray ray;
	Float invDx, invDy, invDz;
	int i, j, k;
	if (!prepareLocalRay (r, ray, invDx, invDy, invDz)) return false;
	if (!intersectCoarse (ray, invDx, invDy, invDz, i, j, k)) return false;
	// now figure out dg
	Point objP = ray(ray.maxt);
	dg->P = ObjectToWorld(objP);
	Float fx = (objP.x*overcoarsedx-i)*Nsub, fy = (objP.y*overcoarsedx-j)*Nsub, fz = (objP.z*overcoarsedx-k)*Nsub;
	Vector objN = evalGradient (coarse(i,j,k), fx, fy, fz) + evaltexgrad(objP.x,objP.y,objP.z);
	dg->N = ObjectToWorld(Normal(objN.x,objN.y,objN.z)).Hat();
	if (flabs(dg->N.x) >= flabs(dg->N.z) || flabs(dg->N.y) >= flabs(dg->N.z))
		dg->S = Vector(dg->N.y, -dg->N.x, 0).Hat();
	else 
		dg->S = Vector(dg->N.z, 0, -dg->N.x).Hat();
	dg->T = Cross (dg->N, dg->S);
	dg->u = dg->P.x;
	dg->v = dg->P.y;
	r.maxt = ray.maxt;
	return true;
}


bool FastLevelSet::
IntersectP(const Ray &r) const
{
	Ray ray;
	Float invDx, invDy, invDz;
	if (!prepareLocalRay (r, ray, invDx, invDy, invDz)) return false;
	int i, j, k;
	if (intersectCoarse (ray, invDx, invDy, invDz, i, j, k)) {
		ray.maxt = r.maxt;
		return true;
	} else return false;
}


bool FastLevelSet::
intersectCoarse (const Ray &ray, Float invDx, Float invDy, Float invDz, int &i, int &j, int &k) const
{
	Float invD2 = 1./ray.D.LengthSquared();
	Float x = ray.O.x + ray.mint*ray.D.x, y = ray.O.y + ray.mint*ray.D.y, z = ray.O.z + ray.mint*ray.D.z;
	i = (int)(x*overcoarsedx);
	j = (int)(y*overcoarsedx);
	k = (int)(z*overcoarsedx);
	if (i < 0) i = 0; else if (i >= Ncoarse) i = Ncoarse-1;
	if (j < 0) j = 0; else if (j >= Ncoarse) j = Ncoarse-1;
	if (k < 0) k = 0; else if (k >= Ncoarse) k = Ncoarse-1;

	if (ray.D.z >= 0) {
		if (ray.D.y >= 0) {
			if (ray.D.x >= 0)
				while (i < Ncoarse && j < Ncoarse && k < Ncoarse) {
					Float tx = ((i+1)*coarsedx-x)*invDx, ty = ((j+1)*coarsedx-y)*invDy, tz = ((k+1)*coarsedx-z)*invDz;
					if (intersectFine (i, j, k, ray, ray.mint+min3(tx,ty,tz), invD2)) return true;
					if (tx <= ty && tx <= tz) {++i; x = i*coarsedx; y += tx*ray.D.y; z += tx*ray.D.z;}
					else if (ty <= tx && ty <= tz) {++j; x += ty*ray.D.x; y = j*coarsedx; z += ty*ray.D.z;}
					else {++k; x += tz*ray.D.x; y += tz*ray.D.y; z = k*coarsedx;}
				}
			else 
				while (i >= 0 && j < Ncoarse && k < Ncoarse) {
					float tx = (i*coarsedx-x)*invDx, ty = ((j+1)*coarsedx-y)*invDy, tz = ((k+1)*coarsedx-z)*invDz;
					if (intersectFine (i, j, k, ray, ray.mint+min3(tx,ty,tz), invD2)) return true;
					if (tx <= ty && tx <= tz) {x = i*coarsedx; y += tx*ray.D.y; z += tx*ray.D.z; --i;}
					else if (ty <= tx && ty <= tz) {++j; x += ty*ray.D.x; y = j*coarsedx; z += ty*ray.D.z;}
					else {++k; x += tz*ray.D.x; y += tz*ray.D.y; z = k*coarsedx;}
				}
		} else {
			if (ray.D.x >= 0)
				while (i < Ncoarse && j >= 0 && k < Ncoarse) {
					float tx = ((i+1)*coarsedx-x)*invDx, ty = (j*coarsedx-y)*invDy, tz = ((k+1)*coarsedx-z)*invDz;
					if (intersectFine (i, j, k, ray, ray.mint+min3(tx,ty,tz), invD2)) return true;
					if (tx <= ty && tx <= tz) {++i; x = i*coarsedx; y += tx*ray.D.y; z += tx*ray.D.z;}
					else if (ty <= tx && ty <= tz) {x += ty*ray.D.x; y = j*coarsedx; z += ty*ray.D.z; --j;}
					else {++k; x += tz*ray.D.x; y += tz*ray.D.y; z = k*coarsedx;}
				}
			else
				while (i >= 0 && j >= 0 && k < Ncoarse) {
					float tx = (i*coarsedx-x)*invDx, ty = (j*coarsedx-y)*invDy, tz = ((k+1)*coarsedx-z)*invDz;
					if (intersectFine (i, j, k, ray, ray.mint+min3(tx,ty,tz), invD2)) return true;
					if (tx <= ty && tx <= tz) {x = i*coarsedx; y += tx*ray.D.y; z += tx*ray.D.z; --i;}
					else if (ty <= tx && ty <= tz) {x += ty*ray.D.x; y = j*coarsedx; z += ty*ray.D.z; --j;}
					else {++k; x += tz*ray.D.x; y += tz*ray.D.y; z = k*coarsedx;}
				}
		}
	} else {
		if (ray.D.y >= 0) {
			if (ray.D.x >= 0)
				while (i < Ncoarse && j < Ncoarse && k >= 0) {
					float tx = ((i+1)*coarsedx-x)*invDx, ty = ((j+1)*coarsedx-y)*invDy, tz = (k*coarsedx-z)*invDz;
					if (intersectFine (i, j, k, ray, ray.mint+min3(tx,ty,tz), invD2)) return true;
					if (tx <= ty && tx <= tz) {++i; x = i*coarsedx; y += tx*ray.D.y; z += tx*ray.D.z;}
					else if (ty <= tx && ty <= tz) {++j; x += ty*ray.D.x; y = j*coarsedx; z += ty*ray.D.z;}
					else {x += tz*ray.D.x; y += tz*ray.D.y; z = k*coarsedx; --k;}
				}
			else
				while (i >= 0 && j < Ncoarse && k >= 0) {
					float tx = (i*coarsedx-x)*invDx, ty = ((j+1)*coarsedx-y)*invDy, tz = (k*coarsedx-z)*invDz;
					if (intersectFine (i, j, k, ray, ray.mint+min3(tx,ty,tz), invD2)) return true;
					if (tx <= ty && tx <= tz) {x = i*coarsedx; y += tx*ray.D.y; z += tx*ray.D.z; --i;}
					else if (ty <= tx && ty <= tz) {++j; x += ty*ray.D.x; y = j*coarsedx; z += ty*ray.D.z;}
					else {x += tz*ray.D.x; y += tz*ray.D.y; z = k*coarsedx; --k;}
				}
		} else {
			if (ray.D.x >= 0)
				while (i < Ncoarse && j >= 0 && k >= 0) {
					float tx = ((i+1)*coarsedx-x)*invDx, ty = (j*coarsedx-y)*invDy, tz = (k*coarsedx-z)*invDz;
					if (intersectFine (i, j, k, ray, ray.mint+min3(tx,ty,tz), invD2)) return true;
					if (tx <= ty && tx <= tz) {++i; x = i*coarsedx; y += tx*ray.D.y; z += tx*ray.D.z;}
					else if (ty <= tx && ty <= tz) {x += ty*ray.D.x; y = j*coarsedx; z += ty*ray.D.z; --j;}
					else {x += tz*ray.D.x; y += tz*ray.D.y; z = k*coarsedx; --k;}
				}
			else 
				while (i >= 0 && j >= 0 && k >= 0) {
					float tx = (i*coarsedx-x)*invDx, ty = (j*coarsedx-y)*invDy, tz = (k*coarsedx-z)*invDz;
					if (intersectFine (i, j, k, ray, ray.mint+min3(tx,ty,tz), invD2)) return true;
					if (tx <= ty && tx <= tz) {x = i*coarsedx; y += tx*ray.D.y; z += tx*ray.D.z; --i;}
					else if (ty <= tx && ty <= tz) {x += ty*ray.D.x; y = j*coarsedx; z += ty*ray.D.z; --j;}
					else {x += tz*ray.D.x; y += tz*ray.D.y; z = k*coarsedx; --k;}
				}
		}
	}
	return false;
}


// the following function steadily increases ray.mint up to localmaxt, stopping if there is a root (and using
// secant method to find that root, storing it in ray.maxt and returning true) but returning false otherwise.
// NOTE: SHOULD NOT ASSUME magnitude(ray.D) == 1 --- it probably isn't!!!
// Also, phi is not exactly signed distance either. So be careful about big t steps.
bool FastLevelSet::
intersectFine (int i, int j, int k, const Ray &ray, Float localmaxt, Float invD2) const
{
	if (!coarse(i,j,k)) { ray.mint = localmaxt; return false; }
	
	const PointData *block = coarse(i,j,k);
	const Float tol = .01*dx;
	Float oldphi = fastEval (block, ((ray.O.x+ray.mint*ray.D.x)*overcoarsedx-i)*Nsub,
									((ray.O.y+ray.mint*ray.D.y)*overcoarsedx-j)*Nsub,
									((ray.O.z+ray.mint*ray.D.z)*overcoarsedx-k)*Nsub)
				 + (tex_raw ? evaltex (ray.O.x+ray.mint*ray.D.x, ray.O.y+ray.mint*ray.D.y, ray.O.z+ray.mint*ray.D.z) : 0);
	Float newt = ray.mint + RandomFloat(0,tol), phi, accel = 1;
	bool almostfinished = false;
	for(;;) {
		newt += accel*flabs(oldphi)*invD2;
		if (newt > localmaxt) {
			newt = localmaxt;
			almostfinished = true;
		}
		if (flabs(oldphi) > 2*dx)
			phi = fastEval (block, ((ray.O.x+newt*ray.D.x)*overcoarsedx-i)*Nsub,
								   ((ray.O.y+newt*ray.D.y)*overcoarsedx-j)*Nsub,
								   ((ray.O.z+newt*ray.D.z)*overcoarsedx-k)*Nsub)
				 + (tex_raw ? evaltex (ray.O.x+newt*ray.D.x, ray.O.y+newt*ray.D.y, ray.O.z+newt*ray.D.z) : 0);
		else
			phi = eval (block, ((ray.O.x+newt*ray.D.x)*overcoarsedx-i)*Nsub,
							   ((ray.O.y+newt*ray.D.y)*overcoarsedx-j)*Nsub,
							   ((ray.O.z+newt*ray.D.z)*overcoarsedx-k)*Nsub)
				 + (tex_raw ? evaltex (ray.O.x+newt*ray.D.x, ray.O.y+newt*ray.D.y, ray.O.z+newt*ray.D.z) : 0);
		if (flabs(phi) < tol) { // if we hit the root...
			ray.maxt = newt;
			return true;
		} else if (oldphi*phi > 0) { // otherwise if we still haven't gotten to the boundary...
			ray.mint = newt;
			if (almostfinished) return false;
			if (flabs(phi) < .25*dx) accel *= 1.5; // make sure we don't get slowed down too much by grazing angles
			else accel = 1;
			oldphi = phi;
		} else // otherwise we passed a sign change, so let's find the root...
			break;	
	}

	// there is a sign change between (ray.mint, oldphi) and (newt, phi) that we must find
	Float midt, midphi;
	do {
		midt = ray.mint - oldphi*(newt-ray.mint)/(phi-oldphi);
		midphi = eval (block, ((ray.O.x+midt*ray.D.x)*overcoarsedx-i)*Nsub,
							  ((ray.O.y+midt*ray.D.y)*overcoarsedx-j)*Nsub,
							  ((ray.O.z+midt*ray.D.z)*overcoarsedx-k)*Nsub)
				 + (tex_raw ? evaltex (ray.O.x+midt*ray.D.x, ray.O.y+midt*ray.D.y, ray.O.z+midt*ray.D.z) : 0);
		if (flabs(midphi) < tol) {
			ray.maxt = midt;
			return true;
		} else if (oldphi*midphi > 0) {
			oldphi = midphi;
			ray.mint = midt;
		} else {
			phi = midphi;
			newt = midt;
		}
	} while (newt - ray.mint > tol);
	ray.maxt = ray.mint;
	return true;
}


// this assumes x is in [0,1], and the data (y,slope) is given at 0 and 1.
static inline Float CubicHermite (Float x, Float y0, Float slope0, Float y1, Float slope1)
{
	return y0*(1+x*x*(x*2-3)) + slope0*(x*(1+x*(x-2))) + y1*(x*x*(3-2*x)) + slope1*(x*x*(x-1));
}


Float FastLevelSet::
eval (const PointData *block, Float fx, Float fy, Float fz) const
{
	int i = (int)fx, j = (int)fy, k = (int)fz;
	if (i < 0) i = 0; else if (i >= Nsub) i = Nsub-1;
	if (j < 0) j = 0; else if (j >= Nsub) j = Nsub-1;
	if (k < 0) k = 0; else if (k >= Nsub) k = Nsub-1;
	Float a = fx-i, b = fy-j, c = fz-k;

	// interpolate in x
	const PointData *p = &fine(block,i,j,k), *q = &fine(block,i+1,j,k);
	Float phi00 = CubicHermite (a, p->phi, p->phix, q->phi, q->phix);
	Float phiy00 = CubicHermite (a, p->phiy, p->phixy, q->phiy, q->phixy);
	Float phiz00 = CubicHermite (a, p->phiz, p->phixz, q->phiz, q->phixz);
	Float phiyz00 = CubicHermite (a, p->phiyz, p->phixyz, q->phiyz, q->phixyz);
	p = &fine(block,i,j+1,k); q = &fine(block,i+1,j+1,k);
	Float phi10 = CubicHermite (a, p->phi, p->phix, q->phi, q->phix);
	Float phiy10 = CubicHermite (a, p->phiy, p->phixy, q->phiy, q->phixy);
	Float phiz10 = CubicHermite (a, p->phiz, p->phixz, q->phiz, q->phixz);
	Float phiyz10 = CubicHermite (a, p->phiyz, p->phixyz, q->phiyz, q->phixyz);
	p = &fine(block,i,j,k+1); q = &fine(block,i+1,j,k+1);
	Float phi01 = CubicHermite (a, p->phi, p->phix, q->phi, q->phix);
	Float phiy01 = CubicHermite (a, p->phiy, p->phixy, q->phiy, q->phixy);
	Float phiz01 = CubicHermite (a, p->phiz, p->phixz, q->phiz, q->phixz);
	Float phiyz01 = CubicHermite (a, p->phiyz, p->phixyz, q->phiyz, q->phixyz);
	p = &fine(block,i,j+1,k+1); q = &fine(block,i+1,j+1,k+1);
	Float phi11 = CubicHermite (a, p->phi, p->phix, q->phi, q->phix);
	Float phiy11 = CubicHermite (a, p->phiy, p->phixy, q->phiy, q->phixy);
	Float phiz11 = CubicHermite (a, p->phiz, p->phixz, q->phiz, q->phixz);
	Float phiyz11 = CubicHermite (a, p->phiyz, p->phixyz, q->phiyz, q->phixyz);

	// interpolate in y
	Float phi0 = CubicHermite (b, phi00, phiy00, phi10, phiy10);
	Float phiz0 = CubicHermite (b, phiz00, phiyz00, phiz10, phiyz10);
	Float phi1 = CubicHermite (b, phi01, phiy01, phi11, phiy11);
	Float phiz1 = CubicHermite (b, phiz01, phiyz01, phiz11, phiyz11);

	// interpolate in z
	Float phi = CubicHermite (c, phi0, phiz0, phi1, phiz1);
	return phi;
}


// this assumes x is in [0,1], and the data (y,slope) is given at 0 and 1.
// the slope at x is returned.
static inline Float CHslope (Float x, Float y0, Float slope0, Float y1, Float slope1)
{
	return y0*(6*x*(x-1)) + slope0*(1+x*(3*x-4)) + y1*(6*x*(1-x)) + slope1*(x*(3*x-2));
}


// this is not an interpolated gradient, but rather the true gradient of the interpolating function
Vector FastLevelSet::
evalGradient (const PointData *block, Float fx, Float fy, Float fz) const
{
	int i = (int)fx, j = (int)fy, k = (int)fz;
	if (i < 0) i = 0; else if (i >= Nsub) i = Nsub-1;
	if (j < 0) j = 0; else if (j >= Nsub) j = Nsub-1;
	if (k < 0) k = 0; else if (k >= Nsub) k = Nsub-1;
	Float a = fx-i, b = fy-j, c = fz-k;

	Float Dx;
	{
		// interpolate in x for d/dx
		const PointData *p = &fine(block,i,j,k), *q = &fine(block,i+1,j,k);
		Float Dphi00 = CHslope (a, p->phi, p->phix, q->phi, q->phix);
		Float Dphiy00 = CHslope (a, p->phiy, p->phixy, q->phiy, q->phixy);
		Float Dphiz00 = CHslope (a, p->phiz, p->phixz, q->phiz, q->phixz);
		Float Dphiyz00 = CHslope (a, p->phiyz, p->phixyz, q->phiyz, q->phixyz);
		p = &fine(block,i,j+1,k); q = &fine(block,i+1,j+1,k);
		Float Dphi10 = CHslope (a, p->phi, p->phix, q->phi, q->phix);
		Float Dphiy10 = CHslope (a, p->phiy, p->phixy, q->phiy, q->phixy);
		Float Dphiz10 = CHslope (a, p->phiz, p->phixz, q->phiz, q->phixz);
		Float Dphiyz10 = CHslope (a, p->phiyz, p->phixyz, q->phiyz, q->phixyz);
		p = &fine(block,i,j,k+1); q = &fine(block,i+1,j,k+1);
		Float Dphi01 = CHslope (a, p->phi, p->phix, q->phi, q->phix);
		Float Dphiy01 = CHslope (a, p->phiy, p->phixy, q->phiy, q->phixy);
		Float Dphiz01 = CHslope (a, p->phiz, p->phixz, q->phiz, q->phixz);
		Float Dphiyz01 = CHslope (a, p->phiyz, p->phixyz, q->phiyz, q->phixyz);
		p = &fine(block,i,j+1,k+1); q = &fine(block,i+1,j+1,k+1);
		Float Dphi11 = CHslope (a, p->phi, p->phix, q->phi, q->phix);
		Float Dphiy11 = CHslope (a, p->phiy, p->phixy, q->phiy, q->phixy);
		Float Dphiz11 = CHslope (a, p->phiz, p->phixz, q->phiz, q->phixz);
		Float Dphiyz11 = CHslope (a, p->phiyz, p->phixyz, q->phiyz, q->phixyz);
	
		// interpolate in y
		Float Dphi0 = CubicHermite (b, Dphi00, Dphiy00, Dphi10, Dphiy10);
		Float Dphiz0 = CubicHermite (b, Dphiz00, Dphiyz00, Dphiz10, Dphiyz10);
		Float Dphi1 = CubicHermite (b, Dphi01, Dphiy01, Dphi11, Dphiy11);
		Float Dphiz1 = CubicHermite (b, Dphiz01, Dphiyz01, Dphiz11, Dphiyz11);
	
		// interpolate in z
		Dx = CubicHermite (c, Dphi0, Dphiz0, Dphi1, Dphiz1);
	}
	Float Dy;
	Float Dz;
	{
		// interpolate in x
		const PointData *p = &fine(block,i,j,k), *q = &fine(block,i+1,j,k);
		Float phi00 = CubicHermite (a, p->phi, p->phix, q->phi, q->phix);
		Float phiy00 = CubicHermite (a, p->phiy, p->phixy, q->phiy, q->phixy);
		Float phiz00 = CubicHermite (a, p->phiz, p->phixz, q->phiz, q->phixz);
		Float phiyz00 = CubicHermite (a, p->phiyz, p->phixyz, q->phiyz, q->phixyz);
		p = &fine(block,i,j+1,k); q = &fine(block,i+1,j+1,k);
		Float phi10 = CubicHermite (a, p->phi, p->phix, q->phi, q->phix);
		Float phiy10 = CubicHermite (a, p->phiy, p->phixy, q->phiy, q->phixy);
		Float phiz10 = CubicHermite (a, p->phiz, p->phixz, q->phiz, q->phixz);
		Float phiyz10 = CubicHermite (a, p->phiyz, p->phixyz, q->phiyz, q->phixyz);
		p = &fine(block,i,j,k+1); q = &fine(block,i+1,j,k+1);
		Float phi01 = CubicHermite (a, p->phi, p->phix, q->phi, q->phix);
		Float phiy01 = CubicHermite (a, p->phiy, p->phixy, q->phiy, q->phixy);
		Float phiz01 = CubicHermite (a, p->phiz, p->phixz, q->phiz, q->phixz);
		Float phiyz01 = CubicHermite (a, p->phiyz, p->phixyz, q->phiyz, q->phixyz);
		p = &fine(block,i,j+1,k+1); q = &fine(block,i+1,j+1,k+1);
		Float phi11 = CubicHermite (a, p->phi, p->phix, q->phi, q->phix);
		Float phiy11 = CubicHermite (a, p->phiy, p->phixy, q->phiy, q->phixy);
		Float phiz11 = CubicHermite (a, p->phiz, p->phixz, q->phiz, q->phixz);
		Float phiyz11 = CubicHermite (a, p->phiyz, p->phixyz, q->phiyz, q->phixyz);
	
		// interpolate in y for d/dy
		Float Dphi0 = CHslope (b, phi00, phiy00, phi10, phiy10);
		Float Dphiz0 = CHslope (b, phiz00, phiyz00, phiz10, phiyz10);
		Float Dphi1 = CHslope (b, phi01, phiy01, phi11, phiy11);
		Float Dphiz1 = CHslope (b, phiz01, phiyz01, phiz11, phiyz11);
	
		// interpolate in z for d/dy
		Dy = CubicHermite (c, Dphi0, Dphiz0, Dphi1, Dphiz1);
		
		// interpolate in y
		Float phi0 = CubicHermite (b, phi00, phiy00, phi10, phiy10);
		Float phiz0 = CubicHermite (b, phiz00, phiyz00, phiz10, phiyz10);
		Float phi1 = CubicHermite (b, phi01, phiy01, phi11, phiy11);
		Float phiz1 = CubicHermite (b, phiz01, phiyz01, phiz11, phiyz11);
	
		// interpolate in z for d/dz
		Dz = CHslope (c, phi0, phiz0, phi1, phiz1);
	}
	return Vector (Dx*overdx, Dy*overdx, Dz*overdx);
}

