#include "image.h"
#include "camera.h"
#include "scene.h"
#include "geometry.h"
#include <unistd.h>
#include <errno.h>
Image::Image(int xres, int yres, Float cropx0, Float cropx1,
		Float cropy0, Float cropy1, bool doRGB, bool doAlpha,
		Imager *im, Float gain, Float gamma, int colQuantOne,
		int colQuantMin, int colQuantMax, Float colQuantDither,
		char *display, Sampler *samp) {
	XResolution = xres;
	YResolution = yres;
	CropLeft = cropx0;
	CropBottom = cropy0;
	CropRight = cropx1;
	CropTop = cropy1;
	Gain = gain;
	Gamma = gamma;
	imager = im;
	ColorQuantOne = colQuantOne;
	ColorQuantMin = colQuantMin;
	ColorQuantMax = colQuantMax;
	ColorQuantDither = colQuantDither;
	DisplayName = Strdup(display);
	ImageViewer = NULL;
	sampler = samp;
	int RasterCropLeft = Clamp((int)ceil(XResolution * CropLeft),
		0, XResolution);
	int RasterCropRight =  Clamp((int)ceil(XResolution * CropRight),
		0, XResolution);
	int RasterCropBottom = Clamp((int)ceil(YResolution * CropBottom),
		0, YResolution);
	int RasterCropTop =    Clamp((int)ceil(YResolution * CropTop),
		0, YResolution);
	SampleCropLeft =   Clamp(int(RasterCropLeft   - sampler->FilterXWidth), 0,
		XResolution);
	SampleCropRight =  Clamp(int(RasterCropRight  + sampler->FilterXWidth), 0,
		XResolution);
	SampleCropBottom = Clamp(int(RasterCropBottom - sampler->FilterYWidth), 0,
		YResolution);
	SampleCropTop =    Clamp(int(RasterCropTop    + sampler->FilterYWidth), 0,
		YResolution);
	XBase  = Clamp((int)ceil(XResolution * CropLeft),
		0, XResolution);
	XWidth = Clamp((int)ceil(XResolution * CropRight),
		0, XResolution) - XBase;
	YBase  = Clamp((int)ceil(YResolution * CropBottom),
		0, YResolution);
	YWidth = Clamp((int)ceil(YResolution * CropTop),
		0, YResolution) - YBase;
	if (doRGB) Pixels = new Spectrum[XWidth * YWidth];
	else Pixels = NULL;
	if (doAlpha) {
		Alphas = new Float[XWidth * YWidth];
		for (int i = 0; i < XWidth * YWidth; ++i)
			Alphas[i] = 0.;
	}
	else Alphas = NULL;
	WeightSums = new Float[XWidth * YWidth];
	for (int i = 0; i < XWidth * YWidth; ++i)
		WeightSums[i] = 0.;
}
Image::~Image() {
	delete imager;
	delete[] DisplayName;
	delete[] Pixels;
	delete[] Alphas;
	delete[] WeightSums;
}
void Image::AddSampleBasic(const Point &Praster,
	const Spectrum &radiance, Float alpha) {
	if (Praster.x < XBase || Praster.x >= XBase + XWidth ||
		Praster.y < YBase || Praster.y >= YBase + YWidth)
		return;
	int x = int(Praster.x), y = int(Praster.y);
	int offset = (y - YBase) * XWidth + (x - XBase);
	if (Pixels) Pixels[offset] += radiance;
	if (Alphas) Alphas[offset] += alpha;
	WeightSums[offset] += 1;
}
void Image::Write() const {
	Float *RGBOut = NULL, *AlphaOut = NULL;
	if (Pixels) {
		RGBOut = new Float[3 * XWidth * YWidth];
		for (int offset = 0; offset < XWidth * YWidth; ++offset)
			Pixels[offset].ConvertToRGB(&RGBOut[3 * offset]);
	}
	if (Alphas) {
		AlphaOut = new Float[XWidth * YWidth];
		memcpy(AlphaOut, Alphas, XWidth * YWidth * sizeof(Float));
	}
	for (int offset = 0; offset < XWidth * YWidth; ++offset) {
		if (WeightSums[offset] == 0.)
			continue;
	
		Float invWt = 1. / WeightSums[offset];
		if (RGBOut) {
			for (int c = 0; c < 3; ++c) {
				RGBOut[3*offset + c] *= invWt;
				if (RGBOut[3*offset + c] < 0.)
					RGBOut[3*offset + c] = 0.;
			}
		}
		if (AlphaOut) AlphaOut[offset] *= invWt;
	}
	if (RGBOut && AlphaOut) {
		for (int i = 0; i < XWidth * YWidth; ++i) {
			for (int j = 0; j < 3; ++j)
				RGBOut[3*i+j] *= AlphaOut[i];
		}
	}
	// XXX
	if (imager)
		; // XXXX
	if ((RGBOut || AlphaOut) && (Gain != 1. || Gamma != 1.)) {
		Float invGamma = 1. / Gamma;
		if (RGBOut)
			for (int offset = 0; offset < 3 * XWidth * YWidth; ++offset)
				RGBOut[offset] = pow(RGBOut[offset] * Gain, invGamma);
		if (AlphaOut)
			for (int offset = 0; offset < XWidth * YWidth; ++offset)
				AlphaOut[offset] =
					pow(AlphaOut[offset] * Gain, invGamma);
	
	}
	// XXX
	if (imager)
		; // XXXX
	if (ColorQuantOne != 0 && RGBOut)
		scaleAndDither(ColorQuantOne, ColorQuantMin,
			ColorQuantMax, ColorQuantDither, RGBOut,
			3 * XWidth * YWidth);
	if (ColorQuantOne != 0 && AlphaOut)
		scaleAndDither(ColorQuantOne, ColorQuantMin,
			ColorQuantMax, ColorQuantDither, AlphaOut,
			XWidth * YWidth);
	cerr << endl << "lrt: Writing " << DisplayName << "!" << endl;
	if (ColorQuantOne != 0.)
		TIFFWrite8Bit(DisplayName, RGBOut, AlphaOut, XWidth, YWidth);
	else
		TIFFWriteFloat(DisplayName, RGBOut, AlphaOut, XWidth, YWidth);
	if (ImageViewer != NULL && !fork())
		if (execlp(ImageViewer, ImageViewer, DisplayName,
				NULL) == -1) {
			Error("Unable to run image viewing program %s: %s",
				ImageViewer, strerror(errno));
			exit(1);
	}
	delete[] RGBOut;
	delete[] AlphaOut;
}
void Image::scaleAndDither(Float one, Float min, Float max,
		Float ditheramp, Float *pixels, int nSamples) {
	for (int i = 0; i < nSamples; ++i) {
		pixels[i] = Round(pixels[i] * one +
			RandomFloat(-ditheramp, ditheramp));
		pixels[i] = Clamp(pixels[i], min, max);
	}
}
void Image::AddSample(const Point &Praster, const Spectrum &radiance,
			Float alpha, Float wt) {
	Float HalfXWidth = sampler->FilterXWidth / 2.;
	Float HalfYWidth = sampler->FilterYWidth / 2.;
	int x0 = max((int)ceil (Praster.x - .5 - HalfXWidth), XBase);
	int x1 = min((int)floor(Praster.x - .5 + HalfXWidth), XBase + XWidth - 1);
	int y0 = max((int)ceil (Praster.y - .5 - HalfYWidth), YBase);
	int y1 = min((int)floor(Praster.y - .5 + HalfYWidth), YBase + YWidth - 1);
	for (int y = y0; y <= y1; ++y)
		for (int x = x0; x <= x1; ++x) {
			Float fx = x - Praster.x;
			Float fy = y - Praster.y;
			Float localWt = 
			  wt*sampler->filter->Apply(fx, fy, XWidth, YWidth);
			int offset = (y - YBase) * XWidth + (x - XBase);
			if (Pixels) Pixels[offset] += radiance * localWt;
			if (Alphas) Alphas[offset] += alpha * localWt;
			WeightSums[offset] += localWt;
		}
}
Imager::~Imager() { }
