#include "ri.h"
#include "mbdofcamera.h"
#include "film.h"
#include "scene.h"
#include "transport.h"
#include "primitives.h"
#include "geometry.h"

MbDOFPerspectiveCamera::MbDOFPerspectiveCamera(const Transform *world2cam,
		int nTransforms, Float hither, Float yon,
		Float shutterOpen, float shutterClose,
		Float fstop, Float focalLen, Float focalDist,
		RtToken mode, Float fov, Film *film)
	: Camera(world2cam[0],
		Perspective(fov, 1. / 1. /*scene->image->PixelAspectRatio*/,
			hither, yon) * Scale(-1, -1, 1),
		hither, yon, film) {
  
        WorldToCamera0 = world2cam[0];
	if(nTransforms > 1) {
	  WorldToCamera1 = world2cam[1];
	} else {
	  WorldToCamera1 = world2cam[0];
	}

	CameraToWorld0 = Transform(WorldToCamera0.GetInverse());
	CameraToWorld1 = Transform(WorldToCamera1.GetInverse());

	ShutterStart = shutterOpen;
	ShutterEnd = shutterClose;
	FStop = fstop;
	FocalLength = focalLen;
	FocalDistance = focalDist;
	Mode = mode;
	cerr << "Camera Mode=" << Mode << endl;
	//	if(Mode!=LRT_MANUAL)
	//  Warning("MbDOFPerspectiveCamera - only manual mode supported (so far).");

	// Adjust for exposure ... For part 3, you'll need to compute
	// your FStop and ShutterEnd before setting this.
	//	film->image->Gain *= ShutterEnd/(FStop*FStop);
}

// compute Shirley's Mapping, allowing negative values of theta
// Note:  first comparisons (the ones against 0) are superfluous
void Shirley(Float u1, Float u2, Float& r, Float& theta) {
  const Float rat = 0.785398163;
  // change u1 and u2 to be uniform over [-1, 1]
  u1 *= 2;
  u1 -= 1;
  u2 *= 2;
  u2 -= 1;

  if ((u1 >= 0) && (u2 <= u1) && (u2  >= -u1)) {  // rightmost "quadrant"
    r = u1;
    theta = rat*u2/u1;
  } else if ((u2 >= 0) && (u1 <= u2) && (u1 >= -u2)) {  // topmost
    r = u2;
    theta = rat*(2 - (u1/u2));
  } else if ((u1 <= 0) && (u2 <= -u1) && (u2 >= u1)) {  // leftmost
    r = - u1;
    theta = rat*(4 + (u2/u1));
  } else {  // bottommost
    assert((u2 <= 0) && (u1 >= u2) && (u1 <= -u2));
    r = -u2;
    theta = rat*(6 - (u1/u2));
  }
}

bool MbDOFPerspectiveCamera::GenerateRay(Float sample[5], Ray &ray) const {
  if (sample[0] < film->GetImage()->SampleCropLeft ||
      sample[0] > film->GetImage()->SampleCropRight ||
      sample[1] < film->GetImage()->SampleCropBottom ||
      sample[1] > film->GetImage()->SampleCropTop)
    return false;

  Float t = sample[4]*ShutterEnd;

  if (FStop == RI_INFINITY) {  // pinhole camera case
	Point Pcamera = RasterToCamera(Point(sample[0], sample[1], 0));
	Point PcameraWorld = Lerp(t,
				  CameraToWorld0(Pcamera),
				  CameraToWorld1(Pcamera));
	Point PpinholeWorld = Lerp(t,
				   CameraToWorld0(Point(0, 0, 0)),
				   CameraToWorld1(Point(0, 0, 0)));

	ray = Ray(PpinholeWorld, (PcameraWorld - PpinholeWorld).Hat());
	return true;

  } else {
	Point Phither = RasterToCamera(Point(sample[0], sample[1], 0));
	Point Pfocal(Phither.x*FocalDistance/Phither.z,
		     Phither.y*FocalDistance/Phither.z,
		     FocalDistance);
	Point PfocalWorld = Lerp(t,
			  CameraToWorld0(Pfocal),
			  CameraToWorld1(Pfocal));

	Float apRad = FocalLength/(2*FStop);  // halve for radius
	Float r = 0, theta = 0; // dummy initializations
	Shirley(sample[2], sample[3], r, theta);
	Point Paperture(apRad*r*cos(theta), apRad*r*sin(theta), 0);
	Point PapertureWorld = Lerp(t,
			  CameraToWorld0(Paperture),
			  CameraToWorld1(Paperture));

	ray = Ray(PapertureWorld, (PfocalWorld - PapertureWorld).Hat());
	return true;
  }
}

// Note always uses time 0!
void MbDOFPerspectiveCamera::LensToPoint(Float u, Float v,
					 Point p, Ray& ray) {
  Float r, theta;
  Shirley(u, v, r, theta);
  Point pLens(ApertureRadius*r*cos(theta), ApertureRadius*r*sin(theta), 0);
  Point origin = CameraToWorld0(pLens);
  ray = Ray(origin, (p - origin).Hat());
}

void MbDOFPerspectiveCamera::FromLensParallelTo(Float u, Float v,
						Vector dir, Ray &result) {
  Float r, theta;
  Shirley(u, v, r, theta);
  Point pLens(ApertureRadius*r*cos(theta), ApertureRadius*r*sin(theta), 0);
  Point origin = CameraToWorld0(pLens);
  result = Ray(origin, dir.Hat());
}

void FindPrimaryRange(Point *hits, int nx, int ny, Float &z0, Float &z1);

void MbDOFPerspectiveCamera::Calibrate(Scene *scene,
				       Integrator *integrator) {

  Point *orig;
  Point *shifted;
  Float bbar;
  Float *b;
  Float z0, z1;
  int nx = 16, ny = 16;
  int bnx = 16, bny = 16;
  int n = nx*ny, m = bnx*bny;

  orig = HitArray(scene, 0, scene->image->GetXResolution(),
		  0, scene->image->GetYResolution(), nx, ny, shifted);
  FindPrimaryRange(orig, nx, ny, z0, z1);
  bbar = Brightness(scene, integrator, 0, scene->image->GetXResolution(),
		    0, scene->image->GetYResolution(), bnx, bny, b);

  if (Mode == LRT_SHUTTER_PRIORITY) {
    ShutterPriorityCalibrate(orig, n, z0, z1, b, m, bbar);
  } else if (Mode == LRT_APERTURE_PRIORITY) {
    AperturePriorityCalibrate(orig, shifted, n, z0, z1, b, m);
  } else if (Mode == LRT_PROGRAMMED) {
    ProgrammedCalibrate(orig, shifted, n, z0, z1, b, m, bbar);
  }

  film->image->Gain *= (ShutterEnd*FocalLength*FocalLength)/(FStop*FStop);

  ApertureRadius = FocalLength/(2*FStop);
  Point p0P = RasterToCamera(Point(0, 0, 0));
  Point p1P = RasterToCamera(Point(scene->image->GetXResolution(),
				   scene->image->GetYResolution(), 0));
  x0 = p0P.x*FocalDistance/p0P.z;
  y0 = p0P.y*FocalDistance/p0P.z;
  x1 = p1P.x*FocalDistance/p1P.z;
  y1 = p1P.y*FocalDistance/p1P.z;
  xRasterRatio = scene->image->GetXResolution()/(x0 - x1);
  yRasterRatio = scene->image->GetYResolution()/(y0 - y1);
  RasterAreaOverFocalPlaneArea = xRasterRatio*yRasterRatio;
  if (RasterAreaOverFocalPlaneArea < 0)
    RasterAreaOverFocalPlaneArea = -RasterAreaOverFocalPlaneArea;

  cerr << endl << "ShutterEnd = " << ShutterEnd << endl
       << "FStop = " << FStop << endl
       << "FocalDistance = " << FocalDistance << endl;

}

// Computes a flattened 2D array of points in camera space hit at time 0 by
// rays through the lens center from an array of points.  The points are cell 
// centers of an nx by ny grid on the rectangle from (left, top, 0) to 
// (right, bottom, 0) in raster space.  Misses lead to entries of
// (0, 0, RI_INFINITY).
Point *MbDOFPerspectiveCamera::HitArray(Scene *scene, Float left, Float right,
					Float top, Float bottom,
					int nx, int ny, Point * &shifted) {
  Transform r2w0 = CameraToWorld0*RasterToCamera;
  Point *a = new Point[nx*ny];
  shifted = new Point[nx*ny];
  Ray ray;
  Point C = CameraToWorld0(Point(0,0,0));
  Surf s;

  for (int xi = 0; xi < nx; xi++) {
    for (int yi = 0; yi < ny; yi++) {
      Point W = r2w0(Point(Lerp((2*xi + 1.)/(2*nx), left, right),
			   Lerp((2*yi + 1.)/(2*ny), top, bottom),
			   0));
      ray = Ray(C, (W - C).Hat());
      if (scene->Intersect(ray, &s)) {
	a[xi + nx*yi] = WorldToCamera0(s.dgGeom.P);
	shifted[xi + nx*yi] = WorldToCamera1(s.dgGeom.P);
      } else {
	shifted[xi + nx*yi] = a[xi + nx*yi] = Point(0, 0, RI_INFINITY);
      }
    }
  }
  return a;
}

Float MbDOFPerspectiveCamera::Brightness(Scene *scene, Integrator *integrator,
					  Float left, Float right,
					  Float top, Float bottom,
					  int nx, int ny,
					  Float * &b) {
  Transform r2w0 = CameraToWorld0*RasterToCamera;
  b = new Float[nx*ny];
  Ray ray;
  Float alpha;
  Point C = CameraToWorld0(Point(0,0,0));
  Spectrum L;
  Float rgb[3];
  Float brightness;
  Float bAccum = 0;
  int nhits = 0;

  for (int xi = 0; xi < nx; xi++) {
    for (int yi = 0; yi < ny; yi++) {
      Point W = r2w0(Point(Lerp((2*xi + 1.)/(2*nx), left, right),
			   Lerp((2*yi + 1.)/(2*ny), top, bottom),
			   0));
      ray = Ray(C, (W - C).Hat());
      L = integrator->L(scene, ray, &alpha);
      if (alpha > 0) {
	L.ConvertToRGB(rgb);
	brightness = (rgb[0] >= rgb[1] ? rgb[0] : rgb[1]);
	if (rgb[2] > brightness) brightness = rgb[2];
	nhits++;
	b[xi + nx*yi] = brightness;
	bAccum += brightness;
      } else {
	b[xi + nx*yi] = -1;
      }
    }
  }
  assert(nhits);
  return (bAccum/nhits);

}

void FindPrimaryRange(Point *hits, int nx, int ny, Float &z0, Float &z1) {

  // find closest z in central quarter
  Float min = hits[nx/4 + nx*(ny/4)].z;
  Float z;
  for (int i = nx/4; i < nx - nx/4; i++) {
    for (int j = ny/4; j < ny - ny/4; j++) {
      z = hits[i + nx*j].z;
	if (z < min) min = z;
    }
  }

  // initialize near and far primary planes to the z found above
  z0 = z1 = min;

  // iteratively extend the range to include any z's which don't involve
  // more than a 10% change
  bool changed = true;
  while (changed) {
    changed = false;
    for (int i = 0; i < nx*ny; i++) {
      z = hits[i].z;
      if (z < z0 && z >= z0/1.1) {
	changed = true;
	z0 = z;
      } else if (z > z1 && z <= 1.1*z1) {
	changed = true;
	z1 = z;
      }
    }
  }
}

Float Misexposure(Float *b, int n, Float duration, Float aperture) {
  Float factor = duration*aperture*aperture;
  Float accum = 0;
  int nhits = 0;

  for (int i = 0; i < n; i++) {
    if (b[i] != -1) {
      nhits++;
      accum += Clamp(factor*b[i], 0., 1.);
    }
  }
  
  assert(nhits);
  return ((accum/nhits) - 0.5);
}

Float Misfocus(Point *a, int n,
	       Float focalDist, Float z0, Float z1,
	       Float focalLen, Float fstop) {
  int nin = 0;
  int nout = 0;
  Float delta = (focalDist - focalLen)*fstop/(100*focalLen);
  Float fz0 = (focalLen*focalDist)/(focalLen + delta);
  Float fz1 = (focalLen*focalDist)/(focalLen - delta);
  Float z;

  for (int i = 0; i < n; i++) {
    z = a[i].z;
    if (z >= z0 && z <= z1) {
      if (z >= fz0 && z <= fz1) {
	nin++;
      } else {
	nout++;
      }
    }
  }

  return (nout/(nin + 0.00005));
}

Float MotionBlur(Point *orig, Point *shift, int n, Float time, Transform c2r) {
  Float maxBlur = 0;
  Float blur;
  Point start, end, startRast, endRast;
  Vector move;

  for (int i = 0; i < n; i++) {
    if (orig[i].z != RI_INFINITY) {
      start = orig[i];
      end = Lerp(time, start, shift[i]);
      startRast = c2r(start);
      endRast = c2r(end);
      move = endRast - startRast;
      blur = sqrt(move.x*move.x + move.y*move.y);
      if (blur > maxBlur) maxBlur = blur;
    }
  }

  return maxBlur;
}

Float AutoFocus(Point *a, int n,
		Float z0, Float z1,
		Float minzEst, Float maxzEst,
		Float focalLen, Float fstop,
		Float onePlusE, int numPts) {
  if ((maxzEst/minzEst) < onePlusE) return ((minzEst + maxzEst)/2);

  Float delta = (maxzEst - minzEst)/(numPts + 1);
  Float z = minzEst + delta;
  Float bestz = z;
  Float leastMisfocus = Misfocus(a, n, z, z0, z1, focalLen, fstop);
  Float misfocus;

  for (int i = 1; i < numPts; i++) {
    z += delta;
    misfocus = Misfocus(a, n, z, z0, z1, focalLen, fstop);
    if (misfocus < leastMisfocus) {
      bestz = z;
      leastMisfocus = misfocus;
    }
  }
  return AutoFocus(a, n, z0, z1, bestz - delta, bestz + delta,
		   focalLen, fstop, onePlusE, 4);
}

Float AutoShutter(Point *orig, Point *shift, int n,
		  Float *b, int m, Float aperture,
		  Float minEst, Float maxEst,
		  Float epsilon, Float numPts,
		  Transform c2r) {
  if ((maxEst - minEst) < epsilon) return ((minEst + maxEst)/2);

  Float delta = (maxEst - minEst)/(numPts + 1);
  Float bestDur = -1; // sentinel for undefined
  Float dur = minEst + delta;
  Float leastBadness = -1;
  Float badness, misexposure, adjMisexp, mblur;

  for (int i = 0; i < numPts; i++) {
    misexposure = Misexposure(b, m, dur, aperture);
    mblur = MotionBlur(orig, shift, n, dur, c2r);
      adjMisexp = misexposure < 0 ?
	          1000*misexposure*misexposure :
	          2*misexposure*misexposure*misexposure;
      badness = adjMisexp*adjMisexp + mblur*mblur;
    if ((bestDur < 0) || (badness < leastBadness)) {
      bestDur = dur;
      leastBadness = badness;
    }
    dur += delta;
  }

  assert(bestDur > 0);
  return AutoShutter(orig, shift, n, b, m, aperture,
		     bestDur - delta, bestDur + delta,
		     epsilon, 4, c2r);
}

void MbDOFPerspectiveCamera::AperturePriorityCalibrate(Point *orig,
						       Point *shift, int n,
						       Float z0, Float z1,
						       Float *b, int m) {
  ShutterEnd = AutoShutter(orig, shift, n, b, m, FocalLength/FStop,
			   0, 1, 0.0001, 5,
			   Transform(RasterToCamera.GetInverse()));
  FocalDistance = AutoFocus(orig, n, z0, z1, z0, z1,
			    FocalLength, FStop, 1.0001, 20);
}

Float AutoFStop(Point *a, int n, Float z0, Float z1, Float *b, int m,
		Float duration, Float focalLen,
		Float minEst, Float maxEst, Float onePlusE, Float numPts) {
  if (maxEst/minEst < onePlusE) return ((maxEst + minEst)/2);

  Float delta = (maxEst - minEst)/(numPts + 1);
  Float bestFStop = -1;
  Float fstop = minEst + delta;
  Float leastBadness = -1, badness, misexposure, adjMisexp, misfocus;
  Float focalDist;
  Float focalEpsilon = Clamp((Float) (maxEst/(onePlusE*minEst))*1.00001,
			     (Float) 1.0001, (Float) 1.05);

  for (int i = 0; i < numPts; i++) {
    misexposure = Misexposure(b, m, duration, focalLen/fstop);
    focalDist = AutoFocus(a, n, z0, z1, z0, z1,
			  focalLen, fstop, focalEpsilon, 20);
    misfocus = Misfocus(a, n, focalDist, z0, z1, focalLen, fstop);
    adjMisexp = misexposure < 0 ?
      1000*misexposure*misexposure :
      2*misexposure*misexposure*misexposure;
    badness = adjMisexp*adjMisexp + misfocus;
    if ((bestFStop < 0) || (badness < leastBadness)) {
      bestFStop = fstop;
      leastBadness = badness;
    }
    fstop += delta;
  }

  assert(bestFStop > 0);
  return AutoFStop(a, n, z0, z1, b, m, duration, focalLen,
		   bestFStop - delta, bestFStop + delta, onePlusE, 4);
}

void MbDOFPerspectiveCamera::ShutterPriorityCalibrate(Point *orig, int n,
						      Float z0, Float z1,
						      Float *b, int m,
						      Float bbar) {
  Float maxFStop = 3.*FocalLength*sqrt(ShutterEnd*bbar);
  FStop = AutoFStop(orig, n, z0, z1, b, m, ShutterEnd, FocalLength,
		    0, maxFStop, 1.0001, (int) (maxFStop*6));
  FocalDistance = AutoFocus(orig, n, z0, z1, z0, z1, FocalLength, FStop,
			    1.0001, 20);
}

void AutoEverything(Point *orig, Point *shift, int n, Float z0, Float z1,
		    Float *b, int m, Float focalLen,
		    Float minDur, Float maxDur,
		    Float epsilonDur, Float numDurVals,
		    Float minFStop, Float maxFStop,
		    Float onePlusEpsilonFStop, Float numFStopVals,
		    Transform c2r,
		    Float &duration, Float &fstop, Float &focalDist) {
  if ((maxDur - minDur < epsilonDur) &&
      (maxFStop/minFStop < onePlusEpsilonFStop)) {
    duration = (minDur + maxDur)/2;
    fstop = (minFStop + maxFStop)/2;
    focalDist = AutoFocus(orig, n, z0, z1, z0, z1,
			  focalLen, fstop, 1.0001, 20);
    return;
  }

  Float deltaDur = (maxDur - minDur)/(numDurVals + 1);
  Float deltaFStop = (maxFStop - minFStop)/(numFStopVals + 1);
  Float DurAtBestPt = -1.;
  Float FStopAtBestPt = -1.;
  Float DurAtCurrPt, FStopAtCurrPt, focalDistAtCurrPt;
  Float badness, minBadness = -1., misexposure, adjMisexp, mblur, misfocus;
  Float focalEpsilon =
    Clamp((Float) (maxFStop/(onePlusEpsilonFStop*minFStop))*1.00001,
	  (Float) 1.0001, (Float) 1.05);

  DurAtCurrPt = minDur + deltaDur;
  for (int i = 0; i < numDurVals; i++) {
    FStopAtCurrPt = minFStop + deltaFStop;
    for (int j = 0; j < numFStopVals; j++) {
      misexposure = Misexposure(b, m, DurAtCurrPt, focalLen/FStopAtCurrPt);
      mblur = MotionBlur(orig, shift, n, DurAtCurrPt, c2r);
      focalDistAtCurrPt = AutoFocus(orig, n, z0, z1, z0, z1,
				    focalLen, FStopAtCurrPt, focalEpsilon, 20);
      misfocus = Misfocus(orig, n, focalDistAtCurrPt, z0, z1,
			  focalLen, FStopAtCurrPt);
      adjMisexp = misexposure < 0 ?
	          1000*misexposure*misexposure :
	          2*misexposure*misexposure*misexposure;
      badness = adjMisexp*adjMisexp
	        + mblur*mblur + misfocus;
      if ((DurAtBestPt < 0) || (badness < minBadness)) {
	DurAtBestPt = DurAtCurrPt;
	FStopAtBestPt = FStopAtCurrPt;
	minBadness = badness;
      }
      FStopAtCurrPt += deltaFStop;
    }
    DurAtCurrPt += deltaDur;
  }
  
  assert(DurAtBestPt > 0);
  AutoEverything(orig, shift, n, z0, z1, b, m, focalLen,
		 DurAtBestPt - deltaDur, DurAtBestPt + deltaDur,
		 epsilonDur, 4,
		 FStopAtBestPt - deltaFStop , FStopAtBestPt + deltaFStop,
		 onePlusEpsilonFStop, 4, c2r,
		 duration, fstop, focalDist);
}

void MbDOFPerspectiveCamera::ProgrammedCalibrate(Point *orig, Point *shift,
						 int n, Float z0, Float z1,
						 Float *b, int m,
						 Float bbar) {
  Float capFStop = 3.*FocalLength*sqrt(bbar);  // base on max duration of 1

  AutoEverything(orig, shift, n, z0, z1, b, m, FocalLength, 0, 1, .0001, 40,
		 0, capFStop, 1.0001, (int) (capFStop*41),
		 Transform(RasterToCamera.GetInverse()),
		 ShutterEnd, FStop, FocalDistance);
}

Float MbDOFPerspectiveCamera::ApertureScaleFactor(Float z) {
  Float factor;
    factor = (FocalDistance - z)/z;
  if (z < FocalDistance) {
    return factor;
  } else {
    return -factor;
  }
}

#define PI (3.141592654)

Float MbDOFPerspectiveCamera::AreaOfConfusion(Float z) {
  Float radiusOnFocalPlane = ApertureRadius*ApertureScaleFactor(z);
  Float areaOnFocalPlane = PI*radiusOnFocalPlane*radiusOnFocalPlane;

  return RasterAreaOverFocalPlaneArea*areaOnFocalPlane;
}

Float MbDOFPerspectiveCamera::ConfusionAtInfinity() {
  return RasterAreaOverFocalPlaneArea*PI*ApertureRadius*ApertureRadius;
}

void MbDOFPerspectiveCamera::TraceToFilm(Ray& r, Float& xRaster,
					 Float& yRaster) {
   Ray ray = WorldToCamera0(r);
   if (ray.O.z != 0) {cerr << "ray.O.z = " << ray.O.z << endl;}
   Float d = FocalDistance/ray.D.z;
   Point pFocal = ray(d);
   xRaster = (x0 - pFocal.x)*xRasterRatio;
   yRaster = (y0 - pFocal.y)*yRasterRatio;
 }
