#include "light.h"
#include "primitives.h"
#include "color.h"
#include "scene.h"
Light::~Light() {
}
Spectrum Light::dE(const Scene *scene,
	const DifferentialGeometry &dg,	const Vector &w) const {
	return Spectrum(0.);
}
Spectrum Light::dE(const Scene *scene,
	const DifferentialGeometry &dg,	Vector *w) const {
	*w = Vector(0,0,0);
	return Spectrum(0.);
}
bool Light::visible(const Scene *scene, const Point &x1,
	const Point &x2) const {
	if (!CastsShadows) return 1.;
	Ray r(x1, x2-x1, 1e-3, .999);
	return scene->IntersectP(r) ? false : true;
}
float Light::unoccluded(const Scene *scene, const Point &x,
		const Vector &w) const {
	if (!CastsShadows) return 1.;
	Ray r(x, w, 1e-3);
	return scene->IntersectP(r) ? 0. : 1.;
}
Spectrum Light::sample_dE(const Scene *scene,
		const DifferentialGeometry &dg, Vector *wo,
		Float *weight) const {
	*weight = 1;
	return dE(scene, dg, wo);
}
PointLight::PointLight(bool shadows, const Transform &world2light,
	const Spectrum &power, const Point &Plight)
	: Light(shadows, world2light, power) {
	lightPos = LightToWorld(Plight);
}
Spectrum PointLight::dE(const Scene *scene,
		const DifferentialGeometry &dg,	Vector *w) const {
	*w = lightPos - dg.P;
	w->Normalize();
	if (visible(scene, dg.P, lightPos))
		return I(*w) * fabs(Dot(*w, dg.N)) /
			DistanceSquared(lightPos, dg.P);
	return Spectrum(0.);
}
InfinitePointLight::InfinitePointLight(bool shadows,
	const Transform &world2light, const Spectrum &power,
	const Vector &dir)
	: Light(shadows, world2light, power) {
	lightDir = LightToWorld(dir);
}
Spectrum InfinitePointLight::dE(const Scene *scene,
		const DifferentialGeometry &dg,	Vector *w) const {
	*w = - lightDir;
	w->Normalize();
	return unoccluded(scene, dg.P, *w) *
		Power * fabs(Dot(*w, dg.N));
}
AreaLight::AreaLight(bool shadows, const Transform &world2light,
	const Spectrum &power, Shape *s)
	: Light(shadows, world2light, power) {
	shape = s;
	Area = shape->area();
}
Spectrum AreaLight::dE(const Scene *scene,
		const DifferentialGeometry &dg,	const Vector &w) const {
	DifferentialGeometry hit;
	Ray ray(dg.P, w);
	if (shape->Intersect(ray, &hit) &&
		visible( scene, dg.P, hit.P ))
	    return L(hit.P, -w);
	return 0.;
}
Spectrum AreaLight::sample_dE(const Scene *scene,
		const DifferentialGeometry &dg, Vector *wo,
		Float *wt) const {
	shape->sample(dg, wo);
	*wt = shape->weight(dg, *wo);
	return dE(scene, dg, *wo);
}
Spectrum InfiniteAreaLight::dE(const Scene *scene,
	 	const DifferentialGeometry &dg,	const Vector &w) const {
	return unoccluded(scene, dg.P, w) * Power *
		max(0.f, Dot(w, dg.N));
}
Spectrum InfiniteAreaLight::sample_dE(const Scene *scene,
		const DifferentialGeometry &dg, Vector *wo,
		Float *weight) {
	Float u1 = RandomFloat(), u2 = RandomFloat();
	Float r = sqrtf(u2);
	Float theta = 2. * M_PI * u1;
	Float x = r * cos(theta);
	Float y = r * sin(theta);
	Float z = sqrtf(1. - x*x - y*y);
	if (RandomFloat() < .5) z *= -1;
	*wo = Vector(x, y, z);
	*wo = Vector(dg.S.x * wo->x + dg.T.x * wo->y + dg.N.x * wo->z,
		     dg.S.y * wo->x + dg.T.y * wo->y + dg.N.y * wo->z,
		     dg.S.z * wo->x + dg.T.z * wo->y + dg.N.z * wo->z);
	*weight = 2.*M_PI / fabs(wo->z);
	return dE(scene, dg, *wo);
}
AmbientLight::AmbientLight(const Spectrum &p)
	: Light(false, Transform(), p) {
}
Spectrum AmbientLight::dE(const Scene *,
		const DifferentialGeometry &, const Vector &) const {
	return Power;
}

Spectrum AmbientLight::dE(const Scene *scene,
		const DifferentialGeometry &dg,  Vector *w) const {
	*w = Vector(0, 0, 0);
	return Power;
}
