#include "lrt.h"
#include "color.h"
#include "reflection.h"
#include "shapes.h"
#include <stdarg.h>

BSDF::BSDF(const DifferentialGeometry &dg, BxDF *r, ...)
	: N(dg.N), S(dg.S), T(dg.T)
{
	va_list args;
	va_start(args, r);
	while (r != NULL) {
		Float wt = va_arg(args, double);
		if (wt > 0) {
			bxdfs.push_back(r);
			weights.push_back(wt);
		}
		r = va_arg(args, BxDF *);
	}
}
BSDF::~BSDF() {
	for (u_int i = 0; i < bxdfs.size(); ++i)
		delete bxdfs[i];
}
int BSDF::NumSpecular() const {
	int n = 0;
	for (u_int i = 0; i < bxdfs.size(); ++i)
		if (bxdfs[i]->IsSpecular()) ++n;
	return n;
}
Spectrum BSDF::f(const Spectrum &de, const Vector &wiW, const Vector &woW) const {
	Vector wi = WorldToLocal(wiW), wo = WorldToLocal(woW);
	Spectrum f = 0.;
	for (u_int i = 0; i < bxdfs.size(); ++i)
		f += weights[i] * bxdfs[i]->f(de, wi, wo);
	return f;
}
Spectrum BSDF::f_delta(int component, const Vector &w,
		Vector *wi) const {
	BxDF *spec = NULL;
	Float weight = 0.;
	for (u_int i = 0; i < bxdfs.size(); ++i) {
		if (bxdfs[i]->IsSpecular()) {
			if (component-- == 0) {
				spec = bxdfs[i];
				weight = weights[i];
				break;
			}
		}
	}
	if (spec) {
		Vector wo = WorldToLocal(w);
		Spectrum f = weight * spec->f_delta(wo, wi);
		*wi = LocalToWorld(*wi);
		return f;
	}
	return 0.;
}
Spectrum LambertianReflection::f(const Spectrum &de, const Vector &wi,
		const Vector &wo) const {
  if (1 ||  sameHemisphere(wi, wo))
		return R / M_PI;
	else
		return 0.;
}
Spectrum LambertianTransmission::f(const Spectrum &de, const Vector &wi,
		const Vector &wo) const {
	if (!sameHemisphere(wi, wo))
		return T / M_PI;
	else
		return 0.;
}
Microfacet::Microfacet(const Spectrum &reflectance,
	MicrofacetDistribution *d) {
	R = reflectance;
	distribution = d;
}
Microfacet::~Microfacet() {
	delete distribution;
}
Spectrum Microfacet::f(const Spectrum &de, const Vector &wi, const Vector &wo) const {
	if (sameHemisphere(wi, wo)) {
		Float cosThetaI = fabs(wi.z);
		Float cosThetaO = fabs(wo.z);
		return R * distribution->D(wi, wo) * G(wi, wo) *
			F(wi, wo) / (4. * cosThetaI * cosThetaO);
	}
	else return 0.;
}
SpecularReflection::SpecularReflection(const Spectrum &r) {
	R = r;
}
Spectrum SpecularReflection::f_delta(const Vector &wo,
		Vector *wi) const {
	*wi = Vector(-wo.x, -wo.y, wo.z);
	return R;
}
SpecularTransmission::SpecularTransmission(const Spectrum &r,
		Float indexi, Float indext) {
	R = r;
	eta = indexi / indext;
}
Spectrum SpecularTransmission::f_delta(const Vector &wo,
		Vector *wi) const {
	Float cosi = wo.z;
	Vector N(0,0,1);
	Float e = eta;
	if (cosi < 0.) {
		N = -N;
		cosi = -cosi;
		e = 1./eta;
	}

	Float cost = 1.0 - e * e * (1.0 - cosi * cosi);
	if (cost < 0.)
		return 0.; // *wi = Vector(-wo.x, -wo.y, wo.z);
	else {
		Float k = e * cosi - sqrtf(cost);
		*wi = k * N + e * -wo;
	}
	return R;
}


void BxDF::sample_f(const Vector &wi, Vector *wo) const {
	Float u1 = RandomFloat(), u2 = RandomFloat();
	Float r = sqrtf(u2);
	Float theta = 2. * M_PI * u1;
	Float x = r * cos(theta);
	Float y = r * sin(theta);
	Float z = sqrtf(1. - x*x - y*y);
	*wo = Vector(x, y, z);
	if (wi.z * wo->z < 0.) *wo = -*wo;
}
Float BxDF::weight(const Vector &wi, const Vector &wo) const {
	return M_PI / fabs(wo.Hat().z);
}
void LambertianTransmission::sample_f(const Vector &wi,
		Vector *wo) const {
	BxDF::sample_f(wi, wo);
	wo->z *= -1;
}
Spectrum BxDF::rho(const Vector &w) const {
	Spectrum r = 0.;
	int nSamples = 10;
	for (int i = 0; i < nSamples; ++i) {
		Vector wi;
		sample_f(w, &wi);
		r += f(r, w, wi) * weight(w, wi);
	}
	return r / nSamples;
}
Spectrum BxDF::rho() const {
	Spectrum r = 0.;
	int nSamples = 10;
	for (int i = 0; i < nSamples; ++i) {
		Vector wo, wi;
		while (1) {
			wo = Vector(RandomFloat(-1, 1), RandomFloat(-1, 1),
				RandomFloat(-1, 1));
			if (wo.LengthSquared() < 1.) break;
		}
		wo = wo.Hat();
		if (wo.z < 0.) wo.z *= -1;
		Float weight_o = 2.*M_PI;
		sample_f(wo, &wi);
		Float weight_i = weight(wo, wi);
		r += f(r, wo, wi) * weight_o * weight_i / M_PI;
	}
	return r / nSamples;
}
void Microfacet::sample_f(const Vector &wo, Vector *wi) const {
	distribution->sample_f(wo, wi);
}
Float Microfacet::weight(const Vector &wo, const Vector &wi) const {
	return distribution->weight(wo, wi);
}
void Blinn::sample_f(const Vector &wo, Vector *wi) const {
	float costheta = pow(RandomFloat(), 1. / (exponent+1));
	float sintheta = sqrtf(1. - costheta*costheta);
	float phi = RandomFloat() * 2. * M_PI;
	Vector H(sintheta * sin(phi), sintheta * cos(phi), costheta);
	*wi = -wo + 2. * Dot(wo, H) * H;
}
Float Blinn::weight(const Vector &wo, const Vector &wi) const {
	if (wi.z * wo.z < 0.) return 0;
	Vector H = (wi + wo).Hat();
	float costheta = H.z;
	return (4. * Dot(wi, H)) /
		((exponent + 1) * pow(costheta, exponent));

}
Spectrum BSDF::sample_f(const Spectrum &de, const Vector &wiW, Vector *wo) const {
	Vector wi = WorldToLocal(wiW);
	int which = int(RandomFloat() * bxdfs.size());
	bxdfs[which]->sample_f(wi, wo);
	*wo = LocalToWorld(*wo);
	return f(de, wiW, *wo);
}
Float BSDF::weight(const Vector &wiW, const Vector &woW) const {
	Vector wi = WorldToLocal(wiW), wo = WorldToLocal(woW);
	Float wt = 0;
	for (u_int i = 0; i < bxdfs.size(); ++i)
		wt += bxdfs[i]->weight(wi, wo);
	return wt / bxdfs.size();
}
Spectrum BxDF::rho() const;
Spectrum BxDF::rho(const Vector &w) const;

// xyz color curves (380 nm to 780 nm)

const Float ThinFilmReflection::xyz[81][3] = {
  { 0.0002, 0.0000, 0.0007 },
  { 0.0007, 0.0001, 0.0029 },
  { 0.0024, 0.0003, 0.0105 },
  { 0.0072, 0.0008, 0.0323 }, 
  { 0.0191, 0.0020, 0.0860 }, 
  { 0.0434, 0.0045, 0.1971 },
  { 0.0847, 0.0088, 0.3894 },
  { 0.1406, 0.0145, 0.6568 },
  { 0.2045, 0.0214, 0.9725 },
  { 0.2647, 0.0295, 1.2825 },
  { 0.3147, 0.0387, 1.5535 },
  { 0.3577, 0.0496, 1.7985 },
  { 0.3837, 0.0621, 1.9673 },
  { 0.3867, 0.0747, 2.0273 },
  { 0.3707, 0.0895, 1.9948 },
  { 0.3430, 0.1063, 1.9007 },
  { 0.3023, 0.1282, 1.7454 },
  { 0.2541, 0.1528, 1.5549 },
  { 0.1956, 0.1852, 1.3176 },
  { 0.1323, 0.2199, 1.0302 },
  { 0.0805, 0.2536, 0.7721 },
  { 0.0411, 0.2977, 0.5701 },
  { 0.0162, 0.3391, 0.4153 },
  { 0.0051, 0.3954, 0.3024 },
  { 0.0038, 0.4608, 0.2185 },
  { 0.0154, 0.5314, 0.1592 },
  { 0.0375, 0.6067, 0.1120 },
  { 0.0714, 0.6857, 0.0822 },
  { 0.1177, 0.7618, 0.0607 },
  { 0.1730, 0.8233, 0.0431 },
  { 0.2365, 0.8752, 0.0305 },
  { 0.3042, 0.9238, 0.0206 },
  { 0.3768, 0.9620, 0.0137 },
  { 0.4516, 0.9822, 0.0079 },
  { 0.5298, 0.9918, 0.0040 },
  { 0.6161, 0.9991, 0.0011 },
  { 0.7052, 0.9973, 0.0000 },
  { 0.7938, 0.9824, 0.0000 },
  { 0.8787, 0.9556, 0.0000 },
  { 0.9512, 0.9152, 0.0000 },
  { 1.0142, 0.8689, 0.0000 },
  { 1.0743, 0.8256, 0.0000 },
  { 1.1185, 0.7774, 0.0000 },
  { 1.1343, 0.7204, 0.0000 },
  { 1.1240, 0.6583, 0.0000 },
  { 1.0891, 0.5939, 0.0000 },
  { 1.0305, 0.5280, 0.0000 },
  { 0.9507, 0.4618, 0.0000 },
  { 0.8563, 0.3981, 0.0000 },
  { 0.7549, 0.3396, 0.0000 },
  { 0.6475, 0.2835, 0.0000 },
  { 0.5351, 0.2283, 0.0000 },
  { 0.4316, 0.1798, 0.0000 },
  { 0.3437, 0.1402, 0.0000 },
  { 0.2683, 0.1076, 0.0000 },
  { 0.2043, 0.0812, 0.0000 },
  { 0.1526, 0.0603, 0.0000 },
  { 0.1122, 0.0441, 0.0000 },
  { 0.0813, 0.0318, 0.0000 },
  { 0.0579, 0.0226, 0.0000 },
  { 0.0409, 0.0159, 0.0000 },
  { 0.0286, 0.0111, 0.0000 },
  { 0.0199, 0.0077, 0.0000 },
  { 0.0138, 0.0054, 0.0000 },
  { 0.0096, 0.0037, 0.0000 },
  { 0.0066, 0.0026, 0.0000 },
  { 0.0046, 0.0018, 0.0000 },
  { 0.0031, 0.0012, 0.0000 },
  { 0.0022, 0.0008, 0.0000 },
  { 0.0015, 0.0006, 0.0000 },
  { 0.0010, 0.0004, 0.0000 },
  { 0.0007, 0.0003, 0.0000 },
  { 0.0005, 0.0002, 0.0000 },
  { 0.0004, 0.0001, 0.0000 },
  { 0.0003, 0.0001, 0.0000 },
  { 0.0002, 0.0001, 0.0000 },
  { 0.0001, 0.0000, 0.0000 },
  { 0.0001, 0.0000, 0.0000 },
  { 0.0001, 0.0000, 0.0000 },
  { 0.0000, 0.0000, 0.0000 },
  { 0.0000, 0.0000, 0.0000 }
};

const Float ThinFilmReflection::xyz2rgb[3][3] = {
  { 3.240479, -1.537150, -0.498535 },
  { -0.969256, 1.875992, 0.041556 },
  { 0.055648, -0.204043, 1.057311 }
};

Spectrum ThinFilmReflection::f(const Spectrum &de, const Vector &wi, const
			       Vector &wo) const {
  Float r, g, b;
  Float ni = 1.44;
  Float t;
  if(tfcn != -1.) t = th - mod * (1. - tfcn);
  else t = th;

  // calculate intensities for the reflection and refraction waves
  // these will be used for calculating the total intensity later
  Float lightRgb[3], matRgb[3];
  de.ConvertToRGB(lightRgb);
  D.ConvertToRGB(matRgb);

  Float i1[81];
  Float i2[81];
  
  for (int i =380; i <= 780; i+=5) {
    if (i >= 580) {
      i1[(i - 380) / 5] = lightRgb[0] * 2.4;
      i2[(i - 380) / 5] = matRgb[0];
    } else if (i >= 500) {
      i1[(i - 380) / 5] = lightRgb[1] * 3;
      i2[(i - 380) / 5] = matRgb[1];
    } else {
      i1[(i - 380) / 5] = lightRgb[2] * 3;
      i2[(i - 380) / 5] = matRgb[2];
    }
  }

  // depending on the angle of incidence to the normal of the surface,
  // calculate path length d as a function of the
  // thickness of the film and angle and refractive index

  Float costheta = Dot(N, wi);
  Float denom = cos(asin(sqrt((1 - costheta * costheta) / ni)));

  Float d = 2 * t / denom;


  // for each wavelength in XYZ spectrum, calculate the total phase and
  // then sum, then calculate the total outgoing intensity 

  Float phase, intensity[81];
  for(int i = 380; i <= 780; i+=5) {
    Float a1 = i1[(i - 380) / 5];
    Float a2 = i2[(i - 380) / 5] * a1 * trans;
    a1 *= (1- trans);
    
    phase = cos(2 * M_PI * d / (float)i);
    intensity[(i - 380) / 5] = sqrt(a1*a1 + a2*a2 + 2*a1*a2*phase);
  }
  // finally, after finding final intensity, convert intensity to X, Y, Z
  // color, then to RGB color and return the spectrum

  Float x_bar = 0., y_bar = 0., z_bar = 0., k = 0.;
  
  for(int j = 0; j <= 80; j++) {
    x_bar += (intensity[j] * xyz[j][0]);
    y_bar += (intensity[j] * xyz[j][1]);
    z_bar += (intensity[j] * xyz[j][2]);
  }

  k = 1. / 208.;
  x_bar *= k;
  y_bar *= k;
  z_bar *= k;

  r = xyz2rgb[0][0] * x_bar + xyz2rgb[0][1] * y_bar + xyz2rgb[0][2] * z_bar;
  g = xyz2rgb[1][0] * x_bar + xyz2rgb[1][1] * y_bar + xyz2rgb[1][2] * z_bar;
  b = xyz2rgb[2][0] * x_bar + xyz2rgb[2][1] * y_bar + xyz2rgb[2][2] * z_bar;

  //printf("R: %f    G: %f     B: %f\n", r, g, b);

  Spectrum reflection = Spectrum(r, g, b);

  return reflection;
}
