#include "scene.h"
#include "camera.h"
#include "primitives.h"
#include "image.h"
#include "transport.h"
#include "accel.h"
#include "sampling.h"
#include "photonmap.h"

#define USE_ENV_LIGHT

extern float sun_intensity_factor;

extern float time_of_day;

extern int num_photons;

Scene::~Scene() {
	delete camera;
	delete sampler;
	delete image;
	delete integrator;
}
Spectrum Scene::L(const Ray &ray) const {
	return integrator->L(this, ray, NULL);
}
bool Scene::IntersectP(const Ray &ray) const {
	return prims->IntersectP(ray);
}
Scene::Scene(Camera *cam, Integrator *in, Image *img, Sampler *s,
		const vector<Primitive *> &pr,
		const vector<Light *> &lts) {
	lights = lts;
	prims = new GridAccelerator(pr);
	camera = cam;
	image = img;
	sampler = s;
	integrator = in;
#ifdef USE_ENV_LIGHT
//lights.push_back(new EnvironmentLight(true, Transform(), Spectrum(0.), time_of_day));
	envLight = new EnvironmentLight(true, Transform(), Spectrum(0.), 
					time_of_day);
	Point sunPos = Point(0.,0.,0.)+(1000.0*envLight->sunVector);
	sunLight = new PointLight(true, Transform(),
				  envLight->skyBase*sun_intensity_factor,
				  sunPos);
	lights.push_back(sunLight);
#endif // USE_ENV_LIGHT

}
void Scene::Render() {
  int max_rays = image->GetXResolution() * image->GetYResolution() * 
    sampler->XMax/sampler->XResolution * sampler->YMax/sampler->YResolution;
  
  PhotonMap *photonmap = new PhotonMap(num_photons, 0);
  photonmap->build_map(this);

  cerr << "Rendering: " << endl;
  Float sample[5];
  while (sampler->GetNextImageSample(sample)) {
    Ray ray;
    if (!camera->GenerateRay(sample, ray))
      continue;
    static int eyeRaysTraced = 0;
    if (eyeRaysTraced == 0)
      StatsRegisterCounter(STATS_BASIC, "Camera", "Eye Rays Traced",
			   &eyeRaysTraced);
    ++eyeRaysTraced;
    //if (eyeRaysTraced % 10000 == 0) cerr << '+';
    if(eyeRaysTraced % 10 == 0){
      fprintf(stdout, "%.2f percent\r", (((float)eyeRaysTraced 
				   	/ (float)max_rays)*100.));
    }
    Float alpha;
    integrator->setPhotonMap(photonmap);
    Spectrum L = integrator->L(this, ray, &alpha);
    Float screenz = camera->WorldToScreen(ray(ray.maxt)).z;
    if (screenz > 1.) {
      L = 0.;
      alpha = 0.;
    }
    Point Praster(sample[0], sample[1], screenz);
    image->AddSample(Praster, L, alpha);
  }
  image->Write();
  cerr << endl;
}

bool Scene::Intersect(const Ray &ray, Surf *surf) const {
	return prims->Intersect(ray, surf);
}
