#include "lrt.h"
#include "primitives.h"
#include "color.h"
#include "light.h"
#include "scene.h"
#include "reflection.h"
#include "sampling.h"
#include "materials.h"
#include "transport.h"
Integrator::~Integrator() {
}

#define DIRECT_SKY_LIGHT

float direct_sky_factor = 1.0;
float direct_sun_factor = 0.75;
float indirect_factor = 5.0;

extern int photons_per_estimate;
extern float max_photon_distance;

Spectrum PhotonVisIntegrator::L(const Scene *scene,
		const Ray &ray, Float *alpha) const {
  Surf surf;
  if(photonmap == NULL){
    cerr << "PhotonMap is NULL!!!" << endl;
    exit(1);
  }
  else if (scene->Intersect(ray, &surf)) {
    float irradiance[3];
    float pos[3];
    float normal[3];
    float max_dist = 100.0;
    int nphotons = 250;

    pos[0] = surf.dgGeom.P.x;
    pos[1] = surf.dgGeom.P.y;
    pos[2] = surf.dgGeom.P.z;
    normal[0] = surf.dgGeom.N.x;
    normal[1] = surf.dgGeom.N.y;
    normal[2] = surf.dgGeom.N.z;
    
    photonmap->irradiance_estimate(irradiance, pos, normal, max_dist, nphotons);
    Spectrum spectrum = Spectrum(irradiance[0],irradiance[1],irradiance[2]); 
    return spectrum;
  }
  else {
    if (alpha) *alpha = 0.;
    return Spectrum(0.);
  }
}

Spectrum PhotonMapIntegrator::L(const Scene *scene, const Ray &ray,
				Float *alpha) const {
  Surf surf;
  if(photonmap == NULL){
    cerr << "PhotonMap is NULL!!!" << endl;
    exit(1);
  } else if (scene->Intersect(ray, &surf)) {
    Spectrum L(0.);
    L += DirectL(scene, ray, surf, alpha);
    L += indirect_factor*IndirectL(scene, ray, surf, alpha);
    return L;
  } else {
    if (alpha) *alpha = 0.;
    //    return Spectrum(0.);
    Spectrum base = Spectrum(0.2);
    return (base + (0.05*scene->envLight->skyBase));
  }
}

//static int shadow_rays = 16;
extern int env_rays;

static BBox skylightBound = BBox(Point(-9.2,23.8,35.4), Point(0.4,26.2,-23.0));

Spectrum PhotonMapIntegrator::DirectL(const Scene *scene, const Ray &ray,
				      Surf &surf, Float *alpha) const {
  Spectrum L(0.);
  if (alpha) *alpha = 1.;
  L += surf.Le(-ray.D);
  BSDF *bsdf = surf.getBSDF();
  Vector wi;
//  float wt;
//#ifndef DIRECT_SKY_LIGHT
  for (u_int i = 0; i < scene->lights.size(); ++i) {
    Spectrum dE = scene->lights[i]->dE(scene, surf.dgShading, &wi);
    /*Spectrum curL = Spectrum(0.);
    for (int sRay = 0; sRay < shadow_rays; sRay++) {
      Spectrum dE = scene->lights[i]->sample_dE(scene, surf.dgShading,
						&wi, &wt);
      curL += wt * bsdf->f(-ray.D, wi) * dE;
    }
    L += (curL/shadow_rays); */
    L += direct_sun_factor*bsdf->f(-ray.D, wi) * dE;
  }
//#endif
  
  // EnvironmentLight:
  /*
  Spectrum curL = Spectrum(0.);
  for (int envRay = 0; envRay < env_rays; envRay++) {
    Spectrum dE = scene->envLight->sample_dE(scene, surf.dgShading, &wi, &wt);
    curL += wt * bsdf->f(-ray.D, wi) * dE;
  }
  L += (curL/env_rays);
  */
  
#ifdef DIRECT_SKY_LIGHT
  if (bsdf->get_specular().Intensity() == 0.) {
    Spectrum curL = Spectrum(0.);
    Vector *wo = new Vector[env_rays];
    gen_stratified_samples(wo, env_rays);
    for (int envRay = 0; envRay < env_rays; envRay++) {
      Ray r = Ray(surf.dgShading.P, wo[envRay]);
      if (skylightBound.Inside(surf.dgShading.P) ||
	  skylightBound.IntersectP(r)) {
	r.maxt = FLT_MAX; // ensure maxt is restored after IntersectP
	if (!scene->IntersectP(r)) {
	  Spectrum dE = scene->envLight->skyBase;
	  curL += bsdf->f(-ray.D, wo[envRay]) * dE;
	}
      }
    }
    L += direct_sky_factor*(curL / env_rays);
    delete wo;
  }// else {
   // cerr << "Didn't do sky light on specular surface" << endl;
  //}
#endif

  if (++RayDepth < MaxDepth) {
    for (int i = 0; i < bsdf->NumSpecular(); ++i) {
      Spectrum fr = bsdf->f_delta(i, -ray.D, &wi);
      if (fr != Spectrum(0.))
	L += scene->L(Ray(surf.dgShading.P, wi)) * fr;
    }
  }
  --RayDepth;
  delete bsdf;
  return L;
}
int PhotonMapIntegrator::RayDepth = 0;

Spectrum PhotonMapIntegrator::IndirectL(const Scene *scene, const Ray &ray,
					Surf &surf, Float *alpha) const {
  //  BSDF *bsdf = surf.getBSDF();
  if(photonmap == NULL){
    cerr << "PhotonMap is NULL!!!" << endl;
    exit(1);
  }
  else {
    float irradiance[3];
    float pos[3];
    float normal[3];
    float max_dist = max_photon_distance;
    int nphotons = photons_per_estimate;

    pos[0] = surf.dgGeom.P.x;
    pos[1] = surf.dgGeom.P.y;
    pos[2] = surf.dgGeom.P.z;
    normal[0] = surf.dgGeom.N.x;
    normal[1] = surf.dgGeom.N.y;
    normal[2] = surf.dgGeom.N.z;
    
    photonmap->irradiance_estimate(irradiance, pos, normal, max_dist, nphotons);
    Spectrum spectrum = Spectrum(irradiance[0],irradiance[1],irradiance[2]); 
    return spectrum;
  }
}

Spectrum FinalGatheringIntegrator::L(const Scene *scene,
		    const Ray &ray, Float *alpha) const {
  Surf surf;
  if(photonmap == NULL){
    cerr << "PhotonMap is NULL!!!" << endl;
    exit(1);
  } else if (scene->Intersect(ray, &surf)) {
    Spectrum L(0.);
    L += DirectL(scene, ray, surf, alpha);
    L += indirect_factor*IndirectL(scene, ray, surf, alpha);
    return L;
  } else {
    if (alpha) *alpha = 0.;
    return Spectrum(0.);
  }
}

Spectrum FinalGatheringIntegrator::DirectL(const Scene *scene,
              const Ray &ray, Surf &surf, Float *alpha) const {
  Spectrum L(0.);
  if (alpha) *alpha = 1.;
  L += surf.Le(-ray.D);
  BSDF *bsdf = surf.getBSDF();
  Vector wi;
  for (u_int i = 0; i < scene->lights.size(); ++i) {
    Spectrum dE = scene->lights[i]->dE(scene, surf.dgShading, &wi);
    L += bsdf->f(-ray.D, wi) * dE;
  }
  if (++RayDepth < MaxDepth) {
    for (int i = 0; i < bsdf->NumSpecular(); ++i) {
      Spectrum fr = bsdf->f_delta(i, -ray.D, &wi);
      if (fr != Spectrum(0.))
	L += scene->L(Ray(surf.dgShading.P, wi)) * fr;
    }
  }
  --RayDepth;
  delete bsdf;
  return L;
}
int FinalGatheringIntegrator::RayDepth = 0;

//static int gather_samples_diff = 32;
//static int gather_samples_spec = 4;
//static int gather_samples = gather_samples_diff + gather_samples_spec;
static int gather_samples = 36;

#define NO_INTERSECTION    -1.0

#if 0
Spectrum FinalGatheringIntegrator::IndirectL(const Scene *scene,
                const Ray &ray, Surf &surf, Float *alpha) const {
  //  Spectrum diffL = Spectrum(0.);
  //  Spectrum specL = Spectrum(0.);
  Spectrum L = Spectrum(0.);
  if (alpha) *alpha = 1.;
  float pos[3];
  float normal[3];
  BSDF *bsdf = surf.getBSDF();

  pos[0] = surf.dgGeom.P.x;
  pos[1] = surf.dgGeom.P.y;
  pos[2] = surf.dgGeom.P.z;
  normal[0] = surf.dgGeom.N.x;
  normal[1] = surf.dgGeom.N.y;
  normal[2] = surf.dgGeom.N.z;

  // Integrate indirect contributions over BRDFs
  Vector wo;
  Float weight;
  Ray indirectRay;
  Float indirectAlpha;
  Spectrum indirectSpectrum;
  int nSamples = 0;

  for (int i = 0; i < gather_samples; i++) {
    Spectrum f = bsdf->sample_f(-ray.D, &wo);
    weight = bsdf->weight(-ray.D, wo);
    if (f != Spectrum(0.) && weight != 0.) {
      // be sure samples are actually being calculated, not firing off into space
      int nTries = 0;
      do {
	nTries++;
	indirectRay = Ray(Point(pos[0], pos[1], pos[2]), wo);
	indirectSpectrum =
	  IndirectContribution(scene, indirectRay, &indirectAlpha);
	if (nTries >= 16) break;
	//	if (indirectAlpha == NO_INTERSECTION) cerr << "No intersection in indirect sampling...re-firing after " << nTries << "tries" << endl;
      } while(indirectAlpha == NO_INTERSECTION);
      nSamples += nTries;

      // Proper component: reflection of indirect component
      L += f * weight * indirectSpectrum * fabs(Dot(wo.Hat(), surf.dgShading.N.Hat()));
      // Visualization: indirect illumination of point
      //L += indirectSpectrum;
    }
  }

  L /= nSamples;
  return L;
}

#else

Spectrum FinalGatheringIntegrator::IndirectL(const Scene *scene,
                const Ray &ray, Surf &surf, Float *alpha) const {
  //  Spectrum diffL = Spectrum(0.);
  //  Spectrum specL = Spectrum(0.);
  Spectrum L = Spectrum(0.);
  if (alpha) *alpha = 1.;
  float pos[3];
  float normal[3];
  BSDF *bsdf = surf.getBSDF();

  pos[0] = surf.dgGeom.P.x;
  pos[1] = surf.dgGeom.P.y;
  pos[2] = surf.dgGeom.P.z;
  normal[0] = surf.dgGeom.N.x;
  normal[1] = surf.dgGeom.N.y;
  normal[2] = surf.dgGeom.N.z;

  // Integrate indirect contributions over BRDFs
  Vector *wo = new Vector[gather_samples];
//  Float weight;
  Ray indirectRay;
  Float indirectAlpha;
  Spectrum indirectSpectrum;

  gen_stratified_samples(wo, gather_samples);

  for (int i = 0; i < gather_samples; i++) {
    Spectrum f = bsdf->f(-ray.D, wo[i]);
    float weight = bsdf->weight(-ray.D, wo[i]);
    if (f != Spectrum(0.) && weight != 0.) {
      indirectRay = Ray(Point(pos[0], pos[1], pos[2]), wo[i]);
      indirectSpectrum =
	IndirectContribution(scene, indirectRay, &indirectAlpha);
      
      // Proper component: reflection of indirect component
      L += f * weight * indirectSpectrum * fabs(Dot(wo[i].Hat(), surf.dgShading.N.Hat()));
      // Visualization: indirect illumination of point
      //L += indirectSpectrum;
    }
  }

  L /= gather_samples;

  delete wo;
  return L;
}

#endif

Spectrum FinalGatheringIntegrator::IndirectContribution(
	      const Scene *scene, const Ray &ray,
	      Float *alpha) const {
  Surf surf;
  if(photonmap == NULL){
    cerr << "PhotonMap is NULL!!!" << endl;
    exit(1);
  }
  else if (scene->Intersect(ray, &surf)) {
    float irradiance[3];
    float pos[3];
    float normal[3];
    float max_dist = 500.0;
    int nphotons = 100;

    pos[0] = surf.dgGeom.P.x;
    pos[1] = surf.dgGeom.P.y;
    pos[2] = surf.dgGeom.P.z;
    normal[0] = surf.dgGeom.N.x;
    normal[1] = surf.dgGeom.N.y;
    normal[2] = surf.dgGeom.N.z;
    
    photonmap->irradiance_estimate(irradiance, pos, normal, max_dist, nphotons);
    Spectrum spectrum = Spectrum(irradiance[0],irradiance[1],irradiance[2]);
    if (alpha) *alpha = 1.;
    return spectrum;
  }
  else {
    // HACK: alpha = -1. indicates no intersection
    if (alpha) *alpha = NO_INTERSECTION;
    return Spectrum(0.);
  }
}

Spectrum RayCastingIntegrator::L(const Scene *scene,
		const Ray &ray, Float *alpha) const {
	Surf surf;
	if (scene->Intersect(ray, &surf)) {
		Spectrum L(0.);
		if (alpha) *alpha = 1.;
		L += surf.Le(-ray.D);
		BSDF *bsdf = surf.getBSDF();
		Vector wi;
		for (u_int i = 0; i < scene->lights.size(); ++i) {
			Spectrum dE = scene->lights[i]->dE(scene, surf.dgShading, &wi);
			L += bsdf->f(-ray.D, wi) * dE;
		}
		delete bsdf;
		return L;
	}
	else {
		if (alpha) *alpha = 0.;
		return Spectrum(0.);
	}
}
Spectrum WhittedIntegrator::L(const Scene *scene, const Ray &ray,
		Float *alpha) const {
	Surf surf;
	if (scene->Intersect(ray, &surf)) {
		Spectrum L(0.);
		if (alpha) *alpha = 1.;
		L += surf.Le(-ray.D);
		BSDF *bsdf = surf.getBSDF();
		Vector wi;
		for (u_int i = 0; i < scene->lights.size(); ++i) {
			Spectrum dE = scene->lights[i]->dE(scene, surf.dgShading, &wi);
			L += bsdf->f(-ray.D, wi) * dE;
		}
		if (++RayDepth < MaxDepth) {
			for (int i = 0; i < bsdf->NumSpecular(); ++i) {
				Spectrum fr = bsdf->f_delta(i, -ray.D, &wi);
				if (fr != Spectrum(0.))
					L += scene->L(Ray(surf.dgShading.P, wi)) * fr;
			}
		}
		--RayDepth;
		delete bsdf;
		return L;
	}
	else {
		if (alpha) *alpha = 0.;
		return Spectrum(0.);
	}
}
int WhittedIntegrator::RayDepth = 0;
Spectrum PathIntegrator::L(const Scene *scene,
		const Ray &r, Float *alpha) const {
	Spectrum pathWeight = 1.;
	Spectrum L = 0.;
	Ray ray = r;
	if (alpha) *alpha = 1.;
	int pathLength = 0;
	while (1) {
		Surf surf;
		if (!scene->Intersect(ray, &surf)) {
			if (pathLength == 0 && alpha) *alpha = 0.;
			break;
		}
		if (pathLength == 0)
			L += surf.Le(-ray.D);
		BSDF *bsdf = surf.getBSDF();
		Vector wi;
		Float weight;
		int lightNum = int(RandomFloat() * scene->lights.size());
		const Light *light = scene->lights[lightNum];
		Spectrum dE = light->sample_dE(scene,
			surf.dgShading, &wi, &weight);
		L += pathWeight * weight * bsdf->f(-ray.D, wi) * dE;
		Spectrum f = bsdf->sample_f(-ray.D, &wi);
		weight = bsdf->weight(-ray.D, wi);
		if (f == Spectrum(0.) || weight == 0.)
			break;
		pathWeight *= f * weight * fabs(Dot(wi.Hat(), surf.dgShading.N.Hat()));
		ray = Ray(surf.dgGeom.P, wi);
		delete bsdf;
		if (pathLength > 3 && pathWeight.Intensity() < .1) {
			Float rrProbability = .8;
			if (RandomFloat() < rrProbability)
				break;
			pathWeight /= 1. - rrProbability;
		}
		++pathLength;
	}
	return L;
}
Spectrum MCIntegrator::L(const Scene *scene,
		const Ray &r, Float *alpha) const {
	Spectrum pathWeight = 1.;
	Spectrum L = 0.;
	Ray ray = r;
	if (alpha) *alpha = 1.;
	int pathLength = 0;
	while (1) {
		Surf surf;
		if (!scene->Intersect(ray, &surf)) {
			if (pathLength == 0 && alpha) *alpha = 0.;
			break;
		}
		BSDF *bsdf = surf.getBSDF();
		Vector wi;
		Float weight, lightWeight, bsdfWeight;
		int lightNum = int(RandomFloat() * scene->lights.size());
		const Light *light = scene->lights[lightNum];
		Spectrum dE = light->sample_dE(scene,
			surf.dgShading, &wi, &lightWeight);
		bsdfWeight = bsdf->weight(-ray.D, wi);
		weight = .5 * lightWeight + .5 * bsdfWeight;
		L += pathWeight * weight * bsdf->f(-ray.D, wi) * dE;
		Spectrum f = bsdf->sample_f(-ray.D, &wi);
		bsdfWeight = bsdf->weight(-ray.D, wi);
		lightWeight = 0;
		for (u_int i = 0; i < scene->lights.size(); ++i)
			lightWeight += scene->lights[i]->weight(surf.dgGeom.P, wi);
		lightWeight /= scene->lights.size();
		weight = .5 * lightWeight + .5 * bsdfWeight;
		if (f == Spectrum(0.) || weight == 0.)
			break;
		pathWeight *= f * weight * fabs(Dot(wi.Hat(), surf.dgShading.N.Hat()));
		ray = Ray(surf.dgGeom.P, wi);
		delete bsdf;
		if (pathLength > 3 && pathWeight.Intensity() < .1) {
			Float rrProbability = .8;
			if (RandomFloat() < rrProbability)
				break;
			pathWeight /= 1. - rrProbability;
		}
		++pathLength;
	}
	return L;
}
