#include "lrt.h"
#include "primitives.h"
#include "color.h"
#include "light.h"
#include "scene.h"
#include "reflection.h"
#include "sampling.h"
#include "materials.h"
#include "transport.h"
Integrator::~Integrator() {
}
Spectrum RayCastingIntegrator::L(const Scene *scene,
		const Ray &ray, Float *alpha) const {
	Surf surf;
	if (scene->Intersect(ray, &surf)) {
		Spectrum L(0.);
		if (alpha) *alpha = 1.;
		L += surf.Le(-ray.D);
		BSDF *bsdf = surf.getBSDF();
		Vector wi;
		for (u_int i = 0; i < scene->lights.size(); ++i) {
			Spectrum dE = scene->lights[i]->dE(scene, surf.dgShading, &wi);
			L += bsdf->f(-ray.D, wi) * dE;
		}
		delete bsdf;
		return L;
	}
	else {
		if (alpha) *alpha = 0.;
		return Spectrum(0.);
	}
}
Spectrum WhittedIntegrator::L(const Scene *scene, const Ray &ray,
		Float *alpha) const {
	Surf surf;
	if (scene->Intersect(ray, &surf)) {
		Spectrum L(0.);
		if (alpha) *alpha = 1.;
		L += surf.Le(-ray.D);
		BSDF *bsdf = surf.getBSDF();
		Vector wi;
		for (u_int i = 0; i < scene->lights.size(); ++i) {
			Spectrum dE = scene->lights[i]->dE(scene, surf.dgShading, &wi);
			L += bsdf->f(-ray.D, wi) * dE;
		}
		if (++RayDepth < MaxDepth) {
			for (int i = 0; i < bsdf->NumSpecular(); ++i) {
				Spectrum fr = bsdf->f_delta(i, -ray.D, &wi);
				if (fr != Spectrum(0.))
					L += scene->L(Ray(surf.dgShading.P, wi)) * fr;
			}
		}
		--RayDepth;
		delete bsdf;
		return L;
	}
	else {
		if (alpha) *alpha = 0.;
		return Spectrum(0.);
	}
}
int WhittedIntegrator::RayDepth = 0;
Spectrum PathIntegrator::L(const Scene *scene,
		const Ray &r, Float *alpha) const {
	Vector wi;
	Surf surf;
	int pathLength = 0;
	Spectrum pathWeight = 1.;
	Spectrum L = 0.;
	Ray ray = r;
	if (alpha) *alpha = 1.;
	while (1) {
		if (!scene->Intersect(ray, &surf)) {
			if (pathLength == 0 && alpha) *alpha = 0.;
			break;
		}
		L += pathWeight * surf.Le(-ray.D);
		BSDF *bsdf = surf.getBSDF();
		int lightNum = int(RandomFloat() * scene->lights.size());
		Float weight;
		Spectrum dE = scene->lights[lightNum]->sample_dE(scene, surf.dgShading, &wi, &weight);
		L += pathWeight * weight * bsdf->f(-ray.D, wi) * dE;
		Spectrum f = bsdf->sample_f(-ray.D, &wi, &weight);
		if (f == Spectrum(0.) || weight == 0.)
			break;
		pathWeight *= f * weight;
		ray = Ray(surf.dgGeom.P, wi);
		delete bsdf;
		if (pathLength > 3 && pathWeight.Intensity() < .1) {
			Float rrProbability = .8;
			if (RandomFloat() < rrProbability)
				break;
			pathWeight /= 1. - rrProbability;
		}
		++pathLength;
	}
	return L;
}
Spectrum MCIntegrator::L(const Scene *scene,
	const Ray &r, Float *alpha) const {
	Vector wi;
	Surf surf;
	int pathLength = 0;
	Spectrum pathWeight = 1.;
	Spectrum L = 0.;
	Ray ray = r;
	if (alpha) *alpha = 1.;
	while (1) {
		if (!scene->Intersect(ray, &surf)) {
			if (pathLength == 0 && alpha) *alpha = 0.;
			break;
		}
		L += pathWeight * surf.Le(-ray.D);
		BSDF *bsdf = surf.getBSDF();
		Float weight, lightWeight, bsdfWeight;
		int lightNum = int(RandomFloat() * scene->lights.size());
		Spectrum dE = scene->lights[lightNum]->sample_dE(scene, surf.dgShading, &wi, &lightWeight);
		bsdfWeight = bsdf->weight(-ray.D, wi);
		weight = .5 * lightWeight + .5 * bsdfWeight;
		L += pathWeight * weight * bsdf->f(-ray.D, wi) * dE;
		Spectrum f = bsdf->sample_f(-ray.D, &wi, &bsdfWeight);
		lightWeight = 0;
		for (u_int i = 0; i < scene->lights.size(); ++i)
			lightWeight += scene->lights[i]->weight(surf.dgGeom.P, wi);
		lightWeight /= scene->lights.size();
		weight = .5 * lightWeight + .5 * bsdfWeight;
		if (f == Spectrum(0.) || weight == 0.)
			break;
		pathWeight *= f * weight;
		ray = Ray(surf.dgGeom.P, wi);
		delete bsdf;
		if (pathLength > 3 && pathWeight.Intensity() < .1) {
			Float rrProbability = .8;
			if (RandomFloat() < rrProbability)
				break;
			pathWeight /= 1. - rrProbability;
		}
		++pathLength;
	}
	return L;
}
