#include "ri.h"
#include "mbdofcamera.h"
#include "film.h"
#include "scene.h"
#include "transport.h"

MbDOFPerspectiveCamera::MbDOFPerspectiveCamera(const Transform *world2cam,
		int nTransforms, Float hither, Float yon,
		Float shutterOpen, float shutterClose,
		Float fstop, Float focalLen, Float focalDist,
		RtToken autoMode, RtToken meteringMode, Float fov, Film *film)
	: Camera(world2cam[0],
		Perspective(fov, 1. / 1. /*scene->image->PixelAspectRatio*/,
			hither, yon) * Scale(-1, -1, 1),
		hither, yon, film) {
  
        WorldToCamera0 = world2cam[0];
	if(nTransforms > 1) {
	  WorldToCamera1 = world2cam[1];
	} else {
	  WorldToCamera1 = world2cam[0];
	}

        CameraToWorld0 = Transform( WorldToCamera0.GetInverse() );
        CameraToWorld1 = Transform( WorldToCamera1.GetInverse() );

	Fov = fov;
	ShutterStart = shutterOpen;
	ShutterEnd = shutterClose;
	FStop = fstop;
	FocalLength = focalLen;
	FocalDistance = focalDist;
	Mode = autoMode;
	MeteringMode = meteringMode;
	cerr << "Camera Mode=" << Mode << endl;

        if ( FStop != RI_INFINITY && Mode == LRT_MANUAL )
	    film->image->Gain *= ShutterEnd/(FStop*FStop);
}

void MbDOFPerspectiveCamera::AutoFocus( const Scene *scene ) {

    // Get center of film plane

    Image *image = film->GetImage();
    Float imageWidth = image->SampleCropRight - image->SampleCropLeft;
    Float imageHeight = image->SampleCropTop - image->SampleCropBottom;
    Float centerX = 0.5 * imageWidth;
    Float centerY = 0.5 * imageHeight;

    // For all film points within a small rectangular area centered at the center
    // of the film plane, shoot a ray into the world from that film point and see 
    // what the depth of the first intersected object is.  If there is an
    // intersection, store it in a list.
    // If there are no intersections at all, increase the focusing area and
    // try again.

    Point Pcamera;
    Ray cameraRay;
    Ray worldRay;
    vector<Float> depths;
    Float medianDepth = 0;
    int roiWidth = (int) (0.05 * imageWidth);
    int roiHeight = (int) (0.05 * imageHeight);
    bool focusFound = false;

    while ( !focusFound && roiWidth <= imageWidth && roiHeight <= imageHeight ) {

        depths.clear();

        for (int i = 0; i < roiWidth; i++) {
            for (int j = 0; j < roiHeight; j++) {
                Pcamera = RasterToCamera( 
                    Point( (centerX - roiWidth/2) + i, (centerY - roiHeight/2) + j, 0 ));
                cameraRay = Ray( Point(0, 0, 0), 
                    Vector( Pcamera.x, Pcamera.y, Pcamera.z ) );
                worldRay = CameraToWorld0( cameraRay );
                
                if ( scene->IntersectP( worldRay ) ) {
                    depths.push_back( cameraRay( worldRay.maxt ).z );
                }
            }
        }

        // Get the median depth

        if ( depths.size() > 0 ) {
            vector<Float>::iterator iter = depths.begin();
            iter += depths.size() / 2;
            nth_element( depths.begin(), iter, depths.end() );

            medianDepth = *iter;
            focusFound = true;
        }
        else {
            roiWidth += (int) (0.05 * imageWidth);
            roiHeight += (int) (0.05 * imageHeight);
        }
    }

    // If we got no intersections in the whole image, the scene is empty, so
    // arbitrarily set the focal distance to 0.0.
    
    if ( depths.size() == 0 ) {
        fprintf( stderr, "Failed to find focus : Empty scene.\n" );
        FocalDistance = 0.0;
    }
    else {
        // Having gotten the median depth, we roughly know how far away the main
        // object is.  Now make another pass through the depth samples and collect
        // depth samples that lie between a certain depth range of the median
        // depth; then calculate the mean of those depths to use as our focusing
        // distance.

        Float accDepth = 0.0;
        int numSamples = 0;
        vector<Float>::iterator iter;
        for (iter = depths.begin(); iter != depths.end(); iter++) {
            if ( *iter >= (0.9 * medianDepth) && *iter <= (1.1 * medianDepth) ) {
                numSamples++;
                accDepth += *iter;
            }    
        }
        FocalDistance = accDepth / numSamples;
    }

    printf("Focal distance : %f\n", FocalDistance); 
}


void MbDOFPerspectiveCamera::ComputeAutoExposure(const Scene *scene) {

    if ( FStop == RI_INFINITY ) {
	cerr << "Infinite aperture; pinhole camera." << endl;
	return;
    }

    Image *image = film->GetImage();
    Float imageWidth = image->SampleCropRight - image->SampleCropLeft;
    Float imageHeight = image->SampleCropTop - image->SampleCropBottom;
    Float sampleX, sampleY;
    Float lum = 0;
    const int numFullFrameTestRays = 3000;
    const float centerROISizeRatio = 0.2;
    const float numCenterROITestRays = 0.5 * numFullFrameTestRays;
    const float spotAreaRatio = 0.03;
    const float numSpotTestRays = spotAreaRatio * numFullFrameTestRays;
    int numSamples = 0;

    // If the metering mode is either averaging or centerweighted averaging,
    // the whole scene is taken into account when determining exposure, so 
    // send out a bunch of rays randomly distributed over the whole scene
    // and collect their luminance values.

    if ( MeteringMode != LRT_SPOT ) {
	for (int i = 0; i < numFullFrameTestRays; i++) {
	    sampleX = RandomFloat() * imageWidth;
	    sampleY = RandomFloat() * imageHeight;

	    if ( CollectLuminance( scene, sampleX, sampleY, &lum ) )
		numSamples++;
	}
    }

    // If the metering mode is centerweighted averaging, a central region of
    // interest is important, so simulate the extra weighting by sending out
    // more rays in that region of interest.

    if ( MeteringMode == LRT_CENTERWEIGHTED_AVERAGING ) {
	for (int i = 0; i < numCenterROITestRays; i++) {
	    sampleX = (RandomFloat() * imageWidth * centerROISizeRatio) +
		(imageWidth * (1 - centerROISizeRatio)) / 2;
	    sampleY = (RandomFloat() * imageHeight * centerROISizeRatio) +
		(imageHeight * (1 - centerROISizeRatio)) / 2;

	    if ( CollectLuminance( scene, sampleX, sampleY, &lum ) )
		numSamples++;
	}
    }

    // If we are doing spot metering, only a small circular area in the middle
    // of the frame is taken into account when deciding exposure.

    if ( MeteringMode == LRT_SPOT ) {
	Float spotRadius = sqrt( spotAreaRatio * imageWidth * imageHeight / M_PI );

	for (int i = 0; i < numSpotTestRays; i++) {
	    Float theta = 2 * M_PI * RandomFloat();
	    Float r = spotRadius * sqrt( RandomFloat() );
	    sampleX = r * cos( theta ) + (imageWidth / 2);
	    sampleY = r * sin( theta ) + (imageHeight / 2);

	    if ( CollectLuminance( scene, sampleX, sampleY, &lum ) )
		numSamples++;
	}
    }

    // From the average luminance of all the test rays sent out, estimate the
    // number of stops that need to be opened up (or stopped down).  As with
    // regular light meters, the scene is assumed to be 18% gray (neutral gray,
    // or RGB = (127, 127, 127) in the digital world).

    Float numStopsUnder = log( 127 / (lum / numSamples) ) / log( 2 );

    // In aperture- or shutter-priority mode, adjust the shutter speed or f-stop
    // to bring the scene exposure as close to neutral gray as possible.
    
    if ( Mode == LRT_APERTURE_PRIORITY ) {
	ShutterEnd = 0.0025 * (4 / M_PI) * FStop * FStop * pow( 2, numStopsUnder );
    }
    else if ( Mode == LRT_SHUTTER_PRIORITY ) {
	FStop = sqrt( M_PI / (4 * 0.0025) * ShutterEnd * pow( 2, -numStopsUnder ) );
    }
    else {  
	// Program mode

	// See if we can choose a shutter speed that minimizes motion blur.
	// Consider the subject to be the object at the focusing distance.

	Point pWorld = CameraToWorld0( Point(0, 0, FocalDistance) );
	Point pCamera = WorldToCamera1( pWorld );
	Transform CameraToRaster = Transform( RasterToCamera.GetInverse() );
	Point pRaster0 = CameraToRaster( Point(0, 0, FocalDistance) );
	Point pRaster1 = CameraToRaster( pCamera );

	Float dx = fabs( pRaster0.x - pRaster1.x );
	Float dy = fabs( pRaster0.y - pRaster1.y );
	Float imageDim = imageWidth > imageHeight? imageWidth : imageHeight;
	Float imageShift = dx > dy? dx / imageDim : dy / imageDim;

	// Aim for a 1% shift

	if ( imageShift > 0.01 ) {
	    ShutterEnd = 0.01 / imageShift;
	    FStop = sqrt( M_PI / (4 * 0.0025) * ShutterEnd * pow( 2, -numStopsUnder ) );

	    if ( FStop < 1.0 ) {
		FStop = 1.0;
		ShutterEnd = 0.0025 * (4 / M_PI) * FStop * FStop * 
		    pow( 2, numStopsUnder );
	    } 
	    else if ( FStop > 64.0 ) {
		FStop = 64.0;
		ShutterEnd = 0.0025 * (4 / M_PI) * FStop * FStop * 
		    pow( 2, numStopsUnder );
	    }
	}
	else {
	    // No significant motion in this scene, so we are free to choose
	    // whatever shutter speed and aperture we like.
	    // I have chosen to use the following rule of thumb :
	    // Wide-angle - landscapes - want everything in focus.
	    // Normal - snapshots - decent depth of field.
	    // Telephoto - portraits - blur background.

	    // Wide-angle : Set FStop such that focal distance is hyperfocal distance

	    if ( Fov > 63.0 ) {
		cerr << "Wide angle" << endl;
		FStop = (FocalLength * FocalLength) / (FocalDistance * 0.005);
	    }

	    // Normal : Set FStop such that far depth of field is 1.5x the distance
	    // of the focal distance.

	    if ( Fov >= 27.0 && Fov < 63.0 ) {
		cerr << "Normal lens" << endl;
		FStop = FocalLength / (3.0 * (FocalDistance / 1000.0));
	    }

	    // Telephoto : Set FStop such that far depth of field is 1.05x the distance
	    // of the focal distance.

	    if ( Fov < 27.0 ) {
		cerr << "Telephoto" << endl;
		FStop = FocalLength / (21.0 * (FocalDistance / 1000.0));
	    }
    
	    if ( FStop > 64.0 )
		FStop = 64.0; 

	    ShutterEnd = 0.0025 * (4 / M_PI) * FStop * FStop * pow( 2, numStopsUnder );
	}
    }
 
    film->image->Gain *= ShutterEnd/(FStop*FStop);

    if ( MeteringMode == LRT_AVERAGING )
	cerr << "Exposure meter : Averaging" << endl;
    else if ( MeteringMode == LRT_CENTERWEIGHTED_AVERAGING )
	cerr << "Exposure meter : Centerweighted averaging" << endl;
    else
	cerr << "Exposure meter : Spot" << endl;

    cerr << "Num samples : " << numSamples << endl;
    cerr << "Average luminance : " << lum / numSamples << endl;
    cerr << "Shutter speed : " << ShutterEnd << " s " << endl;
    cerr << "Fstop : " << FStop << endl;
}

bool MbDOFPerspectiveCamera::CollectLuminance( const Scene *scene,
    Float sampleX, Float sampleY, Float *lum ) {

    Point Pcamera = RasterToCamera(Point(sampleX, sampleY, 0));
    Ray cameraRay = Ray(Point(0, 0, 0), Vector(Pcamera.x, Pcamera.y, Pcamera.z));
    Ray worldRay = CameraToWorld0( cameraRay );
    Spectrum L = scene->L( worldRay );
    if ( WorldToScreen( worldRay( worldRay.maxt ) ).z > 1. || 
	 L.Intensity() == 0.0 ) {
	return false;
    }

    *lum += L.Intensity(); 
    return true;
}

bool MbDOFPerspectiveCamera::GenerateRay(Float sample[5], Ray &ray) const {
   
    // Check sample against crop window

    if (sample[0] < film->GetImage()->SampleCropLeft ||
        sample[0] > film->GetImage()->SampleCropRight ||
        sample[1] < film->GetImage()->SampleCropBottom ||
        sample[1] > film->GetImage()->SampleCropTop)
        return false;

    // Compute the primary ray.

    Point Pcamera = RasterToCamera(Point(sample[0], sample[1], 0));
    Ray cameraRay = Ray(Point(0, 0, 0), Vector(Pcamera.x, Pcamera.y, Pcamera.z));

    if ( FStop != RI_INFINITY ) {

        // If we have a finite aperture, simulate depth of field.
        // Get the destination point on the focal plane

        Point focalPlanePoint; 
        Float t = FocalDistance / cameraRay.D.z;
        focalPlanePoint = cameraRay( t );

        // Choose a random point on the aperture and construct a ray connecting
        // that point to the destination point on the focal plane.

        Float apertureRadius = 0.5 * (FocalLength / FStop);
        Float theta = 2 * M_PI * sample[2];
        Float r = apertureRadius * sqrt( sample[3] );

        Point aperturePoint;
        aperturePoint.x = r * cos( theta );
        aperturePoint.y = r * sin( theta );
        aperturePoint.z = 0.0;

        cameraRay.O = aperturePoint;
        cameraRay.D = focalPlanePoint - aperturePoint;
    }

    // Now we have the ray in camera space.
    // Do motion blur by choosing a random time and transforming the ray
    // into the world space ray for that time.
 
    Ray ray0 = CameraToWorld0( cameraRay );
    Ray ray1 = CameraToWorld1( cameraRay );
/*
    Float sampleTime = sample[4] * ShutterEnd;
    if ( sampleTime > 1.0 )
	sampleTime = 1.0;
    
    ray.O.x = (1 - sampleTime) * ray0.O.x + sampleTime * ray1.O.x;
    ray.O.y = (1 - sampleTime) * ray0.O.y + sampleTime * ray1.O.y;
    ray.O.z = (1 - sampleTime) * ray0.O.z + sampleTime * ray1.O.z;

    ray.D = (1 - sampleTime) * ray0.D + sampleTime * ray1.D;
*/
    ray = ray0;

    return true;
}
