#include "scene.h"
#include "camera.h"
#include "primitives.h"
#include "image.h"
#include "transport.h"
#include "accel.h"
#include "sampling.h"
#include "shapes.h"
#include "light.h"
#include "reflection.h"

Scene::~Scene() {
	delete camera;
	delete sampler;
	delete image;
	delete integrator;
        delete causticPhotonMap;
        delete volumePhotonMap;
}
Spectrum Scene::L(const Ray &ray) const {
	return integrator->L(this, ray, NULL);
}
bool Scene::IntersectP(const Ray &ray) const {
	return prims->IntersectP(ray);
}
Scene::Scene(Camera *cam, Integrator *in, Image *img, Sampler *s,
		const vector<Primitive *> &pr,
		const vector<Light *> &lts,
                const int numCaustics, const int numVolume ) {
	lights = lts;
	prims = new GridAccelerator(pr);
	camera = cam;
	image = img;
	sampler = s;
	integrator = in;
        numCausticPhotons = numCaustics;
        numVolumePhotons = numVolume;
}
void Scene::Render() {

        BuildCausticPhotonMap();
        BuildVolumePhotonMap();

	cerr << "Rendering: ";
	Float sample[5];
	while (sampler->GetNextImageSample(sample)) {
		Ray ray;
		if (!camera->GenerateRay(sample, ray))
			continue;
		static int eyeRaysTraced = 0;
		if (eyeRaysTraced == 0)
			StatsRegisterCounter(STATS_BASIC, "Camera", "Eye Rays Traced",
				&eyeRaysTraced);
		++eyeRaysTraced;
		if (eyeRaysTraced % 10000 == 0) cerr << '+';
		Float alpha;
		Spectrum L = integrator->L(this, ray, &alpha);
		Float screenz = camera->WorldToScreen(ray(ray.maxt)).z;
		if (screenz > 1.) {
			L = 0.;
			alpha = 0.;
		}
		Point Praster(sample[0], sample[1], screenz);
		image->AddSample(Praster, L, alpha);
	}
	image->Write();
	cerr << endl;
}

void
Scene::BuildCausticPhotonMap( void ) {
    cerr << "Tracing " << numCausticPhotons << " caustic photons..." << endl;

    const int numPhotons = numCausticPhotons;
    int numLights = lights.size();
    int numEmittedPhotons = 0;
    int numGatheredPhotons = 0;

    causticPhotonMap = new PhotonMap( numPhotons );

/*    causticPhotonMap->readFromDisk( 
	"/home/weyn/348b/project/data/causticPhotons.dat",
	499999 );
    return;
*/
    int *numLightPhotons = new int[numLights];

    numLightPhotons[0] = int( 0.47/1.97 * numPhotons );
    numLightPhotons[1] = int( 1/1.97 * numPhotons );
    numLightPhotons[2] = int( 0.5/1.97 * numPhotons );

    Ray ray;
    Point lightPoint;

    for ( int i = 0; i < numLights; i++ ) {
        AreaLight *areaLight = dynamic_cast< AreaLight * >( lights[i] );
        assert( areaLight != NULL );

        numEmittedPhotons = 0;
        numGatheredPhotons = 0;

        while ( numGatheredPhotons < numLightPhotons[i] ) {
            numEmittedPhotons++;

            areaLight->sampleLightRay( &ray );
            lightPoint = ray.O;

            // Shoot ray into scene and let it bounce around specularly until it 
            // hits a diffuse surface.
           
            bool hitDiffuseSurface = false; 
            bool hitSpecularSurface = false;
            Surf surf;
            BSDF * bsdf;
            while ( Intersect( ray, &surf ) ) {
                if ( surf.primitive->areaLight != NULL )
                    break;
                bsdf = surf.getBSDF();
                if ( bsdf->NumSpecular() == 0 ) {
                    hitDiffuseSurface = true;
                    delete bsdf;
                    break;
                }
                hitSpecularSurface = true;
                Vector wr;
                bsdf->dir_delta_rr( -ray.D, &wr );
                delete bsdf;
                if ( wr.x == 0 && wr.y == 0 && wr.z == 0 )     // Photon got absorbed
                    break;
                ray = Ray( surf.dgShading.P, wr );
            } 

            if ( hitSpecularSurface && hitDiffuseSurface ) {
                float power[3];
                float pos[3];
                float dir[3];

                Spectrum flux = areaLight->B( lightPoint ) * 
                    areaLight->GetShape()->area();
		flux.ConvertToRGB( power );
                Point hitPoint( surf.dgShading.P );
                Vector lightDir( ray.D ); 
                pos[0] = hitPoint.x;
                pos[1] = hitPoint.y;
                pos[2] = hitPoint.z;
                dir[0] = lightDir.x;
                dir[1] = lightDir.y;
                dir[2] = lightDir.z;

                causticPhotonMap->store( power, pos, dir );
                numGatheredPhotons++;
            }
        }

	printf("Light %d hit rate : %.3f\n", i+1, ((float) numGatheredPhotons) / 
		numEmittedPhotons * 100);
	printf("Light %d emitted photons : %d\n", i+1, numEmittedPhotons );

        causticPhotonMap->scalePhotonPower( 1.0 / numEmittedPhotons );
    } 

    causticPhotonMap->balance();

//    causticPhotonMap->writeToDisk( "causticPhotons.dat" );
}


void
Scene::BuildVolumePhotonMap( void ) {
    cerr << "Tracing " << numVolumePhotons << " volume photons..." << endl;

    const int numPhotons = numVolumePhotons;
    int numLights = lights.size();
    int numEmittedPhotons = 0;
    int numGatheredPhotons = 0;
    int *numLightPhotons = new int[numLights];

    volumePhotonMap = new PhotonMap( numPhotons );

/*    volumePhotonMap->readFromDisk( 
	"/home/weyn/348b/project/data/volumePhotons.dat",
	100000 );
    return;
*/
    numLightPhotons[0] = int( 1.1/2.58 * numPhotons );
    numLightPhotons[1] = int( 1./2.58 * numPhotons );
    numLightPhotons[2] = int( 0.48/2.58 * numPhotons );

    Ray ray;
    Point lightPoint;

    for ( int i = 0; i < numLights; i++ ) {
        AreaLight *areaLight = dynamic_cast< AreaLight * >( lights[i] );
        assert( areaLight != NULL );

        numEmittedPhotons = 0;
        numGatheredPhotons = 0;

        while ( numGatheredPhotons < numLightPhotons[i] ) {
            numEmittedPhotons++;

            areaLight->sampleLightRay( &ray );
            lightPoint = ray.O;

            // Shoot ray into scene and let it bounce around specularly until it 
            // hits a volumetric surface/object.

            bool hitVolume = false;
            bool multipleHits = false;
            Surf surf;
            Surf * startSurf = NULL;
            BSDF * bsdf;
            Material * material;
            bool photonDead = false;

            while ( !photonDead && SafeIntersect( ray, startSurf, &surf ) ) {
                if ( surf.primitive->areaLight != NULL )
                    break;
                if ( startSurf == NULL )
                    startSurf = new Surf;
                *startSurf = surf;
                bsdf = surf.getBSDF();
                material = surf.primitive->material;
                
                if ( material->IsVolumetric() ) {
                    VolumeMaterial * volMat = dynamic_cast<VolumeMaterial *>(material);
                    assert( volMat != NULL );

                    // First entry into volumetric material.  In this case, liquid.
                    // Do Fresnel scattering, see if we reflect or refract.
                    assert( bsdf->NumSpecular() == 1 );
                    Vector wi;
                    bsdf->f_delta( 0, -ray.D, &wi );
                    delete bsdf;
                
                    if ( (Dot( -ray.D, surf.dgShading.N ) >= 0 && 
                          Dot( wi, surf.dgShading.N ) >= 0 ) ||
                         (Dot( -ray.D, surf.dgShading.N ) < 0 &&
                          Dot( wi, surf.dgShading.N ) < 0) ) {

                        // Reflected, so don't go into the volume
                        assert( wi.x != 0 && wi.y != 0 && wi.z != 0 );
                        ray = Ray( surf.dgShading.P, wi ); 
                        *startSurf = surf;
                    }
                    else {
                        // Refracted into the volume.  Do the volumetric stuff
                        // until we hit another volumetric boundary.
                        ray = Ray( surf.dgShading.P, wi );
                        Float d = -log( RandomFloat() ) / volMat->sigmaT();
                        ray.maxt = d;
                        *startSurf = surf;

                        bool inVolume = true;

                        // Test for very close volumetric boundary.
                        if ( !Intersect( ray, &surf ) ) {
                            Ray backRay;
                            backRay.O = ray.O + 0.03*ray.D;
                            backRay.D = -ray.D;
                            backRay.mint = 0;
                            backRay.maxt = 0.035;
                            Surf surf2;
                            if ( Intersect( backRay, &surf2 ) && 
                                 backRay.maxt < 0.03 - 1e-4 ) {
                                // We have a missed surface.
                                ray.O = ray.O + (0.03 - backRay.maxt + 0.001)*ray.D;
                                inVolume = false;    
                            }
                        }

                        while ( inVolume ) {
                            if ( Intersect( ray, &surf ) ) {
                                // Hit the end of the volume.  See if we reflect or
                                // refract at the Fresnel surface, again.
                                Float hitTime = ray.maxt;
                                material = surf.primitive->material;
                                //fprintf( stderr, "%d ", numEmittedPhotons );
                                if ( !material->IsVolumetric() ) {
                                    photonDead = true;
                                    fprintf( stderr, "Hm. " );
                                    break;
                                }
                                assert( material->IsVolumetric() );
                                volMat = dynamic_cast<VolumeMaterial *>(material);
                                bsdf = surf.getBSDF();
                                bsdf->f_delta( 0, -ray.D, &wi );
                                delete bsdf;

                                if ( (Dot( -ray.D, surf.dgShading.N ) >= 0 && 
                                      Dot( wi, surf.dgShading.N ) >= 0 ) ||
                                     (Dot( -ray.D, surf.dgShading.N ) < 0 &&
                                      Dot( wi, surf.dgShading.N ) < 0) ) {

                                    // Reflected, so go back into the volume
                                    assert( wi.x != 0 && wi.y != 0 && wi.z != 0 );
                                    ray = Ray( surf.dgShading.P, wi ); 
                                    ray.maxt = d - hitTime;
                                }
                                else {
                                    // Refracted.  Leaving the volume.
                                    ray = Ray( surf.dgShading.P, wi ); 
                                    *startSurf = surf;
                                    break;
                                }
                            } 
                            else {
                                // Photon got scattered or absorbed before hitting the
                                // end of the volume.
                                float power[3];
                                float pos[3];
                                float dir[3];
                               
                                Spectrum flux = areaLight->B( lightPoint ) * 
                                    areaLight->GetShape()->area();
                                flux *= volMat->color();
                                flux.ConvertToRGB( power );
                                Point interactionPoint( ray( ray.maxt ) );
                                pos[0] = interactionPoint.x;
                                pos[1] = interactionPoint.y;
                                pos[2] = interactionPoint.z;
                                dir[0] = ray.D.x;
                                dir[1] = ray.D.y;
                                dir[2] = ray.D.z;

                                if ( hitVolume ) 
                                    multipleHits = true;
                                if ( multipleHits )
                                    volumePhotonMap->store( power, pos, dir );
                                hitVolume = true;

                                // Russian roulette to figure out what happened.
                                if ( RandomFloat() <= volMat->albedo() ) {
                                    // Scattered
                                    volMat->SamplePhaseScatter( ray.D, &wi );
                                    ray = Ray( ray( ray.maxt ), wi );     
                                    ray.mint = 0;  // Don't want that 1e-3 hack
                                    d = -log( RandomFloat() ) / volMat->sigmaT();
                                    ray.maxt = d;
                                }
                                else {
                                    // Absorbed, end of story for this photon.
                                    photonDead = true;
                                    break;
                                }
                            }  // if Intersect( ray, surf );
                        }  // While loop.  Exited when photon is no longer in volume.

                    }  // if ray reflected/refracted into volume
                } 
                else {
                    // Non-volumetric surface hit
                    // Bounce around specularly
                    if ( bsdf->NumSpecular() == 0 ) {
                        delete bsdf;
                        break;
                    }
                    Vector wi;
                    bsdf->dir_delta_rr( -ray.D, &wi );
                    delete bsdf;

                    if ( wi.x == 0 && wi.y == 0 && wi.z == 0 )
                        break;      
                    ray = Ray( surf.dgShading.P, wi ); 
                    *startSurf = surf;
                }
            }  // End of while ( Intersect( ray, surf ) )

            if ( hitVolume && multipleHits )
                numGatheredPhotons++;
    
            delete startSurf;

        } // while ( numGatheredPhotons < ((float) numPhotons) / numLights ) 

        volumePhotonMap->scalePhotonPower( 1.0 / numEmittedPhotons );

	printf("Light %d hit rate : %.3f\n", i+1, ((float) numGatheredPhotons) / 
		numEmittedPhotons * 100);
	printf("Light %d emitted photons : %d\n", i+1, numEmittedPhotons );
    } 

    volumePhotonMap->balance();

//    volumePhotonMap->writeToDisk( "volumePhotons.dat" );
}

bool Scene::Intersect(const Ray &ray, Surf *surf) const {
	return prims->Intersect(ray, surf);
}

bool Scene::SafeIntersect( const Ray &ray, Surf * startSurf, Surf *surf ) const {
    Ray backRay;
    Surf surf2;
    
    bool forwardIntersected = Intersect( ray, surf );

    if ( startSurf == NULL )
        return forwardIntersected;

    if ( forwardIntersected ) {
        backRay.O = ray( ray.maxt );
        backRay.D = -ray.D;
        backRay.mint = ray.maxt - 0.03;
        if ( !Intersect( backRay, &surf2 ) )
            return false;
        if ( surf2.primitive != startSurf->primitive && backRay.maxt < ray.maxt ) {
            // We have a missed surface.
            ray.maxt = ray.maxt - backRay.maxt;
            *surf = surf2;
        }
        return true;
    }

    // No forward intersection
    backRay.O = ray.O + 0.03*ray.D;
    backRay.D = -ray.D;
    backRay.mint = 0;
    if ( !Intersect( backRay, &surf2 ) )
        return false;

    if ( surf2.primitive != startSurf->primitive && backRay.maxt < 0.03 ) {
        // We have a missed surface.
        ray.maxt = 0.03 - backRay.maxt;
        *surf = surf2;
        return true;
    }

    return false;
} 
