#include "lrt.h"
#include "primitives.h"
#include "color.h"
#include "light.h"
#include "scene.h"
#include "reflection.h"
#include "sampling.h"
#include "materials.h"
#include "transport.h"
Integrator::~Integrator() {
}
Spectrum RayCastingIntegrator::L(const Scene *scene,
		const Ray &ray, Float *alpha) const {
	Surf surf;
	if (scene->Intersect(ray, &surf)) {
		Spectrum L(0.);
		if (alpha) *alpha = 1.;
		L += surf.Le(-ray.D);
		BSDF *bsdf = surf.getBSDF();
		Vector wi;
		for (u_int i = 0; i < scene->lights.size(); ++i) {
			Spectrum dE = scene->lights[i]->dE(scene, surf.dgShading, &wi);
			L += bsdf->f(-ray.D, wi) * dE;
		}
		delete bsdf;
		return L;
	}
	else {
		if (alpha) *alpha = 0.;
		return Spectrum(0.);
	}
}
Spectrum WhittedIntegrator::L(const Scene *scene, const Ray &ray,
		Float *alpha) const {
	Surf surf;
	if (scene->Intersect(ray, &surf)) {
		Spectrum L(0.);
		if (alpha) *alpha = 1.;
		L += surf.Le(-ray.D);
		BSDF *bsdf = surf.getBSDF();
		Vector wi;
		for (u_int i = 0; i < scene->lights.size(); ++i) {
			Spectrum dE = scene->lights[i]->dE(scene, surf.dgShading, &wi);
			L += bsdf->f(-ray.D, wi) * dE;
		}
		if (++RayDepth < MaxDepth) {
			for (int i = 0; i < bsdf->NumSpecular(); ++i) {
				Spectrum fr = bsdf->f_delta(i, -ray.D, &wi);
				if (fr != Spectrum(0.))
					L += scene->L(Ray(surf.dgShading.P, wi)) * fr;
			}
		}
		--RayDepth;
		delete bsdf;
		return L;
	}
	else {
		if (alpha) *alpha = 0.;
		return Spectrum(0.);
	}
}
int WhittedIntegrator::RayDepth = 0;
Spectrum PathIntegrator::L(const Scene *scene,
		const Ray &r, Float *alpha) const {
    static int rayDepth = 0;
    Spectrum pathWeight = 1.;
    static Spectrum cumPathWeight(1.);
    Spectrum L = 0.;
    Ray ray = r;
    if (alpha) *alpha = 1.;
    static int pathLength = 0;
    static int rayCount = 0;
    int numPathsThisCall = 0; 
    rayCount++;
//    fprintf( stderr, "%d ", rayCount );
//    if ( rayCount == 6492 )
//        int a = 0;

    while (1) {
        Surf surf;
        if (!scene->Intersect(ray, &surf)) {
                if (pathLength == 0 && alpha) *alpha = 0.;
                break;
        }
        if (pathLength == 0)
            L += surf.Le(-ray.D);
        if ( surf.primitive->areaLight != NULL ) 
            break;
        
        BSDF *bsdf = surf.getBSDF();
        Vector wi;
        Float weight;

        if ( surf.primitive->material->IsVolumetric() ) {
	    delete bsdf;

            // Entering a volumetric body
	  
	    static int volPathCount = 0;
	    volPathCount++;
	    if ( volPathCount > 2 ) {
		volPathCount = 0;
		break;
	    }
	
            Spectrum volumeL = GatherVolumetricRadiance( scene, ray, &surf );
    
            L += cumPathWeight * volumeL;

            break;
        }
        else if ( bsdf->NumSpecular() == 0 ) {  // Non-specular surface

            Spectrum photonL = GatherCausticsRadiance( scene, ray, bsdf );

            L += cumPathWeight * photonL * 0.2;

            float randFloat;
            while ( (u_int) (randFloat = RandomFloat() * scene->lights.size()) 
                == scene->lights.size() )
                {} 
            int lightNum = (int) randFloat;
            const Light *light = scene->lights[lightNum];
            Spectrum dE = light->sample_dE(scene,
                    surf.dgShading, &wi, &weight);
            L += cumPathWeight * weight * bsdf->f(-ray.D, wi) * dE;

            Spectrum f = bsdf->sample_f(-ray.D, &wi);
            weight = bsdf->weight(-ray.D, wi);
            delete bsdf;
            if (f == Spectrum(0.) || weight == 0.)
                    break;
            pathWeight *= f * weight * fabs(
                Dot(wi.Hat(), surf.dgShading.N.Hat()));
	    cumPathWeight *= pathWeight;

            ray = Ray(surf.dgGeom.P, wi);
            if (pathLength > 3 && cumPathWeight.Intensity() < .1) {
                    Float rrProbability = .8;
                    if (RandomFloat() < rrProbability)
                            break;
                    cumPathWeight /= 1. - rrProbability;
            }

            ++pathLength;
	    ++numPathsThisCall;
        }
        else {  // Specular surface
            rayDepth++;
            if ( rayDepth < 15 ) {
                float randFloat;
/*
                while ( (int) (randFloat = RandomFloat() * bsdf->NumSpecular())
                    == bsdf->NumSpecular() ) 
                    {} 
                int which = (int) randFloat;
*/
                int which = 0;

                if ( bsdf->NumSpecular() == 2 ) {
                    if ( RandomFloat() <= 0.05 )
                        which = 0;
                    else
                        which = 1;
                }
                        
                Spectrum f = bsdf->f_delta( which, -ray.D, &wi );

                if ( f != Spectrum(0.) ) 
                    L += scene->L( Ray( surf.dgGeom.P, wi ) ) * f;
            }
            delete bsdf;
            rayDepth--;
            break;
        }
    }
    pathLength -= numPathsThisCall;

    if ( pathLength == 0 )
	cumPathWeight = Spectrum(1.);

    return L;
}
          
Spectrum
PathIntegrator::GatherCausticsRadiance( const Scene * scene, const Ray & ray, 
    const BSDF * bsdf ) const {
 
    float maxDist = 1.;
    int nPhotons = 150;

/*       
    float irrad[3];
    float pos[3];
    float normal[3];

    Point hitPoint = ray( ray.maxt );
    pos[0] = hitPoint.x;
    pos[1] = hitPoint.y;
    pos[2] = hitPoint.z;
    Normal norm = surf.dgShading.N;
    normal[0] = norm.x;
    normal[1] = norm.y;
    normal[2] = norm.z;

    scene->photonMap->irradianceEstimate( 
        irrad, pos, normal, maxDist, nPhotons );

    if ( !(irrad[0] == 0. && irrad[1] == 0. && irrad[2] == 0.) ) {
//                L += Spectrum( irrad[0], irrad[1], irrad[2] ) + Spectrum(500.);
        L = Spectrum( 0, 0, 1000 );
//                cerr << "Found photon\n" << endl;
    }
*/

    Point hitPoint = ray( ray.maxt );

    NearestPhotons *np;
    np = new NearestPhotons;
    np->dist2 = new float[nPhotons+1];
    np->index = (const Photon **)new (Photon*)[nPhotons+1]; 
    np->pos[0] = hitPoint.x;
    np->pos[1] = hitPoint.y;
    np->pos[2] = hitPoint.z;
    np->max = nPhotons;
    np->found = 0;
    np->gotHeap = 0;
    np->dist2[0] = maxDist * maxDist;

    scene->causticPhotonMap->locatePhotons( np, 1 );

    Spectrum photonL(0.);

    if ( np->found >= 8 ) {
        float pdir[3];

        for ( int i = 1; i <= np->found; i++ ) {
            const Photon * p = np->index[i];
            scene->causticPhotonMap->photonDir( pdir, p );
            Spectrum power = Spectrum( p->power[0], p->power[1], p->power[2] );
            Vector pDirVec( -pdir[0], -pdir[1], -pdir[2] );
            photonL += bsdf->f( -ray.D, pDirVec ) * power; 
        }

        photonL /= M_PI * np->dist2[0];
/*
        Float s[3];
        photonL.ConvertToRGB( s );
        L = Spectrum( 0, 0, 10000 ); 
*/            }

    delete [] np->dist2;
    delete [] np->index;
    delete np;
    
    return photonL;
}

Spectrum
PathIntegrator::GatherVolumetricRadiance( const Scene * scene, const Ray & r, 
    const Surf * surf ) const {

    VolumeMaterial * material = dynamic_cast<VolumeMaterial *>
        (surf->primitive->material); 
    assert( material != NULL );

    Ray ray = r;
    BSDF * bsdf = surf->getBSDF();
    Float sigmaS = material->sigmaS();
    Float sigmaT = material->sigmaT();
    const Float dx = 0.2;
    Float expSigmaTdx = exp( -sigmaT * dx );
    Spectrum L(0.);
    Float maxDist = 1.;
    int nPhotons = 150;
    Spectrum weight(1.);
    float phaseScatterNormalizer = 5.;

    // Reflect / refract off the specular surface first

    assert( bsdf->NumSpecular() == 1 );
    Vector wi;
    Spectrum f = bsdf->f_delta( 0, -ray.D, &wi );
    delete bsdf;

    if ( Dot( -ray.D, surf->dgGeom.N ) >= 0 && 
         Dot( wi, surf->dgGeom.N ) >= 0 )  {
        
        // Reflected, so don't go into the volume
        L = scene->L( Ray( surf->dgGeom.P, wi ) ) * f;
        return L;
    }

    if ( Dot( -ray.D, surf->dgGeom.N ) < 0 ) {
	fprintf( stderr, "Entered volume from inside!\n" );
	return L;
    }

    // Refracted into the volume.  Start ray-marching.

    ray = Ray( surf->dgGeom.P, wi );
    Surf farSurf;
    Point farPoint;

    if ( scene->Intersect( ray, &farSurf ) ) {
            farPoint = farSurf.dgGeom.P;            
    }
    else {
        // Too close to far surface.  Don't do any volumetric radiance gathering,
        // just interact with the far surface directly.
        Ray backRay;
        backRay.O = ray( 0.03 );
        backRay.D = -ray.D;
        backRay.maxt = 0.03;
        if ( scene->Intersect( backRay, &farSurf ) && backRay.maxt < 0.03 - 1e-4 ) {
            farPoint = farSurf.dgGeom.P;
        }
        else {
            // Internal error.  Far surface didn't seem to exist.
            fprintf( stderr, "Far surface didn't exist upon entry.\n" );
            return L;
        }
    }

    if ( !farSurf.primitive->material->IsVolumetric() ) {
	// We have an alien object inside the volumetric body.
	BSDF * alienBsdf = farSurf.getBSDF();
	if ( alienBsdf->NumSpecular() != 0 ) {
	    fprintf( stderr, "Unknown foreign object in volume.\n");
	    delete alienBsdf;
	    return L;
	}

        alienBsdf->sample_f(-ray.D, &wi);
	delete alienBsdf;

	Ray nextRay( farSurf.dgGeom.P, wi );
	Surf nextSurf;

	if ( !scene->Intersect( nextRay, &nextSurf ) ) {
	    fprintf( stderr, "Failed to hit far surface after foreign hit.\n" );
	    return L;
	}
    
	if ( !nextSurf.primitive->material->IsVolumetric() ) {
	    fprintf( stderr, "Non-volumetric hit after foreign hit.\n" );
	    return L;
	}
    }

    Float t = 0.;
    Spectrum segmentL;

    while ( true ) {

	if ( weight.Intensity() < 0.05 )
	    break;

        // Check if there is enough space left to march, or we're hitting the
        // far surface already.

        Float d =  (farPoint - ray( t )).Length();
        if ( d < dx ) {
            bsdf = farSurf.getBSDF(); 

	    // Check if this is a far volumetric boundary or we are reflecting off
	    // an alien object in the volume.

	    if ( bsdf->NumSpecular() == 0 ) {
		// Gather irradiance using photon map

		Point hitPoint = farSurf.dgGeom.P;

		NearestPhotons *np;
		np = new NearestPhotons;
		np->dist2 = new float[nPhotons+1];
		np->index = (const Photon **)new (Photon*)[nPhotons+1]; 
		np->pos[0] = hitPoint.x;
		np->pos[1] = hitPoint.y;
		np->pos[2] = hitPoint.z;
		np->max = nPhotons;
		np->found = 0;
		np->gotHeap = 0;
		np->dist2[0] = maxDist * maxDist;

		scene->volumePhotonMap->locatePhotons( np, 1 );

		Spectrum photonL(0.);

		if ( np->found >= 8 ) {
		    float pdir[3];

		    for ( int i = 1; i <= np->found; i++ ) {
			const Photon * p = np->index[i];
			scene->volumePhotonMap->photonDir( pdir, p );
			Spectrum power = 
			    Spectrum( p->power[0], p->power[1], p->power[2] );
			Vector pDirVec( pdir[0], pdir[1], pdir[2] );
			photonL += bsdf->f( -ray.D, -pDirVec ) * power; 
		    }

		    photonL /= (4./3.) * M_PI * maxDist * maxDist * maxDist;
		}

		L += photonL;
	    
		Spectrum f = bsdf->sample_f(-ray.D, &wi);
		Spectrum alienBsdfWeight = bsdf->weight(-ray.D, wi);
		delete bsdf;
		if (f == Spectrum(0.) || alienBsdfWeight == 0.) {
		    return L;
		}
		weight *= f * alienBsdfWeight * fabs(
		    Dot(wi.Hat(), farSurf.dgShading.N.Hat()));

		ray = Ray( farSurf.dgGeom.P, wi );

		if ( !scene->Intersect( ray, &farSurf ) ) {
		    fprintf( stderr, "Failed to hit far surface after "
			"secondary foreign hit.\n" );
		    return L;
		}
	    
		if ( !farSurf.primitive->material->IsVolumetric() ) {
		    fprintf( stderr, "Non-volumetric hit after "
			"secondary foreign hit.\n" );
		    return L;
		}
	    }
	    else {
		// Hit volumetric surface, not foreign object

		Spectrum f = bsdf->f_delta( 0, -ray.D, &wi );
		delete bsdf;

		if ( Dot( -ray.D, farSurf.dgGeom.N ) >= 0 ) {
		    fprintf( stderr, "Tried to leave volume from outside!\n" );
		    return L;
		}

		if ( Dot( -ray.D, farSurf.dgGeom.N ) < 0 &&
		     Dot( wi, farSurf.dgGeom.N ) < 0) {
		    
		    // Reflected, so go back into the volume and continue marching
		    ray = Ray( farSurf.dgGeom.P, wi );
		    t = 0;
		    weight *= f;

		    // Find the far surface
		    if ( scene->Intersect( ray, &farSurf ) && 
			 farSurf.primitive->material->IsVolumetric() ) {
			    farPoint = farSurf.dgGeom.P;            
		    }
		    else {
			// Too close to far surface.  
			Ray backRay;
			backRay.O = ray( 0.03 );
			backRay.D = -ray.D;
			backRay.maxt = 0.03;
			if ( scene->Intersect( backRay, &farSurf ) && 
			     backRay.maxt < 0.03 - 1e-4 ) {
                            if ( !farSurf.primitive->material->IsVolumetric() ) {
                                return L;
                            }
			    farPoint = farSurf.dgGeom.P;
			}
			else {
			    // Internal error.  Far surface didn't seem to exist.
			    fprintf( stderr, "Far surface didn't exist.\n" );
			    return L;
			}
		    }
		}
		else {
		    // Ray got refracted out of the volume, so quit marching
		    L += scene->L( Ray( farSurf.dgGeom.P, wi ) ) * f;        
		    return L;
		}
	    }
	}

        // Sum direct illumination contribution from all lights

        segmentL = Spectrum(0.);

        for ( u_int i = 0; i < scene->lights.size(); i++ ) {
            AreaLight *areaLight = dynamic_cast< AreaLight * >( scene->lights[i] );
            assert( areaLight != NULL );

            DifferentialGeometry tmpDG;
            tmpDG.P = ray( t );
            areaLight->GetShape()->sample( tmpDG, &wi );
    
            Ray lightRay( ray( t ), wi );
            Surf hitSurf;
            Float volSurfT = 0.;

            if ( !scene->Intersect( lightRay, &hitSurf ) )
                continue;

            if ( hitSurf.primitive->material->IsVolumetric() ) {
                volSurfT = lightRay.maxt;
                lightRay = Ray( lightRay( lightRay.maxt ), wi );
            }
           
            if ( !scene->Intersect( lightRay, &hitSurf ) )
                continue;

            if ( hitSurf.primitive->areaLight == NULL )
                continue; 

            // At this point, we have an unoccluded light source.
           
            Spectrum lightL = areaLight->L( lightRay( lightRay.maxt ), -wi );
            lightL *= material->color() * exp( -sigmaT * volSurfT );
            lightL *= material->PhaseScatter( -wi, -ray.D ) / phaseScatterNormalizer;
            lightL *= sigmaS * dx;
             
            segmentL += lightL;
        }

        // Now add multiple scattering using the photon map

	Point hitPoint = ray( t );

	NearestPhotons *np;
	np = new NearestPhotons;
	np->dist2 = new float[nPhotons+1];
	np->index = (const Photon **)new (Photon*)[nPhotons+1]; 
	np->pos[0] = hitPoint.x;
	np->pos[1] = hitPoint.y;
	np->pos[2] = hitPoint.z;
	np->max = nPhotons;
	np->found = 0;
	np->gotHeap = 0;
	np->dist2[0] = maxDist * maxDist;

	scene->volumePhotonMap->locatePhotons( np, 1 );

	Spectrum photonL(0.);

	if ( np->found >= 8 ) {
	    float pdir[3];

	    for ( int i = 1; i <= np->found; i++ ) {
		const Photon * p = np->index[i];
		scene->volumePhotonMap->photonDir( pdir, p );
		Spectrum power = Spectrum( p->power[0], p->power[1], p->power[2] );
		Vector pDirVec( pdir[0], pdir[1], pdir[2] );
		photonL += material->PhaseScatter( pDirVec, -ray.D ) * power /
		    phaseScatterNormalizer; 
	    }

	    photonL /= (4./3.) * M_PI * maxDist * maxDist * maxDist;
	    photonL *= dx;
    /*
	    Float s[3];
	    photonL.ConvertToRGB( s );
	    L = Spectrum( 0, 0, 10000 ); 
    */            
	}

	segmentL += photonL;

	delete [] np->dist2;
	delete [] np->index;
	delete np;

        // Add segment radiance and adjust weight for the recursive attenuation

        L += segmentL * weight;
        weight *= expSigmaTdx;

        t += dx; 
    }

    return L;
}

Spectrum MCIntegrator::L(const Scene *scene,
		const Ray &r, Float *alpha) const {
	Spectrum pathWeight = 1.;
	Spectrum L = 0.;
	Ray ray = r;
	if (alpha) *alpha = 1.;
	int pathLength = 0;
	while (1) {
		Surf surf;
		if (!scene->Intersect(ray, &surf)) {
			if (pathLength == 0 && alpha) *alpha = 0.;
			break;
		}
		BSDF *bsdf = surf.getBSDF();
		Vector wi;
		Float weight, lightWeight, bsdfWeight;
		int lightNum = int(RandomFloat() * scene->lights.size());
		const Light *light = scene->lights[lightNum];
		Spectrum dE = light->sample_dE(scene,
			surf.dgShading, &wi, &lightWeight);
		bsdfWeight = bsdf->weight(-ray.D, wi);
		weight = .5 * lightWeight + .5 * bsdfWeight;
                
                weight = lightWeight;

		L += pathWeight * weight * bsdf->f(-ray.D, wi) * dE;
		Spectrum f = bsdf->sample_f(-ray.D, &wi);
		bsdfWeight = bsdf->weight(-ray.D, wi);
		lightWeight = 0;
		for (u_int i = 0; i < scene->lights.size(); ++i)
			lightWeight += scene->lights[i]->weight(surf.dgGeom.P, wi);
		lightWeight /= scene->lights.size();
		weight = .5 * lightWeight + .5 * bsdfWeight;
        
//                weight = lightWeight;

		if (f == Spectrum(0.) || weight == 0.)
			break;
		pathWeight *= f * weight * fabs(Dot(wi.Hat(), surf.dgShading.N.Hat()));
		ray = Ray(surf.dgGeom.P, wi);
		delete bsdf;
		if (pathLength > 3 && pathWeight.Intensity() < .1) {
			Float rrProbability = .8;
			if (RandomFloat() < rrProbability)
				break;
			pathWeight /= 1. - rrProbability;
		}
		++pathLength;
	}
	return L;
}
