/*
 * This file is the center piece of the thin film interferance implementation.
 * It implements a new material "abalone" which can be specified as a material
 * in a RIB file.  
 *
 * For convience all of the shadder's are rapped up into one shadder called
 * AbaloneMicrofacet.  This shader blends a background color (as specified in
 * the RIB) modulated with a noise function with a specular highligh & the thin
 * film interferance. 
 *
 * The thin film interferance is calculated by using a 2D noise function
 * to modulate the distance between the outer and inner surface of the shell.
 * Then taking into account the angle of the light and that of the camera, the distance
 * to and from the inner to outer surface is caluclated.  This distance is then compared
 * to the optimum distance for Red, Green, and Blue light.  An interferance pattern
 * is calculated based upon this.  
 */



#include "lrt.h"
#include "color.h"
#include "reflection.h"
#include "abalone.h"
#include "shapes.h"
#include "perlin.h"



AbaloneMicrofacet::AbaloneMicrofacet(const Spectrum &reflectance,
	MicrofacetDistribution *d, const DifferentialGeometry *dg) {
	R = reflectance;
	distribution = d;
	N = dg->N;
	N.Normalize();
	u = dg->u;
	v = dg->v;
}
AbaloneMicrofacet::~AbaloneMicrofacet() {
	delete distribution;
}
Spectrum AbaloneMicrofacet::f(const Vector &wi, const Vector &wo) const {
	if (sameHemisphere(wi, wo)) {
		double kp = PerlinNoise2D(u, v, 2, 2, 10);
	
		
		float d = 700.0 + 500.0 * kp;
		float n = 1.53;  //index of refraction of Chitin
		
		//Calc to bottom from Observer's perspective

		Vector wiN = wi.Hat();
		float costheta = Dot(N, wiN);	
		//Length of the wo ray to the inner surface.
		float distance = (d)/(fabs(costheta));		

		//calc distance to bottom from light source
		Vector woN = wo.Hat();
		costheta = Dot(N, woN);
		//Length of the wo ray to the inner surface.
		distance += (d)/((costheta));		//add the 2 distances
	

		//get the values for rgb.
		float minD, maxD, r, g, b;
		int factor;
		float relative_dis = distance;
		//red light
		
		factor = (int)(distance / 690.0);
		relative_dis = distance-factor*690.0;

		minD = (690.0)/n;
		maxD = (690.0*(1.5))/n;
		if(relative_dis < minD) {
			relative_dis += 690.0;
			minD *=2.0;
			r = (fabs((minD-maxD)-(relative_dis-maxD))/(minD-maxD));
		} else {

			r = 1.0- (fabs((maxD-minD)-(relative_dis-minD))/(maxD-minD));
		}
		if(r > 1.0) {
			r = 2.0-r;
			
		}

		//green light
		factor = (int) (distance / 540);
		relative_dis = distance-factor*540.0;

		minD = 540.0/n;
		maxD = (540.0*(1.5))/n;
		if(relative_dis < minD) {
			relative_dis += 540.0;
			minD *=2.0;
			g = (fabs((minD-maxD)-(relative_dis-maxD))/(minD-maxD));
		} else {

			g = 1.0- (fabs((maxD-minD)-(relative_dis-minD))/(maxD-minD));
		}
		if(g > 1.0) {
			g = 2.0-g;			
		}

		//blue light
		factor = (int)(distance / 475);
		relative_dis = distance-factor*475.0;

		minD = (475.0)/n;
		maxD = (475.0*(1.5))/n;
		if(relative_dis < minD) {
			relative_dis += 475.0;
			minD *=2.0;
			b = ((fabs((minD-maxD)-(relative_dis-maxD)))/(minD-maxD));
		} else {

			b = 1.0- (fabs((maxD-minD)-(relative_dis-minD))/(maxD-minD));
		}
		if(b > 1.0) {
			b = 2.0-b;
		
		}

		//get back ground color
		Spectrum bg = BackGroundColor(wi, wo);

		return((bg*.3 + (Spectrum(r, g, b)*.01)) * 2.0);

	}
	else return 0.;
}

/*
 * This function caclulates the background color based upon
 * 2d noise and the color specified by the RIB file.  It also
 * adds a specular highligh.
 */
Spectrum AbaloneMicrofacet::BackGroundColor(const Vector &wi, const Vector &wo) const {
	double factor = PerlinNoise2D(u, v, 3, 2, 10);

	//control the range of the noise
	if(factor > 1.) factor = 1.;
	if(factor < -1.) factor = -1.;
	factor += 1.;
	factor *= .5;


	Normal norm = N;

	Normal reflct = 2.0 * norm*(Dot(norm,wo))-Normal(wo);
	reflct.Normalize();
	float costheta = fabs(Dot(wi, reflct));
	float specHighlight = 1.0 * costheta;

	Spectrum difuse = (R*factor/M_PI);
	

	float rgb[3];
	difuse.ConvertToRGB(rgb);

	return((R*factor/M_PI)*2.5 + Spectrum(specHighlight) * .1);

}

void AbaloneMicrofacet::sample_f(const Vector &wo, Vector *wi) const {

	distribution->sample_f(wo, wi);
}
Float AbaloneMicrofacet::weight(const Vector &wo, const Vector &wi) const {

	return distribution->weight(wo, wi);
}
