
#include "heightfield.h"

// RRG
#include <stdio.h>
#include <stdlib.h>


// For ease of use, some global statics used to help define the 
// coordinate system of the notmal used for bump mapping. 

static Vector Xaxis[3], Yaxis[3];
static int axWhich;
static Vector fX, fY;

/* Heightfields are now being generated by use of the Diamond-Square
 * fractal algorithm, not by reading a grid of z-coordinates from a
 * file. In the heightfield constructor, we now create an instance 
 * of the Fractal class, and use the data generated by that class to 
 * initialize the z-coordinates of the heightfield's grid. 
 *
 */

Heightfield::Heightfield(const Transform &o2w, int x, int y, float *zs, bool forBump)
	: Shape(o2w) 
{
  nx = ny = DEGREE;
  z = new float[ny * nx];
  myFractal = new Fractal(DEGREE, forBump);

  myFractal->Transfer(z);

  // RRG: Store the bounding box so we don't 
  // have to recompute at every Intersect() call.
  
  BBox tempBox = Bound(); 
  mBBox = new BBox(tempBox.pMin, tempBox.pMax);
  
  // Set the offset.
  error = 0.0001;
}



Heightfield::~Heightfield() 
{
  delete (myFractal);
  delete[] z;
}

BBox Heightfield::Bound() const 
{
	float minz = z[0], maxz = z[0];

	for (int i = 1; i < nx*ny; ++i) 
	  {
	    if (z[i] < minz) minz = z[i];
	    if (z[i] > maxz) maxz = z[i];
	  }
	return BBox(Point(0,0,minz), Point(1,1,maxz));
}

bool Heightfield::CanIntersect() const 
{
  // RRG
  return (true);
}

void Heightfield::Refine(vector<Shape *> &refined) const 
{
	int ntris = 2*(nx-1)*(ny-1);
	int *verts = new int[3*ntris];
	Point *P = new Point[nx*ny];
	int x, y;
	P = new Point[nx*ny];
	int pos = 0;
	for (y = 0; y < ny; ++y) {
		for (x = 0; x < nx; ++x) {
			P[pos].x = (float)x / (float)(nx-1);
			P[pos].y = (float)y / (float)(ny-1);
			P[pos].z = z[pos];
			++pos;
		}
	}
	int *vp = verts;
	for (y = 0; y < ny-1; ++y) {
		for (x = 0; x < nx-1; ++x) {
	#define VERT(x,y) ((x)+(y)*nx)
			*vp++ = VERT(x, y);
			*vp++ = VERT(x+1, y);
			*vp++ = VERT(x+1, y+1);
	
			*vp++ = VERT(x, y);
			*vp++ = VERT(x+1, y+1);
			*vp++ = VERT(x, y+1);
		}
	#undef VERT
	}
	refined.push_back(new TriangleMesh(ObjectToWorld, ntris,
		nx*ny, verts, P));
	delete[] P;
	delete[] verts;
}


/* Triangulate()
 * =============
 * RRG: Generate a 2-triangle (1-cell) mesh that can be used for
 * intersection tests with a particular ray. (myX, myY) represent
 * the lower left-hand corner of the cell we're interested in. If
 * the remaining parameters are properly specified, this routine
 * instead adds the basic (uninterpolated) normals of the cell's
 * triangles to the "norms" param as part of the Phong
 * interpolation process. 
 */

TriangleMesh *Heightfield::Triangulate(int myX, int myY, bool tri1, bool tri2, Vector &norms, int &num) const
{
  int ntris = 2;
  int *verts = new int[3*ntris];
  Point *P = new Point[4];
  int x, y;

  int pos = 0;

  // Make sure the given coord can be the lower 
  // corner of some cell in our uniform grid. 

  if ((myX < 0) || (myX >= nx)) return (NULL);
  if ((myY < 0) || (myY >= ny)) return (NULL);

  for (y = myY; y <= myY+1; y++) 
    {
      for (x = myX; x <= myX+1; x++) 
	{
	  P[pos].x = (float)x / (float)(nx-1);
	  P[pos].y = (float)y / (float)(ny-1);

	  // Need to jump to the right index in z[].

	  P[pos].z = z[(y*nx + x)];
	  ++pos;

	}
    }
  
  int *vp = verts;
  
  // 2 Triangles only, so just hard-code the points.
  *vp++ = 0; *vp++ = 1; *vp++ = 3;
  *vp++ = 0; *vp++ = 3; *vp++ = 2;

  TriangleMesh *mesh = NULL;

  // Create a triangle mesh . . . 
  if (!(tri1 || tri2))
    {
      mesh = new TriangleMesh(ObjectToWorld, ntris, 4, verts, P);
    }
  else
    {
      // . . . or help determine the interpolated normal 
      // of a vertex of the intersected triangle. 

      if (tri1)
	{
	  // For triangle #1. 
	  Vector t1Side1 = P[1] - P[0];                     // X
 	  Vector t1Side2 = P[3] - P[1];                     // Y
	  Vector t1Norm = Cross(t1Side1, t1Side2); 
	  t1Norm.Normalize();
	  norms = norms + t1Norm;
	  num++;

	  // RRG: Added for bump-mapping. Mimic phong-interpolation of the
	  // triangle normals by interpolating and storing the X- and
	  // Y-axes of the system as well. 
	  
	  Xaxis[axWhich] = Xaxis[axWhich] + t1Side1;
	  Yaxis[axWhich] = Yaxis[axWhich] + t1Side2;

	}

      if (tri2)
	{
	  // For triangle #2
	  Vector t2Side1 = P[3] - P[2];
	  Vector t2Side2 = P[2] - P[0];
	  Vector t2Norm = Cross(t2Side1, t2Side2); 
	  t2Norm.Normalize();
	  norms = norms + t2Norm;
	  num++;

	  // RRG: Bumps. See the above note in the first "IF" stmt. 
	  Xaxis[axWhich] = Xaxis[axWhich] + t2Side1;
	  Yaxis[axWhich] = Yaxis[axWhich] + t2Side2;
	}
    }

  // Free memory. 

  delete[] P;
  delete[] verts;
  
  return (mesh);
}


/* General intersection algorithm: 
 * ------------------------------
 * My (kinda wacky) grid traversal routine splits a grid cell into 
 * four quadrants by extending horizontal and vertical lines through
 * the current traversal point in the grid. (Current location is 
 * basically ray(time)). By comparing the slope of the given ray to 
 * the slope of the line formed by a corner of a quadrant and the
 * traversal point, we know which of the 2 grid cells adjacent to 
 * quadrant the ray will next intersect. Before we start
 * traversing the grid, we call GetSlopeType() to determine which 
 * of the four quadrants need to be examined in each pass. 
 *
 */

bool Heightfield::Intersect(const Ray &origRay, DifferentialGeometry *dg) const
{

  // Convert the ray to object space. 
  Ray ray = WorldToObject(origRay);

  // Taken from PG90 of the LRT book - ray/bounding-box intersections.
  float rayT = ray.maxt;
  if (mBBox->Inside(ray(ray.mint))) 
    {
      rayT = ray.mint;
    }
  else 
    {
      Ray rb = ray;
      if (!mBBox->IntersectP(rb))
	return (false);

      rayT = rb.maxt;
    }

  // Initialize intersection params. 
  float traverse = -1;
  float raySlope = (ray.D.x != 0) ? fabs(ray.D.y/ray.D.x) : (-1);
  
  // Determine the type of ray we're dealing with. 
  SlopeType rayType = GetSlopeType(ray);

  if (rayType == S_ERROR)
    {
      // Hopefully, we never get here. 
      // printf("PROBLEM!\n");
      return (false);
    }

  // Add a little offset to get a point *inside* a cell.
  // This might not actually make a difference. 
  rayT += error;

  // Get our initial point inside the grid along the ray.
  Point gridPt = ray(rayT);

  // Loop until the point is no longer inside the bounding box.
  while (mBBox->Inside(gridPt))
    {
      // Check whether we have an intersection. If so, the
      // "dg" struct will be updated after this call. 
      if (CheckTris(origRay, gridPt, dg))
	{
	  return (true);
	}
     
      // If not, traverse our uniform grid representation.
      traverse = Traverse(ray, gridPt, raySlope, rayType);
      if (traverse == -1) return (false);

      // Update the time index of our ray. 
      rayT += traverse + error;
      gridPt = ray(rayT);
    }
  
  // No triangles were successfully intersected. 
  return (false);
}


/* GetVertexNormal
 * ===============
 * Given the location of a triangle's vertex in the heightfield, 
 * this routine averages the basic normals of the six surrounding
 * triangles and returns a normal to be used in Phong interpolation.
 */

Vector Heightfield::GetVertexNormal(int x, int y) const
{
  Vector norms(0,0,0);
  int num = 0;

  // RRG: Initialize our global statics. Added for bump-mapping. 
  // This is a means of creating a phong-interpolated coordinate
  // system that more fully-defines the orientation of the surface
  // normal. 

  Xaxis[axWhich] = Vector(0, 0, 0);
  Yaxis[axWhich] = Vector(0, 0, 0);
 
  // Calls to the Triangulate() routine. Indicate which of the
  // surrounding triangles we want to use in the calculation. 

  Triangulate(x,   y,   true,  true,  norms, num);
  Triangulate(x-1, y,   false, true,  norms, num);
  Triangulate(x-1, y-1, true,  true,  norms, num);
  Triangulate(x,   y-1, true,  false, norms, num);

  // Average everything. 
  norms = norms/num;
  norms.Normalize();

  // RRG: Average the other two axes (generally perpendicular 
  // to the interpolated surface normal.) 

  Xaxis[axWhich] = Xaxis[axWhich]/num;
  Yaxis[axWhich] = Yaxis[axWhich]/num;

  return (norms);
}


/* PhongIt()
 * =========
 * The main Phong interpolation routine. Given the integer
 * coordinate of of a cell's lower lefthand corner, the 
 * position of the ray within that cell (posX, posY), and
 * an index indicating which of the cell's 2 triangles 
 * we're interested in, use some expensive Phong()
 * interpolation math to return a blended Normal.
 *
 * (posX, posY) are expected to be between 0 and 1.
 * (Percentage offsets along the X,Y-axes.)
 *
 *
 * Formula adapted from:
 * ====================
 *    http://www.cs.iusb.edu/~danav/teach/B481/light_global.html
 *
 */

Normal Heightfield::PhongIt(int cX, int cY, float posX, float posY, int index) const
{
  Vector n1(0,0,0);
  Vector n2(0,0,0);
  Vector n3(0,0,0);
  Normal result;

  if (index == 1)
    {

      // Get the blended normals of the three vertices.
      axWhich = 0; n1 = GetVertexNormal(cX, cY);
      axWhich = 1; n2 = GetVertexNormal(cX+1, cY+1);
      axWhich = 2; n3 = GetVertexNormal(cX, cY+1);

      // Interpolate.
      posY = 1-posY;
      result = Normal((1-posX) * ((1-posY)*n3 + posY*n1) + posX*n2);
      result.Normalize();

      // RRG: For bump mapping . . . instead of just interpolating the
      // surface normal of the returned point, I want to return an 
      // entire coordinate system, to make the bump mapping easier. 

      fX = (1-posX) * ((1-posY)*Xaxis[2] + posY*Xaxis[0]) + posX*Xaxis[1];
      fY = (1-posX) * ((1-posY)*Yaxis[2] + posY*Yaxis[0]) + posX*Yaxis[1];
    }
  else 
    {
      axWhich = 0; n1 = GetVertexNormal(cX, cY);
      axWhich = 1; n2 = GetVertexNormal(cX+1, cY);
      axWhich = 2; n3 = GetVertexNormal(cX+1, cY+1);

      posX = 1-posX;
      result = Normal((1-posY) * ((1-posX)*n2 + posX*n1) + posY*n3);
      result.Normalize();

      fX = (1-posY) * ((1-posX)*Xaxis[1] + posX*Xaxis[0]) + posY*Xaxis[2];
      fY = (1-posY) * ((1-posX)*Yaxis[1] + posX*Yaxis[0]) + posY*Yaxis[2];     
    } 

  return (result);
}



/* CellBox
 * =======
 * Depending on the type of ray we're dealing with, return a 
 * corner of the cell to be used in the slope comparison tests. 
 *
 */

Point Heightfield::CellBox(SlopeType sType, const Point &gridPt) const
{ 
  Point corner(0,0,0);

  // The indices of the 2 points that define our grid. 
  int minX = (int) (gridPt.x * (nx-1));
  int minY = (int) (gridPt.y * (ny-1));

  switch (sType)
    {
    case (S_PX):
    case (S_QUAD1):
      {
	corner.x = (minX + 1)/(float)(nx-1);
	corner.y = (minY + 1)/(float)(ny-1);
	break;
      }

    case (S_PY):
    case (S_QUAD2):
      {
	corner.x = (minX)/(float)(nx-1);
	corner.y = (minY + 1)/(float)(ny-1);
	break;
      }

    case (S_NX):
    case (S_QUAD3):
      {
	corner.x = (minX)/(float)(nx-1);
	corner.y = (minY)/(float)(ny-1);
	break;
      }

    case (S_NY):
    case (S_QUAD4):
      {
	corner.x = (minX+1)/(float)(nx-1);
	corner.y = (minY)/(float)(ny-1);
	break;
      }

    default:
      // Should never reach here. 
      // printf("PLEASE don't reach here\n");
      break;
    }
  
  return (corner);
}


/* Traverse
 * ========
 * Return a time increment that can be used to
 * generate a new Point further along the ray. 
 *
 */

float Heightfield::Traverse(const Ray &ray, const Point &gridPt, float raySlope, SlopeType sType) const
{
  // If the ray is pointing down the Z-axis, we'll never hit a different cell. 
  if (sType == S_Z) return (-1);

  // Generate the testBox. 
  Point corner = CellBox(sType, gridPt);

  float yDelta = fabs(corner.y - gridPt.y);
  float xDelta = fabs(corner.x - gridPt.x);

  // ALWAYS assuming a point is slightly INSIDE a box. 
  // Otherwise xDelta might = 0. This is why we add
  // "error" in Intersect(). 

  float slope = yDelta/xDelta;

  // Return a new time increment. 

  if ((sType == S_PX) || (sType == S_NX) ||(slope > raySlope)) return fabs(xDelta/ray.D.x); 
  else                                                         return fabs(yDelta/ray.D.y);

  //  printf("SHOULDN'T BE HERE!\n");
  return (-1);
}


// Return the slopeType. Called at the beginning of Intersect().

SlopeType Heightfield::GetSlopeType(const Ray &ray) const
{
  Vector dir = ray.D;

  if ((dir.x == 0) && (dir.y == 0)) return (S_Z);

  if ((dir.x == 0) && (dir.y > 0))  return (S_PY);
  if ((dir.x == 0) && (dir.y < 0))  return (S_NY);

  if ((dir.y == 0) && (dir.x > 0))  return (S_PX);  
  if ((dir.y == 0) && (dir.x < 0))  return (S_NX);

  if ((dir.x > 0) && (dir.y > 0))   return (S_QUAD1);
  if ((dir.x < 0) && (dir.y > 0))   return (S_QUAD2);
  if ((dir.x < 0) && (dir.y < 0))   return (S_QUAD3);
  if ((dir.x > 0) && (dir.y < 0))   return (S_QUAD4);

  return (S_ERROR);
}


/* Checks whether it's even worth doing Triangle
 * intersection tests. If the Z-coordinates of the 
 * surrounding grid points don't provide the proper
 * range, it's impossible to have an intersection.
 */

bool Heightfield::CheckZs(int minX, int minY, const Ray &ray) const
{
  float minZ, maxZ; 
  minZ = maxZ = z[minY*nx+minX];

  for (int lX = minX; lX <= minX+1; lX++)
    for (int lY = minY; lY <= minY+1; lY++)
      {
	float lZ = z[lY * nx + lX];

	if (lZ < minZ) minZ = lZ;
	if (lZ > maxZ) maxZ = lZ;
      }

  Point p1(minX/(float (nx-1)), minY/(float (ny-1)), minZ);
  Point p2((minX+1)/(float (nx-1)), (minY+1)/(float (ny-1)), maxZ);

  BBox tempBox(p1, p2);
  return (tempBox.IntersectP(ray));
}


/* CheckTris
 * =========
 * Checks whether the ray will intersect one 
 * of the triangles in the current grid cell.
 *
 */

bool Heightfield::CheckTris(const Ray &ray, const Point &gridPt, DifferentialGeometry *dg) const
{
  // The lower left-hand corner of the cell. 
  float convX = gridPt.x * (nx-1);
  float convY = gridPt.y * (ny-1);

  int minX = (int) (convX);
  int minY = (int) (convY);

  // Don't want to alter the original ray. 
  Ray tempRay = WorldToObject(ray);
  if (!CheckZs(minX, minY, tempRay)) return (false);

  return (HelperFunc(minX, minY, dg, ray));
}


// Pulled functionality out of CheckTris() so that I could use it in my bump-mapping efforts.

bool Heightfield::HelperFunc(int minX, int minY, DifferentialGeometry *dg, const Ray &ray) const
{
  int which = -1;

  // Generate a 2-triangle mesh.  
  Vector blah1; int blah2;
  TriangleMesh *mesh = Triangulate(minX, minY, false, false, blah1, blah2);

  bool result = false;

  // Generate the 2 triangles.
  Triangle *tri1 = new Triangle(ObjectToWorld, mesh, 0);
  Triangle *tri2 = new Triangle(ObjectToWorld, mesh, 1);

  // Test for intersections.

  which++;
  result = tri1->Intersect(ray, dg);

  if (!result) 
    {
      result = tri2->Intersect(ray, dg);
      which++;
    }

  // If we found an intersection, do a little additional processing. 
  // (Note: ray.maxt has been changed.)

  if (result)
    {
      Point tempP = dg->P;
      tempP = WorldToObject(tempP);

      float xScale = tempP.x*(nx-1) - minX;
      float yScale = tempP.y*(ny-1) - minY;

      // Do something sensible for U,V coordinates. 
      dg->u = tempP.x;
      dg->v = tempP.y;

      // Phong interpolation of triangle normals.
      dg->N = ObjectToWorld(PhongIt(minX, minY, xScale, yScale, which));
      dg->N.Normalize();

      // RRG: Pass back the other axes of the coordinate system. 
      dg->X = ObjectToWorld(fX);
      dg->Y = ObjectToWorld(fY);
      dg->X.Normalize();
      dg->Y.Normalize();

    }

  // Free up memory. 

  delete tri1;
  delete tri2;
  delete mesh;

  return (result);
}


// An accessor-type function used to index into heightfields
// that are being used as bump maps. Given a (u,v) coordinate 
// pair, returns the Normal of the heightfield triangle 
// located by (u,v). (u,v) are assumed to be between 0 
// and 1. 

Normal Heightfield::Bump(float u, float v)
{
  // Get the lower-left hand corner of the cell we're interested in.
  int cellX = (int) (u * (nx-1));
  int cellY = (int) (v * (ny-1));

  DifferentialGeometry dg;
  Vector t = Vector(0, 0, -1);
  Ray tempRay(Point(u, v, 10), t);

  bool result = HelperFunc(cellX, cellY, &dg, tempRay);
  if (!result) printf("Error.\n");
  return (dg.N);
}

