#include "materials.h"
#include "color.h"
#include "reflection.h"
#include "texture.h"
#include "abalone.h"

// RRG
#include "perlin.h"
#include "geometry.h"
#include "shapes.h"

DifferentialGeometry Material::bump(const DifferentialGeometry &dgg) const 
{
  return dgg;
}

Matte::~Matte() 
{
	delete Kd;
	
	// RRG: Delete the heighfield used for bump-mapping. 
	delete myField;
}


BSDF *Matte::getBSDF(const Surf *surf) const 
{
  bool doNoise = true;
  Spectrum r = Kd->evaluate(surf);

  // Use Perlin noise to add some interesting color to the sandy surface. 

  if (doNoise)
    {
      float rgb[3];
      r.ConvertToRGB(rgb);

      // A hard-coded secondary color. Sand colors are blended 
      // between the passed in matte color and this color. 

      float rgb2[3]; rgb2[0] = 0.88; rgb2[1] = .73; rgb2[2] = .40;

      // We want high variation, so multiply the texture coordinates
      // by a large # before indexing into the Perlin noise function. 

      float x = surf->dgGeom.u * 4000;
      float y = surf->dgGeom.v * 4000;
      
      double perlin = PerlinNoise2D(x, y, 2, 2, 10);

      // Turn the returned value into a # between 0 and 1. 
      perlin = (perlin + 1.0) / 2;

      // Compute our blended colors. 
      for (int x = 0; x < 3; x++)  
	rgb[x] = ((rgb[x] - rgb2[x]) * perlin) + rgb2[x];

      // Change the color used to create the LambertianReflection(). 
      r.Set(rgb[0], rgb[1], rgb[2]);

      return new BSDF(surf->dgShading, new LambertianReflection(r), 1., NULL);
    }
  else
    {
      return new BSDF(surf->dgShading, new LambertianReflection(r), 1., NULL);
    }

}


// Rahul: Overwrote the virtual Material::bump() function to implement
// bump-mapping. 

DifferentialGeometry Matte::bump(const DifferentialGeometry &dgg) const
{
  DifferentialGeometry newDGG = dgg;

  // The three vectors that define a coordinate 
  // system at the returned surface point.
  Normal myN = dgg.N; Vector myZ = Vector(myN.x, myN.y, myN.z);
  Vector myX = dgg.X;
  Vector myY = dgg.Y;

  // Normalize, just in case. 
  myZ.Normalize();
  myX.Normalize();
  myY.Normalize();
  
  // Index into our heightfield/bump map to get the appropriate surface
  // normal (for offset) at the given U,V coordinates. 

  Normal fieldN = myField->Bump(((dgg.u+0.1)/2.0), (dgg.v+0.1)/2.0);

  // Compute a new surface normal by multiplying the components of the
  // returned normal against the coordinate system stored in the 
  // differential geometry struct. Skew the weight towards the
  // Z-axis (axis of the original normal), so that the results
  // look reasonable.
  
  Vector result = (myX * fieldN.x) + (myY * fieldN.y) + (myZ * (fieldN.z*9));
  newDGG.N = Normal(result.x, result.y, result.z);
  newDGG.N.Normalize();

  return (newDGG);
}


Plastic::~Plastic() {
	delete Kd;
	delete Ks;
	delete roughness;
}
BSDF *Plastic::getBSDF(const Surf *surf) const {
	BxDF *diff = new LambertianReflection(Kd->evaluate(surf));
	BxDF *spec = new Microfacet(Ks->evaluate(surf),
		new Blinn(1. / roughness->evaluate(surf)));
	return new BSDF(surf->dgShading, diff, 1., spec, 1., NULL);
}


Glass::~Glass() {
	delete Kr;
	delete Kt;
	delete index;
}
BSDF *Glass::getBSDF(const Surf *surf) const {
	Spectrum R = Kr->evaluate(surf), T = Kt->evaluate(surf);
	Float ior = index->evaluate(surf);
	BxDF *brdf = NULL, *btdf = NULL;
	if (R.Intensity() > 0.)
		brdf = new SpecularReflection(R);
	if (T.Intensity() > 0.)
		btdf = new SpecularTransmission(T, 1., ior);
	if (brdf && btdf) return new BSDF(surf->dgShading, brdf, 1.,
						btdf, 1., NULL);
	else if (brdf)    return new BSDF(surf->dgShading, brdf, 1., NULL);
	else if (btdf)    return new BSDF(surf->dgShading, btdf, 1., NULL);
	else              return new BSDF(surf->dgShading, NULL);
}
ShinyMetal::~ShinyMetal() {
	delete Ks;
	delete roughness;
	delete Kr;
}
BSDF *ShinyMetal::getBSDF(const Surf *surf) const {
	Spectrum spec = Ks->evaluate(surf);
	Float rough = roughness->evaluate(surf);
	Spectrum R = Kr->evaluate(surf);

	MicrofacetDistribution *md = new Blinn(1. / rough);
	return new BSDF(surf->dgShading, new Microfacet(spec, md), 1.,
		new SpecularReflection(R), 1., NULL, NULL);
}
AbaloneShell::~AbaloneShell() {
	delete Ks;
	delete roughness;
	delete Kr;
}
BSDF *AbaloneShell::getBSDF(const Surf *surf) const {
	Spectrum spec = Ks->evaluate(surf);

	Float rough = roughness->evaluate(surf);
	Spectrum R = Kr->evaluate(surf);
	float spec2[3];
	R.ConvertToRGB(spec2);

	MicrofacetDistribution *md = new Blinn(1. / rough);
	return new BSDF(surf->dgShading, new AbaloneMicrofacet(spec, md, &(surf->dgShading)), 1., NULL);	
}

// Basically identical to the bump-mapping code defined for matte surfaces.

DifferentialGeometry AbaloneShell::bump(const DifferentialGeometry &dgg) const
{
  DifferentialGeometry newDGG = dgg;
  Normal myN = newDGG.N;
      
  Vector myZ = Vector(myN.x, myN.y, myN.z);
  myZ.Normalize();

  Normal fieldN = myField->Bump(((newDGG.u+0.5)/4.0), (newDGG.v+0.5)/4.0);
  
  // All of the abalones in our experiments were planar surfaces. For
  // these tests, we just hand-specify the coordinate system of whatever
  // abalone-facet we're dealing with. 

  Vector result = ((Vector(1,0,0) * fieldN.x) + (Vector(0,0,1)* fieldN.y) + (Vector(0,-1,0) * (fieldN.z*3.0)));
  newDGG.N = Normal(result.x, result.y, result.z);
  newDGG.N.Normalize();

  return (newDGG);
}
