// ============================================================================
// MathLib.cpp
// Copyright 2000, Discreet   -  www.discreet.com
// Developed by Simon Feltman  -  simon.feltman@autodesk.com
// ----------------------------------------------------------------------------
#include "stdafx.h"
#include <math.h>
#include <memory.h>
#include "MathLib.h"

#ifdef WIN32
#pragma warning (disable:4244) // warning: conversion from 'double' to 'float'
#endif

#define DELTA 1e-6     // error tolerance

// ============================================================================
void VectorMultiply(vector_t dst, matrix_t tm, vector_t vec)
{
	dst[X] = (tm[0]*vec[X]) + (tm[4]*vec[Y]) + (tm[8]*vec[Z]) + tm[12];
	dst[Y] = (tm[1]*vec[X]) + (tm[5]*vec[Y]) + (tm[9]*vec[Z]) + tm[13];
	dst[Z] = (tm[2]*vec[X]) + (tm[6]*vec[Y]) + (tm[10]*vec[Z]) + tm[14];
}

// ============================================================================
float VectorLength(vector_t start, vector_t end)
{
	vector_t vec;
	VectorSub(vec, end, start);
	float len = VectorLengthSquared(vec);
	return (float)sqrt(len);
}

// ============================================================================
void VectorNormalize(vector_t vec)
{
	float len = sqrt(VectorLengthSquared(vec));
	vec[X] = vec[X]/len;
	vec[Y] = vec[Y]/len;
	vec[Z] = vec[Z]/len;
}

// ============================================================================
void VectorRotate(vector_t pos, vector_t cen, vector_t ang)
{
	vector_t p;
	VectorSub(p, pos, cen);

	p[Z] += sin(ang[Y]);
	p[X] += cos(ang[Y]);
}

// ============================================================================
void QuatMult(quat_t q1, quat_t q2, quat_t res)
{
	float A, B, C, D, E, F, G, H;

	A = (q1[W] + q1[X])*(q2[W] + q2[X]);
	B = (q1[Z] - q1[Y])*(q2[Y] - q2[Z]);
	C = (q1[X] - q1[W])*(q2[Y] + q2[Z]);
	D = (q1[Y] + q1[Z])*(q2[X] - q2[W]);
	E = (q1[X] + q1[Z])*(q2[X] + q2[Y]);
	F = (q1[X] - q1[Z])*(q2[X] - q2[Y]);
	G = (q1[W] + q1[Y])*(q2[W] - q2[Z]);
	H = (q1[W] - q1[Y])*(q2[W] + q2[Z]);

	res[W] =  B + (-E - F + G + H) * 0.5f;
	res[X] =  A - (E + F + G + H)  * 0.5f; 
	res[Y] = -C + (E - F + G - H)  * 0.5f;
	res[Z] = - D + (E - F - G + H) * 0.5f;
}

// ============================================================================
void QuatSlerp(quat_t from, quat_t to, float t, quat_t res)
{
	quat_t to1;
	double omega, cosom, sinom, scale0, scale1;

	// calc cosine
	cosom = from[X] * to[X] + from[Y] * to[Y] + from[Z] * to[Z] + from[W] * to[W];

	// adjust signs (if necessary)
	if(cosom < 0.0)
	{
		cosom = -cosom;
		to1[0] = - to[X];
		to1[1] = - to[Y];
		to1[2] = - to[Z];
		to1[3] = - to[W];
	}
	else
	{
		to1[0] = to[X];
		to1[1] = to[Y];
		to1[2] = to[Z];
		to1[3] = to[W];
	}

	// calculate coefficients
	if((1.0 - cosom) > DELTA)
	{
		// standard case (slerp)
		omega = acos(cosom);
		sinom = sin(omega);
		scale0 = sin((1.0 - t) * omega) / sinom;
		scale1 = sin(t * omega) / sinom;
	}
	else
	{        
		// "from" and "to" quaternions are very close 
		//  ... so we can do a linear interpolation
		scale0 = 1.0 - t;
		scale1 = t;
	}

	// calculate final values
	res[X] = scale0 * from[X] + scale1 * to1[0];
	res[Y] = scale0 * from[Y] + scale1 * to1[1];
	res[Z] = scale0 * from[Z] + scale1 * to1[2];
	res[W] = scale0 * from[W] + scale1 * to1[3];
}

//added by Yar
void VectorPerp(vector_t dst, vector_t src)
{
	dst[X] = 1;
	dst[Y] = 1;
	dst[Z] = -(src[X]+src[Y])/src[Z];
	VectorNormalize(dst);
}

// ============================================================================
void QuatToMatrix(quat_t quat, matrix_t m)
{
	float wx, wy, wz, xx, yy, yz, xy, xz, zz, x2, y2, z2;

	// calculate coefficients
	x2 = quat[X] + quat[X]; y2 = quat[Y] + quat[Y]; 
	z2 = quat[Z] + quat[Z];
	xx = quat[X] * x2;   xy = quat[X] * y2;   xz = quat[X] * z2;
	yy = quat[Y] * y2;   yz = quat[Y] * z2;   zz = quat[Z] * z2;
	wx = quat[W] * x2;   wy = quat[W] * y2;   wz = quat[W] * z2;

	m[M00] = 1.0 - (yy + zz); 	m[M01] = xy - wz;
	m[M02] = xz + wy;		m[M03] = 0.0;

	m[M10] = xy + wz;		m[M11] = 1.0 - (xx + zz);
	m[M12] = yz - wx;		m[M13] = 0.0;

	m[M20] = xz - wy;		m[M21] = yz + wx;
	m[M22] = 1.0 - (xx + yy);		m[M23] = 0.0;

	m[M30] = 0;			m[M31] = 0;
	m[M32] = 0;			m[M33] = 1;
}

// ============================================================================
void QuatToAngleAxis(quat_t quat, angleaxis_t angleaxis)
{
#if 1
	float scale, tw;
	tw = (float)acos(quat[W]) * 2.f;
	scale = (float)sin(tw * 0.5);
	angleaxis[X] = quat[X] / scale;
	angleaxis[Y] = quat[Y] / scale;
	angleaxis[Z] = quat[Z] / scale;
	angleaxis[W] = (tw * 180.f) / (float)PI;

#else
	double omega, s, x, y, z, w, l, c;
	x = (double)quat[X];
	y = (double)quat[Y];
	z = (double)quat[Z];
	w = (double)quat[W];
	l =  sqrt(x*x + y*y + z*z + w*w);
	if(l == 0.0)
	{
		w = 1.0;	
		y = z = x = 0.0;
	}
	else
	{
		c = 1.0 / l;
		x *= c;
		y *= c;
		z *= c;
		w *= c;
	}

	omega = acos(w);
	angleaxis[W] = RadToDeg((float)(2.0*omega));
	s = sin(omega);	
	if(fabs(s) > 0.000001f)
	{
		angleaxis[X] = (float)(x / s);
		angleaxis[Y] = (float)(y / s);
		angleaxis[Z] = (float)(z / s);
	}

#endif
}

//---------------------------------------------------------
//added by ywoo
void AngleAxisToMatrix(angleaxis_t angleaxis, matrix_t rot_m)
{
	quat_t quat;
	float theta = DegToRad(angleaxis[W]);
	quat[W] = cos(theta/2);
	quat[X] = angleaxis[X]*sin(theta/2);
	quat[Y] = angleaxis[Y]*sin(theta/2);
	quat[Z] = angleaxis[Z]*sin(theta/2);

	QuatToMatrix(quat, rot_m);
}

// ============================================================================
void EulerToQuat(float roll, float pitch, float yaw, quat_t quat)
{
	float cr, cp, cy, sr, sp, sy, cpcy, spsy;

	// calculate trig identities
	cr = cos(roll * 0.5f);
	cp = cos(pitch * 0.5f);
	cy = cos(yaw * 0.5f);

	sr = sin(roll * 0.5f);
	sp = sin(pitch * 0.5f);
	sy = sin(yaw * 0.5f);

	cpcy = cp * cy;
	spsy = sp * sy;

	quat[W] = cr * cpcy + sr * spsy;
	quat[X] = sr * cpcy - cr * spsy;
	quat[Y] = cr * sp * cy + sr * cp * sy;
	quat[Z] = cr * cp * sy - sr * sp * cy;
}

// ============================================================================
void MatrixCopy(matrix_t dst, const matrix_t src)
{
	memcpy(dst, src, sizeof(matrix_t));
}

// ============================================================================
void MatrixMultiply(matrix_t dst, const matrix_t tm1, const matrix_t tm2)
{
   dst[0] = tm1[0] * tm2[0] + tm1[4] * tm2[1] + tm1[8] * tm2[2] + tm1[12] * tm2[3];
   dst[1] = tm1[1] * tm2[0] + tm1[5] * tm2[1] + tm1[9] * tm2[2] + tm1[13] * tm2[3];
   dst[2] = tm1[2] * tm2[0] + tm1[6] * tm2[1] + tm1[10] * tm2[2] + tm1[14] * tm2[3];
   dst[3] = tm1[3] * tm2[0] + tm1[7] * tm2[1] + tm1[11] * tm2[2] + tm1[15] * tm2[3];
   dst[4] = tm1[0] * tm2[4] + tm1[4] * tm2[5] + tm1[8] * tm2[6] + tm1[12] * tm2[7];
   dst[5] = tm1[1] * tm2[4] + tm1[5] * tm2[5] + tm1[9] * tm2[6] + tm1[13] * tm2[7];
   dst[6] = tm1[2] * tm2[4] + tm1[6] * tm2[5] + tm1[10] * tm2[6] + tm1[14] * tm2[7];
   dst[7] = tm1[3] * tm2[4] + tm1[7] * tm2[5] + tm1[11] * tm2[6] + tm1[15] * tm2[7];
   dst[8] = tm1[0] * tm2[8] + tm1[4] * tm2[9] + tm1[8] * tm2[10] + tm1[12] * tm2[11];
   dst[9] = tm1[1] * tm2[8] + tm1[5] * tm2[9] + tm1[9] * tm2[10] + tm1[13] * tm2[11];
   dst[10]= tm1[2] * tm2[8] + tm1[6] * tm2[9] + tm1[10] * tm2[10] + tm1[14] * tm2[11];
   dst[11]= tm1[3] * tm2[8] + tm1[7] * tm2[9] + tm1[11] * tm2[10] + tm1[15] * tm2[11];
   dst[12]= tm1[0] * tm2[12] + tm1[4] * tm2[13] + tm1[8] * tm2[14] + tm1[12] * tm2[15];
   dst[13]= tm1[1] * tm2[12] + tm1[5] * tm2[13] + tm1[9] * tm2[14] + tm1[13] * tm2[15];
   dst[14]= tm1[2] * tm2[12] + tm1[6] * tm2[13] + tm1[10] * tm2[14] + tm1[14] * tm2[15];
   dst[15]= tm1[3] * tm2[12] + tm1[7] * tm2[13] + tm1[11] * tm2[14] + tm1[15] * tm2[15];
}

// ============================================================================
void MatrixMultiply(matrix_t dst, const matrix_t src)
{
	matrix_t dst2;
	MatrixMultiply(dst2, dst, src);
	memcpy(dst, dst2, sizeof(matrix_t));
}

// ============================================================================
void MatrixGetTrans(const matrix_t tm, point_t trans)
{
	trans[X] = tm[MX];
	trans[Y] = tm[MY];
	trans[Z] = tm[MZ];
}

#ifdef WIN32
#pragma warning(default : 4244)
#endif

