// ============================================================================
// MathLib.h
// Copyright 2000, Discreet   -  www.discreet.com
// Developed by Simon Feltman  -  simon.feltman@autodesk.com
// ----------------------------------------------------------------------------
#ifndef __MATHLIB_H__
#define __MATHLIB_H__

#define X 0
#define Y 1
#define Z 2
#define W 3

#define M00 0
#define M01 1
#define M02 2
#define M03 3
#define M10 4
#define M11 5
#define M12 6
#define M13 7
#define M20 8
#define M21 9
#define M22 10
#define M23 11
#define M30 12
#define M31 13
#define M32 14
#define M33 15

#define MX 12
#define MY 13
#define MZ 14
#define MW 15

#ifndef PI
#define PI 3.14159265358979323846264338327950288419716939937510f
#endif

typedef float vector_t[3];
typedef float vertex_t[3];
typedef float point_t[3];
typedef float euler_t[3];
typedef float angles_t[3];
typedef float plane_t[4];
typedef float quat_t[4];
typedef float quaternion_t[4];
typedef float angleaxis_t[4];
typedef float matrix_t[16];
typedef int   face_t[3];

typedef unsigned long ulong;
typedef unsigned int  uint;
typedef unsigned char ubyte;

static const float origin[3] = {0.0, 0.0, 0.0};
static const float x_axis[3] = {1.0, 0.0, 0.0};
static const float y_axis[3] = {0.0, 1.0, 0.0};
static const float z_axis[3] = {0.0, 0.0, 1.0};

static const matrix_t identityTM =
{
	1.f, 0.f, 0.f, 0.f,
	0.f, 1.f, 0.f, 0.f,
	0.f, 0.f, 1.f, 0.f,
	0.f, 0.f, 0.f, 1.f
};

#define RadToDeg(rad) (180.f/PI * rad)
#define DegToRad(deg) (PI/180.f * deg)

#define VectorCopy(dst, src)   (dst[0]=src[0], dst[1]=src[1], dst[2]=src[2])
#define VectorAdd(dst, v1, v2) (dst[X]=v1[X]+v2[X], dst[Y]=v1[Y]+v2[Y], dst[Z]=v1[Z]+v2[Z])
#define VectorSub(dst, v1, v2) (dst[X]=v1[X]-v2[X], dst[Y]=v1[Y]-v2[Y], dst[Z]=v1[Z]-v2[Z])
#define VectorScale(dst, v, f) (dst[X]=v[X]*f, dst[Y]=v[Y]*f, dst[Z]=v[Z]*f)
#define VectorLengthSquared(vec) (vec[X]*vec[X] + vec[Y]*vec[Y] + vec[Z]*vec[Z])

#define QuatCopy(dst, src) (dst[0]=src[0], dst[1]=src[1], dst[2]=src[2], dst[3]=src[3])

#define MatrixIdentity(tm) MatrixCopy(tm, identityTM)

void VectorMultiply(vector_t dst, matrix_t tm, vector_t vec);
float VectorLength(vector_t start, vector_t end);
void VectorNormalize(vector_t vec);
void VectorRotate(vector_t pos, vector_t cen, vector_t ang);
void VectorPerp(vector_t dst, vector_t src);

void QuatMult(quat_t q1, quat_t q2, quat_t res);
void QuatSlerp(quat_t from, quat_t to, float t, quat_t res);
void QuatToMatrix(quat_t quat, matrix_t m);
void QuatToAngleAxis(quat_t quat, angleaxis_t angleaxis);
void AngleAxisToMatrix(angleaxis_t angleaxis, matrix_t rot_m);
void EulerToQuat(float roll, float pitch, float yaw, quat_t quat);

void MatrixCopy(matrix_t dst, const matrix_t src);
void MatrixMultiply(matrix_t dst, const matrix_t tm1, const matrix_t tm2);
void MatrixMultiply(matrix_t dst, const matrix_t src);
void MatrixGetTrans(const matrix_t tm, point_t trans);

#endif //__MATHLIB_H__

