#include "scene.h"
#include "camera.h"
#include "primitives.h"
#include "image.h"
#include "transport.h"
#include "accel.h"
#include "sampling.h"
#include "ri.h"

#define OPTIMAL_BRIGHTNESS 0.79

Scene::~Scene() {
	delete camera;
	delete sampler;
	delete image;
	delete integrator;
}
Spectrum Scene::L(const Ray &ray) const {
	return integrator->L(this, ray, NULL);
}
bool Scene::IntersectP(const Ray &ray) const {
	return prims->IntersectP(ray);
}
Scene::Scene(Camera *cam, Integrator *in, Image *img, Sampler *s,
		const vector<Primitive *> &pr,
		const vector<Light *> &lts) {
	lights = lts;
	prims = new GridAccelerator(pr);
	camera = cam;
	image = img;
	sampler = s;
	integrator = in;
	Transform gridtrans;
	Vector trans(-8,-8,10);
	gridtrans = Translate(trans);
	densityGrid = new DensityGrid(gridtrans, 16,16,5, 10, 10, 10);
	densityGrid->Populate();
}

#define MAGIC_CONSTANT 0.1
#define WEIGHT_ADJUST 0.3

void Scene::SetupAutomaticCamera()
{
  if (!camera->IsAutomatic()) return;

  Float sample[5];
  Float zTotal = 0;
  int numZValues = 0;
  Float totalBrightness = 0;
  int numBrightnessValues = 0;
  Float brightness;
  Float epsilon = 0.001;
  Float maxZ = 0, minZ = 1;
  int xres = sampler->XResolution;
  int yres = sampler->YResolution;

  JitterSampler smallSampler(sampler->XResolution, sampler->YResolution,
			     1, 1, false, sampler->filter);

  while (smallSampler.GetNextImageSample(sample))
    {
      Ray ray;
      if (!camera->GenerateRay(sample, ray))
	continue;
      Float alpha;
      Spectrum L = integrator->L(this, ray, &alpha);
      Float screenz = camera->WorldToScreen(ray(ray.maxt)).z;
      if (screenz>=1-epsilon) 
	{
	  L=0;
	  alpha = 0;
	}
      else 
	{
	  if (sample[0] > xres/2. - xres/10. &&
	       sample[0] < xres/2. + xres/10. &&
	       sample[1] > yres/2. - yres/10. &&
	       sample[1] < yres/2. + yres/10.)
	    {
	      numZValues++;
	      zTotal += camera->WorldToCamera(ray(ray.maxt)).z;
	      //}
	      Float brightness[3];
	      Float maxBrightness;

	      numBrightnessValues ++;
	      L.ConvertToRGB(brightness);
	      maxBrightness = max(brightness[0], 
				  max(brightness[1],
				      brightness[2]));
	  
	      totalBrightness+= maxBrightness;
	    }
	  if (screenz < minZ) minZ = screenz;
	  if (screenz > maxZ) maxZ = screenz;
	}
    }

  Float focalDist = zTotal/numZValues;

  printf("Average z value around the middle = %g\n", focalDist);
  printf("Average intensity around the middle = %g\n", 
	 brightness = totalBrightness/numBrightnessValues);
  camera->SetFocalDistance(focalDist);

  if (camera->GetMode() == LRT_APERTURE_PRIORITY)
    {
      Float fstop = camera->GetFStop();
      Float optimalShutter = (fstop*fstop*OPTIMAL_BRIGHTNESS)/brightness;
      camera->SetShutterEnd(optimalShutter);
    }
  else if (camera->GetMode() == LRT_SHUTTER_PRIORITY)
    {
      Float shutterEnd = camera->GetShutterEnd();
      Float optimalFstop = sqrt((brightness*shutterEnd)/
				OPTIMAL_BRIGHTNESS);

      camera->SetFStop(optimalFstop);
      
    }
  else
    {
      //programmed! We need to strike some sort of balance.
      //we're going to use two factors. One is the max distance between
      //the farthest thing and the focal distance. The other is the 
      //total amount that the camera is moving.
      //My strategy then goes like this: Assign a weighting to both
      //factors. The higher the weighting of the max distance, the 
      //more we'll try to increase the f-stop. The more the camera
      //moves, the less time we want the shutter to be open for.
      //It boils down to a differential equation... the optimal 
      //time the shutter should be open for is proportional to the 
      //cube root of the square of the ratio of the weights.
      //exactly how proportional? who knows. experimentation.

      Float weight1, weight2, totalWeight;
      weight1 = camera->GetDistanceMoved();
      weight2 = WEIGHT_ADJUST*max(maxZ-focalDist, focalDist-minZ);

      printf("Weights: %g %g\n", weight1, weight2);

      Float shutterEnd = MAGIC_CONSTANT * 
	cbrt((weight2*weight2)/(weight1*weight1));

      if (shutterEnd > 1) shutterEnd = 1;
      if (shutterEnd < 0) shutterEnd = 0;
      Float fstop = sqrt((brightness*shutterEnd)/OPTIMAL_BRIGHTNESS);

      camera->SetFStop(fstop);
      camera->SetShutterEnd(shutterEnd);
      
    }

  printf("Shutter = %g\n", camera->GetShutterEnd());
  printf("Fstop = %g\n", camera->GetFStop());

  camera->SetGain();
}

Spectrum Scene::CalculateVolumeSpectrum (Ray &ray, Float &alpha)
{
  return (densityGrid->L(this, ray, alpha));
}
double maxAlpha = 0;

void Scene::Render() {
	cerr << "Rendering: ";
	Float sample[5];

	//SetupAutomaticCamera();

	while (sampler->GetNextImageSample(sample)) {
		Ray ray;
		if (!camera->GenerateRay(sample, ray))
			continue;
		static int eyeRaysTraced = 0;
		if (eyeRaysTraced == 0)
			StatsRegisterCounter(STATS_BASIC, "Camera", "Eye Rays Traced",
				&eyeRaysTraced);
		++eyeRaysTraced;
		if (eyeRaysTraced % 10000 == 0) cerr << '+';
		Float alpha, volAlpha;
		Spectrum vol = CalculateVolumeSpectrum(ray, volAlpha);

		Spectrum Ltemp = integrator->L(this, ray, &alpha);
		if (volAlpha>1) volAlpha = 1;
		Spectrum L = (volAlpha*vol)+(1-volAlpha)*Ltemp;
		//Spectrum L = Ltemp;
		Float rgbL[3];
		//Ltemp.ConvertToRGB(rgbL);
		//printf("L = %g %g %g\n", rgbL[0], rgbL[1], rgbL[2]);
		Float screenz = camera->WorldToScreen(ray(ray.maxt)).z;
		if (screenz > 1.) {
			L = 0.;
			alpha = 0.;
		}
		Point Praster(sample[0], sample[1], screenz);
		image->AddSample(Praster, L, alpha);
	}
	printf("max alpha = %g\n", maxAlpha);
	image->Write();
	cerr << endl;
}
bool Scene::Intersect(const Ray &ray, Surf *surf) const {
	return prims->Intersect(ray, surf);
}
