#ifndef GEOMETRY_H
#define GEOMETRY_H
#include "lrt.h"
#include <float.h>
class Vector {
public:
	Vector(Float xx=0.0f, Float yy=0.0f, Float zz=0.0f)
	  : x(xx), y(yy), z(zz) {}
	explicit Vector(const Normal &n);
	Vector operator+ ( const Vector &v ) const {
		return Vector( x + v.x, y + v.y, z + v.z );
	}
	
	Vector& operator+=( const Vector &v ) {
		x += v.x; y += v.y; z += v.z;
		return *this;
	}
	Vector operator- ( const Vector &v ) const {
		return Vector( x - v.x, y - v.y, z - v.z );
	}
	
	Vector& operator-=( const Vector &v ) {
		x -= v.x; y -= v.y; z -= v.z;
		return *this;
	}
	Vector operator* ( Float f ) const {
		return Vector( f*x, f*y, f*z );
	}
	
	Vector &operator*=( Float f ) {
		x *= f; y *= f; z *= f;
		return *this;
	}
	Vector operator/ ( Float f ) const {
		Float inv = 1.0/f;
		return Vector(x * inv, y * inv, z * inv);
	}
	
	Vector &operator/=( Float f ) {
		Float inv = 1.0/f;
		x *= inv; y *= inv; z *= inv;
		return *this;
	}
	Vector operator-() const {
		return Vector( -x, -y, -z );
	}
	Float LengthSquared() const { return x*x + y*y + z*z; }
	Float Length() const { return sqrt( LengthSquared() ); }
	Vector Hat() const { return (*this)/Length(); }
	void Normalize()   { (*this) /= Length(); }
	Float x, y, z;
};
class Point {
public:
	Point(Float xx=0.0, Float yy=0.0, Float zz=0.0 )
	  : x(xx), y(yy), z(zz) {}
	Point operator+( const Vector &v ) const {
		return Point( x + v.x, y + v.y, z + v.z );
	}
	
	Point &operator+=( const Vector &v ) {
		x += v.x; y += v.y; z += v.z;
		return *this;
	}
	
	Vector operator-( const Point &p ) const {
		return Vector( x - p.x, y - p.y, z - p.z );
	}
	
	Point operator-(const Vector &v) const {
		return Point(x - v.x, y - v.y, z - v.z);
	}
	
	Point &operator-=( const Vector &v ) {
		x -= v.x; y -= v.y; z -= v.z;
		return *this;
	}

    float& operator[](const int index) {
        if(index == 0)
            return x;
        else if(index == 1)
            return y;
        else
            return z;
    }
	Float x,y,z;
};
class Normal {
public:
	Normal( Float xx=0.0, Float yy=0.0, Float zz=0.0 )
	  : x(xx), y(yy), z(zz) {}
	explicit Normal( const Vector &v )
	  : x(v.x), y(v.y), z(v.z) {}
	Normal operator-() const {
		return Normal( -x, -y, -z );
	}
	Normal operator+ ( const Normal &v ) const {
		return Normal( x + v.x, y + v.y, z + v.z );
	}
	
	Normal& operator+=( const Normal &v ) {
		x += v.x; y += v.y; z += v.z;
		return *this;
	}
	Normal operator- ( const Normal &v ) const {
		return Normal( x - v.x, y - v.y, z - v.z );
	}
	
	Normal& operator-=( const Normal &v ) {
		x -= v.x; y -= v.y; z -= v.z;
		return *this;
	}
	Normal operator* ( Float f ) const {
		return Normal( f*x, f*y, f*z );
	}
	
	Normal &operator*=( Float f ) {
		x *= f; y *= f; z *= f;
		return *this;
	}
	Normal operator/ ( Float f ) const {
		Float inv = 1.0/f;
		return Normal(x * inv, y * inv, z * inv);
	}
	
	Normal &operator/=( Float f ) {
		Float inv = 1.0/f;
		x *= inv; y *= inv; z *= inv;
		return *this;
	}
	Float LengthSquared() const { return x*x + y*y + z*z; }
	Float Length() const        { return sqrt( LengthSquared() ); }
	Normal Hat() const          { return (*this) / Length(); }
	void Normalize()            { (*this) /= Length(); }
	Float x,y,z;
};
class Ray {
public:
	Ray(): mint(1e-3), maxt(FLT_MAX) {}
	Ray(const Point &origin, const Vector &direction,
		Float start = 1e-3, Float end = FLT_MAX )
		: O( origin ), D( direction ), mint(start), maxt(end) {
	}
	Point operator()( Float t ) const { return O + D * t; }
	Point O;
	Vector D;
	mutable Float mint, maxt;
};
class BBox {
public:
	BBox( const Point &p ) : pMin(p), pMax(p) { }
	BBox( const Point &p1, const Point &p2 ) {
		pMin = Point(min(p1.x, p2.x),
					 min(p1.y, p2.y),
					 min(p1.z, p2.z));
		pMax = Point(max(p1.x, p2.x),
					 max(p1.y, p2.y),
					 max(p1.z, p2.z));
	}
	friend inline ostream &operator<<(ostream &os, const BBox &b);
	friend BBox Union(const BBox &b, const Point &p);
	friend BBox Union(const BBox &b, const BBox &b2);
	friend BBox Intersection(const BBox &b1, const BBox &b2);
	bool Inside(const Point &pt) const {
		if (pt.x < pMin.x) return false;
		if (pt.x > pMax.x) return false;
		if (pt.y < pMin.y) return false;
		if (pt.y > pMax.y) return false;
		if (pt.z < pMin.z) return false;
		if (pt.z > pMax.z) return false;
		return true;
	}
	bool IntersectP( const Ray &ray ) const;
	friend class HBVAccelerator;
	Point pMin, pMax;
private:
	BBox() {  }
	friend class Transform;
};
inline ostream &operator<<(ostream &os, const Vector &v) {
	os << v.x << ", " << v.y << ", " << v.z;
	return os;
}
inline Vector operator*( Float f, const Vector &v ) { return v*f; }
inline Float Dot(const Vector &v1, const Vector &v2) {
	return v1.x * v2.x + v1.y * v2.y + v1.z * v2.z;
}
inline Vector Cross(const Vector &v1, const Vector &v2) {
	return Vector((v1.y * v2.z) - (v1.z * v2.y),
			(v1.z * v2.x) - (v1.x * v2.z),
			(v1.x * v2.y) - (v1.y * v2.x));
}
inline Float Distance(const Point &p1, const Point &p2) {
	return (p1 - p2).Length();
}
inline Float DistanceSquared(const Point &p1, const Point &p2) {
	return (p1 - p2).LengthSquared();
}
inline ostream &operator<<(ostream &os, const Point &v) {
	os << v.x << ", " << v.y << ", " << v.z;
	return os;
}
inline Normal operator*(Float f, const Normal &n) {
	return Normal( f*n.x, f*n.y, f*n.z );
}
inline Vector::Vector(const Normal &n)
  : x(n.x), y(n.y), z(n.z) { }
inline Float Dot(const Normal &n1, const Vector &v2) {
	return n1.x * v2.x + n1.y * v2.y + n1.z * v2.z;
}
inline Float Dot(const Vector &v1, const Normal &n2) {
	return v1.x * n2.x + v1.y * n2.y + v1.z * n2.z;
}
inline Float Dot(const Normal &n1, const Normal &n2) {
	return n1.x * n2.x + n1.y * n2.y + n1.z * n2.z;
}
inline ostream &operator<<(ostream &os, const BBox &b) {
	os << b.pMin << " -> " << b.pMax;
	return os;
}
Point Lerp(Float t, const Point &p1, const Point &p2);
extern Transform Translate(const Vector &delta);
extern Transform RotateX(Float angle);
extern Transform RotateY(Float angle);
extern Transform RotateZ(Float angle);
extern Transform Rotate(Float angle, const Vector &axis);
extern Transform Scale(Float x, Float y, Float z);
#endif // GEOMETRY_H
