
/*
 * pbrt source code Copyright(c) 1998-2005 Matt Pharr and Greg Humphreys
 *
 * All Rights Reserved.
 * For educational use only; commercial use expressly forbidden.
 * NO WARRANTY, express or implied, for this software.
 * (See file License.txt for complete license)
 */

// diamond.cpp*
#include "pbrt.h"
#include "material.h"
#include "wavelengthfilter.h"

class COREDLL DispersiveTransmission : public BxDF {
public:
	// DispersiveTransmission Public Methods
	DispersiveTransmission(const Spectrum &t, float ei, float et, float dp)
    : BxDF(BxDFType(BSDF_TRANSMISSION | BSDF_SPECULAR)),
		  fresnel(ei, et) {
              T = t;
              etai = ei;
              etat = et;
              disp = dp;
    }
	bool IsDispersive() { return true; }
	Spectrum Sample_f(const Vector &wo, Vector *wi, float u1, float u2, float *pdf) const;
    Spectrum Sample_f(const Vector &wo, Vector *wi, float u1, float u2, float u3, WavelengthFilter *wf, float *pdf) const;
    Spectrum f(const Vector &, const Vector &) const {
		return Spectrum(0.);
	}
	float Pdf(const Vector &wo, const Vector &wi) const {
		return 0.;
	}
private:
    // DispersiveTransmission Private Data
    mutable Spectrum T;
	float etai, etat;
    float disp;
	FresnelDielectric fresnel;
};
Spectrum DispersiveTransmission::Sample_f(const Vector &wo,
                        Vector *wi, float u1, float u2, float *pdf) const {
    WavelengthFilter wf = WavelengthFilter();
    return Sample_f(wo, wi, u1, u2, RandomFloat(), &wf, pdf);
}
Spectrum DispersiveTransmission::Sample_f(const Vector &wo, Vector *wi, float u1, float u2, float u3, WavelengthFilter *waveFilter, float *pdf) const {
    static const float INV_WAVELENGTH_SAMPLES = 1.f / WAVELENGTH_SAMPLES;
    // Figure out which $\eta$ is incident and which is transmitted
	bool entering = CosTheta(wo) > 0.;
    /*if (entering) printf("entering\n");
    else printf("exiting\n");
    exit(1);*/
	float ei = etai, et = etat;
    Spectrum spec = T;
    float wavelength;
    if (waveFilter->isMonochromatic()) {
        // if already monochromatic, use its wavelength
        wavelength = waveFilter->wavelength();
        *pdf = 1.f;
    }
    else {
        // randomly sample wavelength, and weight pdf accordingly
        int i;
        for (i = 1; i <= WAVELENGTH_SAMPLES; i++) {
            if (u3 < i*INV_WAVELENGTH_SAMPLES) {
                waveFilter->setIndex(i);
                break;
            }
        }
        wavelength = waveFilter->wavelength();
        *pdf = INV_WAVELENGTH_SAMPLES;
    }
    // filter transmission spectrum
    waveFilter->filterSpectrum(&spec);
    // approximate wavelength-dependent dispersion with linear interpolation
    float d = disp;
    et += d * 0.5f;
    et -= d * ((wavelength - Spectrum::CIEstart) / Spectrum::nCIE);
    
	if (!entering)
		swap(ei, et);
        
	// Compute transmitted ray direction
	float sini2 = SinTheta2(wo);
	float eta = ei / et;
	float sint2 = eta * eta * sini2;
	// Handle total internal reflection for transmission
	if (sint2 > 1.)  return 0.;
	float cost = sqrtf(max(0.f, 1.f - sint2));
	if (entering) cost = -cost;
	float sintOverSini = eta;
	*wi = Vector(sintOverSini * -wo.x,
	             sintOverSini * -wo.y,
				 cost);
	Spectrum F = fresnel.Evaluate(CosTheta(wo));
	return (ei*ei)/(et*et) * (Spectrum(1.)-F) * spec /
		fabsf(CosTheta(*wi));
}

// Diamond Class Declarations
class Diamond : public Material {
public:
	// Diamond Public Methods
	Diamond(Reference<Texture<Spectrum> > r, Reference<Texture<Spectrum> > t,
			Reference<Texture<float> > i, Reference<Texture<float> > dp, Reference<Texture<float> > bump) {
		Kr = r;
		Kt = t;
        disp = dp;
		index = i;
		bumpMap = bump;
	}
	BSDF *GetBSDF(const DifferentialGeometry &dgGeom, const DifferentialGeometry &dgShading) const;
private:
	// Diamond Private Data
	Reference<Texture<Spectrum> > Kr, Kt;
	Reference<Texture<float> > index;
    Reference<Texture<float> > disp;
	Reference<Texture<float> > bumpMap;
};
// Diamond Method Definitions
BSDF *Diamond::GetBSDF(const DifferentialGeometry &dgGeom, const DifferentialGeometry &dgShading) const {
	// Allocate _BSDF_, possibly doing bump-mapping with _bumpMap_
	DifferentialGeometry dgs;
	if (bumpMap)
		Bump(bumpMap, dgGeom, dgShading, &dgs);
	else
		dgs = dgShading;
	BSDF *bsdf = BSDF_ALLOC(BSDF)(dgs, dgGeom.nn);
	Spectrum R = Kr->Evaluate(dgs).Clamp();
	Spectrum T = Kt->Evaluate(dgs).Clamp();
	float ior = index->Evaluate(dgs);
    float d = disp->Evaluate(dgs);
	if (!R.Black())
		bsdf->Add(BSDF_ALLOC(SpecularReflection)(R,
			BSDF_ALLOC(FresnelDielectric)(1., ior)));
	if (!T.Black())
		bsdf->Add(BSDF_ALLOC(DispersiveTransmission)(T, 1., ior, d));
	return bsdf;
}
extern "C" DLLEXPORT Material * CreateMaterial(const Transform &xform,
		const TextureParams &mp) {
	Reference<Texture<Spectrum> > Kr = mp.GetSpectrumTexture("Kr", Spectrum(1.f));
	Reference<Texture<Spectrum> > Kt = mp.GetSpectrumTexture("Kt", Spectrum(1.f));
	Reference<Texture<float> > index = mp.GetFloatTexture("index", 2.42f);
    Reference<Texture<float> > disp = mp.GetFloatTexture("dp", 0.044f);
	Reference<Texture<float> > bumpMap = mp.GetFloatTexture("bumpmap", 0.f);
	return new Diamond(Kr, Kt, index, disp, bumpMap);
}
