
/*
 * pbrt source code Copyright(c) 1998-2005 Matt Pharr and Greg Humphreys
 *
 * All Rights Reserved.
 * For educational use only; commercial use expressly forbidden.
 * NO WARRANTY, express or implied, for this software.
 * (See file License.txt for complete license)
 */

#define __CODEDEFT__

// photonmap.cpp*
#include "pbrt.h"
#include "transport.h"
#include "scene.h"
#include "mc.h"
#include "kdtree.h"
#include "sampling.h"
// Photonmap Local Declarations
struct Photon;
struct ClosePhoton;
struct PhotonProcess;

#ifdef __CODEDEFT__
class SampleInfo {
public:
	SampleInfo() : Ap(0.f), next(0) {
		Ep = Spectrum(0.f);
	}

	Spectrum Ep;
	float Ap;
	Point pos;
	Vector n;
	SampleInfo* next;
};

class OctreeNode;

// Have an octree per shape
class ModelInfo {
public:
	ModelInfo() : otroot(0), shape(0), next(0) {
	}

	OctreeNode* otroot;
	Shape* shape;
	ModelInfo* next;
};

static float eta;
static float F_dr;

// Model hash table
#define HASH_SIZE 17
ModelInfo* modelHashTable[HASH_SIZE];

class OctreeNode {
public:
	OctreeNode() : Ev(0.f), Av(0.f), parent(0),
		bLeaf(false), numSamples(0), sampleList(0) {
		memset(children, 0, 8*sizeof(OctreeNode*));
	}

	~OctreeNode() {
		// Delete all the children and samples
		if (!bLeaf) {
			for (int i = 0; i < 8; i++) {
				delete children[i];
			}
		} else {
			SampleInfo* cur = sampleList;
			while (cur) {
				SampleInfo* next = cur->next;
				delete cur;
				cur = next;
			}
		}
	}
// These parameters should probably go into pbrt file
#define ETA 1.3f

// Ketchup parameters	
#define SIGMA_S_R 0.219f
#define SIGMA_S_G 0.262f
#define SIGMA_S_B 0.190f

#define SIGMA_A_R 0.00021f
#define SIGMA_A_G 0.00041f
#define SIGMA_A_B 0.00009f

//#define SIGMA_S_R 0.18f
//#define SIGMA_S_G 0.07f
//#define SIGMA_S_B 0.03f

//#define SIGMA_A_R 0.061f
//#define SIGMA_A_G 0.97f
//#define SIGMA_A_B 1.45f

//#define SIGMA_S_R 2.19f
//#define SIGMA_S_G 2.62f
//#define SIGMA_S_B 3.00f

//#define SIGMA_A_R 0.0021f
//#define SIGMA_A_G 0.0041f
//#define SIGMA_A_B 0.0071f
	Spectrum ComputeSampleContribution(Spectrum E, float A, Point P, Point p)
	{
//		eta = ETA; // Let's assume marble's index of refraction
//		F_dr = -1.44f/(eta*eta) + 0.71f/eta + 0.668 + 0.0636*eta;
		float F_dt = 1.f - F_dr;

		float sigma_s_colors[3];
		sigma_s_colors[0] = SIGMA_S_R;
		sigma_s_colors[1] = SIGMA_S_G;
		sigma_s_colors[2] = SIGMA_S_B;
		Spectrum sigma_s(sigma_s_colors);

		float sigma_a_colors[3];
		sigma_a_colors[0] = SIGMA_A_R;
		sigma_a_colors[1] = SIGMA_A_G;
		sigma_a_colors[2] = SIGMA_A_B;
		Spectrum sigma_a(sigma_a_colors);

		Spectrum sigma_t = sigma_s + sigma_a;
		Spectrum sigma_tr = 3.f*sigma_a*sigma_t;
		sigma_tr.Sqrt();

		Spectrum z_r = sigma_t.Pow(Spectrum(-1.f));
		Spectrum z_v = z_r*(1.f + (4.f/3.f)*((1.f+F_dr)/(1.f-F_dr)));
		Spectrum r = Spectrum((P-p).Length());
		Spectrum d_r = (r*r + z_r*z_r).Sqrt();
		Spectrum d_v = (r*r + z_v*z_v).Sqrt();
		Spectrum C_1 = z_r*(sigma_tr + d_r.Pow(-1.f));
		Spectrum C_2 = z_v*(sigma_tr + d_v.Pow(-1.f));
		Spectrum dM_o_term = (INV_PI * 0.25f) * 
			(C_1*Exp(-sigma_tr*d_r)/(d_r*d_r) + C_2*Exp(-sigma_tr*d_v)/(d_v*d_v));
		Spectrum ret = F_dt * dM_o_term * A * E;

		/*
		if (E.y() != 0.f) {
			printf("r : %f\n", r);
			printf("Ey : %f\n", E.y());
			printf("Ratio : %f\n", r*r*E.y());
			E.Print(stdout);
			printf("\n");
		}
		*/
//		if (ret.y() != 0.f) {
//			printf("Sample estimation values\n");
//			ret.Print(stdout);
//			printf("F_dr : %f F_dt : %f\n", F_dr, F_dt);
//			printf("\n");
//		}

		return ret;
	}
#define ANGLE_TOLERANCE (4.f*M_PI/16.f)
//#define ANGLE_TOLERANCE 0.00000001f
	Spectrum ApproximateRadiance(Point pos)
	{
		// Return value
		Spectrum M_o(0.f);

		// Am I a leaf ?
		if (bLeaf) {
			SampleInfo* cur = sampleList;
			while (cur) {
				M_o += this->ComputeSampleContribution(cur->Ep, cur->Ap,
						cur->pos, pos);
				cur = cur->next;
			}
//			printf("Diffusion Leaf Queried\n");
			return M_o;
		}

		// First, test to see if pos is inside
		// this thing.
		if (bb.Inside(pos)) {
			for (int i = 0; i < 8; i++) {
				if (children[i] != 0)
					M_o += children[i]->ApproximateRadiance(pos);
			}
		} else {
			// Let's see if this octrant is enough
			// to be used as a whole
			Vector diff = pos - Pv;
			float distSquared = diff.LengthSquared();
			float delta_w = Av / distSquared;

			if (delta_w < ANGLE_TOLERANCE) {
				M_o = this->ComputeSampleContribution(Ev, Av, Pv, pos);
//				printf("Diffusion Approximated\n");
			} else {
				for (int i = 0; i < 8; i++) {
					if (children[i] != 0)
						M_o += children[i]->ApproximateRadiance(pos);
				}
			}
		}

		return M_o;
	}

	OctreeNode* Create(int numLevels, SampleInfo* samples,
			BBox encloser, OctreeNode* pare) {
		parent = pare;
		bb = encloser;
		if (numLevels == 0) {
			// This is a leaf
			bLeaf = true;
			sampleList = samples;
			SampleInfo* cur = samples;
			while (cur) {
				numSamples++;
				Ev += cur->Ep * cur->Ap;
				Av += cur->Ap;
				Pv += cur->Ep.y() * cur->Ap * cur->pos;
//				Pv += cur->pos;
				cur = cur->next;
			}
			if (Ev.y() > 0.f) {
				Pv /= Ev.y();
			} else {
				Pv = bb.pMin;
			}
			Ev /= Av;
			if (Ev.IsNaN()) {
				printf("Av %f\n", Av);
				Ev.Print(stdout);
				printf("\n");
				Assert(!Ev.IsNaN());
			}
			// Assume uniform samples
//			Pv /= numSamples;
			if (!bb.Inside(Pv)) {
				printf("%f %f %f  %f %f %f  %f %f %f\n",
						Pv.x, Pv.y, Pv.z,
						bb.pMin.x, bb.pMin.y, bb.pMin.z,
						bb.pMax.x, bb.pMax.y, bb.pMax.z);
				printf("The voxel point is not inside !\n");
//				Assert(0);
			}

			return this;
		}
		// Let's divide up the scene into 8 quadrants
		// First in x
		float width		= bb.pMax.x - bb.pMin.x;
		float height 	= bb.pMax.y - bb.pMin.y;
		float thickness	= bb.pMax.z - bb.pMin.z;
		midPoint = Point(bb.pMin.x + width/2.f,
				bb.pMin.y + height/2.f,
				bb.pMin.z + thickness/2.f);
		SampleInfo* octrants[8];
		BBox octrantbbs[8];
		memset(octrants, 0, 8*sizeof(SampleInfo*));
		SampleInfo* cur = samples;

		// Create octrant BBoxes
		octrantbbs[0].pMin = midPoint;
		octrantbbs[0].pMax = bb.pMax;

		octrantbbs[1].pMin.x = bb.pMin.x;
		octrantbbs[1].pMin.y = midPoint.y;
		octrantbbs[1].pMin.z = midPoint.z;
		octrantbbs[1].pMax.x = midPoint.x;
		octrantbbs[1].pMax.y = bb.pMax.y;
		octrantbbs[1].pMax.z = bb.pMax.z;

		octrantbbs[2].pMin.x = bb.pMin.x;
		octrantbbs[2].pMin.y = midPoint.y;
		octrantbbs[2].pMin.z = bb.pMin.z;
		octrantbbs[2].pMax.x = midPoint.x;
		octrantbbs[2].pMax.y = bb.pMax.y;
		octrantbbs[2].pMax.z = midPoint.z;

		octrantbbs[3].pMin.x = midPoint.x;
		octrantbbs[3].pMin.y = midPoint.y;
		octrantbbs[3].pMin.z = bb.pMin.z;
		octrantbbs[3].pMax.x = bb.pMax.x;
		octrantbbs[3].pMax.y = bb.pMax.y;
		octrantbbs[3].pMax.z = midPoint.z;

		octrantbbs[4].pMin.x = midPoint.x;
		octrantbbs[4].pMin.y = bb.pMin.y;
		octrantbbs[4].pMin.z = midPoint.z;
		octrantbbs[4].pMax.x = bb.pMax.x;
		octrantbbs[4].pMax.y = midPoint.y;
		octrantbbs[4].pMax.z = bb.pMax.z;

		octrantbbs[5].pMin.x = bb.pMin.x;
		octrantbbs[5].pMin.y = bb.pMin.y;
		octrantbbs[5].pMin.z = midPoint.z;
		octrantbbs[5].pMax.x = midPoint.x;
		octrantbbs[5].pMax.y = midPoint.y;
		octrantbbs[5].pMax.z = bb.pMax.z;

		octrantbbs[6].pMin = bb.pMin;
		octrantbbs[6].pMax = midPoint;

		octrantbbs[7].pMin.x = midPoint.x;
		octrantbbs[7].pMin.y = bb.pMin.y;
		octrantbbs[7].pMin.z = bb.pMin.z;
		octrantbbs[7].pMax.x = bb.pMax.x;
		octrantbbs[7].pMax.y = midPoint.y;
		octrantbbs[7].pMax.z = midPoint.z;

		// Compute the overall area, average location, total irradiance
		// while dividing'em into octrants
		while (cur) {
			// Compute its octrant
			int octrant = this->FindOctrant(cur->pos);
			Ev += cur->Ep * cur->Ap;
			Av += cur->Ap;
			Pv += cur->Ep.y() * cur->Ap * cur->pos;
//			Pv += cur->pos;
			SampleInfo* next = cur->next;
			cur->next = octrants[octrant];
			octrants[octrant] = cur;
			cur = next;
			numSamples++;
		}

		if (Ev.y() > 0.f) {
			Pv /= Ev.y();
		} else {
			Pv = bb.pMin;
		}

		Ev /= Av;
		if (Ev.IsNaN()) {
			printf("Av %f\n", Av);
			Ev.Print(stdout);
			printf("\n");
			Assert(!Ev.IsNaN());
		}
//		printf("Pv : %f %f %f\n", Pv.x, Pv.y, Pv.z);
//		Pv /= numSamples;
//		printf("y : %f\n", Ev.y());
		if (!bb.Inside(Pv)) {
			printf("%f %f %f  %f %f %f  %f %f %f\n",
					Pv.x, Pv.y, Pv.z,
					bb.pMax.x, bb.pMax.y, bb.pMax.z,
					bb.pMin.x, bb.pMin.y, bb.pMin.z);
			printf("The voxel point is not inside !\n");
//			Assert(0);
		}

		if (Ev.y() == 0) {
			bLeaf = true;
			return this;
		}

		for (int i = 0; i < 8; i++) {
			if (octrants[i] != 0) {
				children[i] = new OctreeNode();
				children[i]->Create(numLevels-1, octrants[i], octrantbbs[i], this);
			}
		}

		return this;
	}

	int FindOctrant(const Point& pos) const
	{
		float xval = pos.x - midPoint.x;
		float yval = pos.y - midPoint.y;
		float zval = pos.z - midPoint.z;

		// 8 possibilities
		// Look at the ordering
		if (xval >= 0.f && yval >= 0.f && zval >= 0.f) {
			return 0;
		}

		if (xval <= 0.f && yval >= 0.f && zval >= 0.f) {
			return 1;
		}

		if (xval <= 0.f && yval >= 0.f && zval <= 0.f) {
			return 2;
		}

		if (xval >= 0.f && yval >= 0.f && zval <= 0.f) {
			return 3;
		}

		if (xval >= 0.f && yval <= 0.f && zval >= 0.f) {
			return 4;
		}

		if (xval <= 0.f && yval <= 0.f && zval >= 0.f) {
			return 5;
		}

		if (xval <= 0.f && yval <= 0.f && zval <= 0.f) {
			return 6;
		}

		if (xval >= 0.f && yval <= 0.f && zval <= 0.f) {
			return 7;
		}
		printf("%f %f %f   %f %f %f\n", pos.x, pos.y, pos.z, midPoint.x, midPoint.y, midPoint.z);
		Assert(0);
		return -1;
	}

	Spectrum	Ev;
	float		Av;
	Point		Pv;

	Point		midPoint;
	BBox		bb;
	OctreeNode* parent;
	OctreeNode* children[8];
	bool		bLeaf;

	// If this is a leaf
	int			numSamples;
	SampleInfo* sampleList;
};
#endif

class PhotonIntegrator : public SurfaceIntegrator {
public:
	// PhotonIntegrator Public Methods
	PhotonIntegrator(int ncaus, int ndir, int nindir, int nLookup, int mdepth,
		float maxdist, bool finalGather, int gatherSamples,
		bool directWithPhotons);
	~PhotonIntegrator();
	Spectrum Li(const Scene *scene, const RayDifferential &ray,
		const Sample *sample, float *alpha) const;
	void RequestSamples(Sample *sample, const Scene *scene);
	void Preprocess(const Scene *);
private:
	// PhotonIntegrator Private Methods
	static inline bool unsuccessful(int needed, int found, int shot) {
		return (found < needed &&
		        (found == 0 || found < shot / 1024));
	}
	static Spectrum LPhoton(KdTree<Photon, PhotonProcess> *map,
		int nPaths, int nLookup, BSDF *bsdf, const Intersection &isect,
		const Vector &w, float maxDistSquared);
#ifdef __CODEDEFT__
	void ComputeIrradiances(SampleInfo* sampleList);
	Spectrum IrradiancePhoton(KdTree<Photon, PhotonProcess> *map, int nPaths,
			Point p, int nLookup, float maxDistSquared, Vector n);
#endif
	// PhotonIntegrator Private Data
	u_int nCausticPhotons, nIndirectPhotons, nDirectPhotons;
	u_int nLookup;
	mutable int specularDepth;
	int maxSpecularDepth;
	float maxDistSquared;
	bool directWithPhotons, finalGather;
	int gatherSamples;
	// Declare sample parameters for light source sampling
	int *lightSampleOffset, lightNumOffset;
	int *bsdfSampleOffset, *bsdfComponentOffset;
	int gatherSampleOffset, gatherComponentOffset;
	int nCausticPaths, nDirectPaths, nIndirectPaths;
	mutable KdTree<Photon, PhotonProcess> *causticMap;
	mutable KdTree<Photon, PhotonProcess> *directMap;
	mutable KdTree<Photon, PhotonProcess> *indirectMap;
};
struct Photon {
	// Photon Constructor
	Photon(const Point &pp, const Spectrum &wt, const Vector &w)
		: p(pp), alpha(wt), wi(w) {
	}
	Photon() { }
	Point p;
	Spectrum alpha;
	Vector wi;
};
struct PhotonProcess {
	// PhotonProcess Public Methods
	PhotonProcess(u_int mp, const Point &p);
	void operator()(const Photon &photon, float dist2, float &maxDistSquared) const;
	const Point &p;
	ClosePhoton *photons;
	u_int nLookup;
	mutable u_int foundPhotons;
};
struct ClosePhoton {
	ClosePhoton(const Photon *p = NULL,
	            float md2 = INFINITY) {
		photon = p;
		distanceSquared = md2;
	}
	bool operator<(const ClosePhoton &p2) const {
		return distanceSquared == p2.distanceSquared ? (photon < p2.photon) :
			distanceSquared < p2.distanceSquared;
	}
	const Photon *photon;
	float distanceSquared;
};
// Photonmap Method Definitions
PhotonIntegrator::PhotonIntegrator(int ncaus, int ndir, int nind,
		int nl,	int mdepth, float mdist, bool fg,
		int gs, bool dp) {
	nCausticPhotons = ncaus;
	nIndirectPhotons = nind;
	nDirectPhotons = ndir;
	nLookup = nl;
	maxDistSquared = mdist * mdist;
	maxSpecularDepth = mdepth;
	causticMap = directMap = indirectMap = NULL;
	specularDepth = 0;
	finalGather = fg;
	gatherSamples = gs;
	directWithPhotons = dp;
}
PhotonIntegrator::~PhotonIntegrator() {
	delete causticMap;
	delete directMap;
	delete indirectMap;
}
void PhotonIntegrator::RequestSamples(Sample *sample,
		const Scene *scene) {
	// Allocate and request samples for sampling all lights
	u_int nLights = scene->lights.size();
	lightSampleOffset = new int[nLights];
	bsdfSampleOffset = new int[nLights];
	bsdfComponentOffset = new int[nLights];
	for (u_int i = 0; i < nLights; ++i) {
		const Light *light = scene->lights[i];
		int lightSamples =
			scene->sampler->RoundSize(light->nSamples);
		lightSampleOffset[i] = sample->Add2D(lightSamples);
		bsdfSampleOffset[i] = sample->Add2D(lightSamples);
		bsdfComponentOffset[i] = sample->Add1D(lightSamples);
	}
	lightNumOffset = -1;
	if (finalGather) {
		gatherSamples = scene->sampler->RoundSize(gatherSamples);
		gatherSampleOffset = sample->Add2D(gatherSamples);
		gatherComponentOffset = sample->Add1D(gatherSamples);
	}
}
void PhotonIntegrator::Preprocess(const Scene *scene) {
	if (scene->lights.size() == 0) return;
	ProgressReporter progress(nCausticPhotons+nDirectPhotons+ // NOBOOK
		nIndirectPhotons, "Shooting photons"); // NOBOOK
	vector<Photon> causticPhotons;
	vector<Photon> directPhotons;
	vector<Photon> indirectPhotons;
	causticPhotons.reserve(nCausticPhotons); // NOBOOK
	directPhotons.reserve(nDirectPhotons); // NOBOOK
	indirectPhotons.reserve(nIndirectPhotons); // NOBOOK
	// Initialize photon shooting statistics
	static StatsCounter nshot("Photon Map",
		"Number of photons shot from lights");
	bool causticDone = (nCausticPhotons == 0);
	bool directDone = (nDirectPhotons == 0);
	bool indirectDone = (nIndirectPhotons == 0);
	while (!causticDone || !directDone || !indirectDone) {
		++nshot;
		// Give up if we're not storing enough photons
		if (nshot > 500000 &&
			(unsuccessful(nCausticPhotons,
			              causticPhotons.size(),
						  nshot) ||
			 unsuccessful(nDirectPhotons,
			              directPhotons.size(),
						  nshot) ||
			 unsuccessful(nIndirectPhotons,
			              indirectPhotons.size(),
						  nshot))) {
			Error("Unable to store enough photons.  Giving up.\n");
			return;
		}
		// Trace a photon path and store contribution
		// Choose 4D sample values for photon
		float u[4];
		u[0] = (float)RadicalInverse((int)nshot+1, 2);
		u[1] = (float)RadicalInverse((int)nshot+1, 3);
		u[2] = (float)RadicalInverse((int)nshot+1, 5);
		u[3] = (float)RadicalInverse((int)nshot+1, 7);
		// Choose light to shoot photon from
		int nLights = int(scene->lights.size());
		int lightNum =
			min(Floor2Int(nLights * (float)RadicalInverse((int)nshot+1, 11)),
			nLights-1);
		Light *light = scene->lights[lightNum];
		float lightPdf = 1.f / nLights;
		// Generate _photonRay_ from light source and initialize _alpha_
		RayDifferential photonRay;
		float pdf;
		Spectrum alpha =
			light->Sample_L(scene, u[0], u[1], u[2], u[3],
				&photonRay, &pdf);
		if (pdf == 0.f || alpha.Black()) continue;
		alpha /= pdf * lightPdf;
		if (!alpha.Black()) {
			// Follow photon path through scene and record intersections
			bool specularPath = false;
			Intersection photonIsect;
			int nIntersections = 0;
			while (scene->Intersect(photonRay, &photonIsect)) {
				++nIntersections;
				// Handle photon/surface intersection
				alpha *= scene->Transmittance(photonRay);
				Vector wo = -photonRay.d;
				BSDF *photonBSDF = photonIsect.GetBSDF(photonRay);
				BxDFType specularType = BxDFType(BSDF_REFLECTION |
					BSDF_TRANSMISSION | BSDF_SPECULAR);
				bool hasNonSpecular = (photonBSDF->NumComponents() >
					photonBSDF->NumComponents(specularType));
				if (hasNonSpecular) {
					// Deposit photon at surface
					Photon photon(photonIsect.dg.p, alpha, wo);
					if (nIntersections == 1) {
						// Process direct lighting photon intersection
						if (!directDone) {
							directPhotons.push_back(photon);
							if (directPhotons.size() == nDirectPhotons) {
								directDone = true;
								nDirectPaths = (int)nshot;
								directMap =
									new KdTree<Photon,
											   PhotonProcess>(directPhotons);
							}
							progress.Update(); // NOBOOK
						}
					}
					else if (specularPath) {
						// Process caustic photon intersection
						if (!causticDone) {
							causticPhotons.push_back(photon);
							if (causticPhotons.size() == nCausticPhotons) {
								causticDone = true;
								nCausticPaths = (int)nshot;
								causticMap =
									new KdTree<Photon,
										       PhotonProcess>(causticPhotons);
							}
							progress.Update();
						}
					}
					else {
						// Process indirect lighting photon intersection
						if (!indirectDone) {
							indirectPhotons.push_back(photon);
							if (indirectPhotons.size() == nIndirectPhotons) {
								indirectDone = true;
								nIndirectPaths = (int)nshot;
								indirectMap =
									new KdTree<Photon,
											   PhotonProcess>(indirectPhotons);
							}
							progress.Update();
						}
					}
				}
				// Sample new photon ray direction
				Vector wi;
				float pdf;
				BxDFType flags;
				// Get random numbers for sampling outgoing photon direction
				float u1, u2, u3;
				if (nIntersections == 1) {
					u1 = (float)RadicalInverse((int)nshot+1, 13);
					u2 = (float)RadicalInverse((int)nshot+1, 17);
					u3 = (float)RadicalInverse((int)nshot+1, 19);
				}
				else {
					u1 = RandomFloat();
					u2 = RandomFloat();
					u3 = RandomFloat();
				}
				Spectrum fr = photonBSDF->Sample_f(wo, &wi, u1, u2, u3,
					&pdf, BSDF_ALL, &flags);
				if (fr.Black() || pdf == 0.f)
					break;
				specularPath = (nIntersections == 1 || specularPath) &&
					((flags & BSDF_SPECULAR) != 0);
				alpha *= fr * AbsDot(wi, photonBSDF->dgShading.nn) / pdf;
				photonRay = RayDifferential(photonIsect.dg.p, wi);
				// Possibly terminate photon path
				if (nIntersections > 3) {
					float continueProbability = .5f;
					if (RandomFloat() > continueProbability)
						break;
					alpha /= continueProbability;
				}
			}
		}
		BSDF::FreeAll();
	}
	progress.Done(); // NOBOOK
#ifdef __CODEDEFT__
	printf("Starting irradiance computations\n");
	eta = ETA; // Let's assume marble's index of refraction
	F_dr = -1.44f/(eta*eta) + 0.71f/eta + 0.668 + 0.0636*eta;
	for (u_int i = 0; i < scene->shapes.size(); i++) {
		if (scene->shapes[i]->HasSamplePoints()) {
			vector<Point> samplePositions;
			vector<Normal> sampleNormals;
			vector<float> sampleAreas;
			scene->shapes[i]->GetSamplePositions(&samplePositions);
			scene->shapes[i]->GetSampleNormals(&sampleNormals);
			scene->shapes[i]->GetSampleAreas(&sampleAreas);
//			float totalArea = scene->shapes[i]->GetTotalArea();
			float numPoints = (float) samplePositions.size();
			vector<Point>::const_iterator cit = samplePositions.begin();
			vector<Normal>::const_iterator nit = sampleNormals.begin();
			vector<float>::const_iterator ait = sampleAreas.begin();
			SampleInfo *sampleHead = 0;
			SampleInfo *cur = 0;
			printf("numPoints : %d\n", (int)numPoints);
//			BBox wb = scene->shapes[i]->WorldBound();
//			printf("%f %f %f   %f %f %f\n", wb.pMin.x, wb.pMin.y, wb.pMin.z,
//					wb.pMax.x, wb.pMax.y, wb.pMax.z);
			BBox objectbb = scene->shapes[i]->ObjectBound();
			BBox worldbb = scene->shapes[i]->WorldBound();
			float obbx = objectbb.pMax.x - objectbb.pMin.x;
			float wbbx = worldbb.pMax.x - worldbb.pMin.x;
			float arearatio = (wbbx/obbx)*(wbbx/obbx);
			printf("%f\n", arearatio);
			while (cit != samplePositions.end()) {
				cur = new SampleInfo();
				cur->next = sampleHead;
				sampleHead = cur;
				cur->pos = *cit;
				cur->pos = scene->shapes[i]->ObjectToWorld(cur->pos);
				Normal worldN = scene->shapes[i]->ObjectToWorld(*nit);
				cur->n = Vector(worldN);
				// Already in world space
//				cur->pos = scene->shapes[i]->ObjectToWorld(cur->pos);
//				printf("%f %f %f\n", cur->pos.x, cur->pos.y, cur->pos.z);
				cur->Ap = *ait * arearatio;
				cit++;
				nit++;
				ait++;
			}

			// Compute irradiance at each sample point
			this->ComputeIrradiances(sampleHead);

			// Create a model
			ModelInfo* newModel = new ModelInfo();
			newModel->shape = scene->shapes[i].ptr;
			newModel->otroot = new OctreeNode();
			int numLevels = (int)ceil((logf(numPoints/8.f)/logf(8.f)));
			printf("Number of octtree levels %d %d\n", (int)numPoints, numLevels);
			if (numLevels < 0) numLevels = 0;
			newModel->otroot->Create( numLevels,
					sampleHead, newModel->shape->WorldBound(), 0 );
			int key = ((int)newModel->shape) % HASH_SIZE;
			newModel->next = modelHashTable[key];
			modelHashTable[key] = newModel;
		}
	}
	printf("Done with irradiance computations\n");
#endif
}
#ifdef __CODEDEFT__
void PhotonIntegrator::ComputeIrradiances(SampleInfo* sampleList)
{
	SampleInfo* cur = sampleList;
	while (cur) {
		if (directMap)
			cur->Ep += this->IrradiancePhoton(directMap, nDirectPaths,
					cur->pos, nLookup, maxDistSquared, cur->n);
		if (causticMap)
			cur->Ep += this->IrradiancePhoton(causticMap, nCausticPaths,
					cur->pos, nLookup, maxDistSquared, cur->n);
		if (indirectMap)
			cur->Ep += this->IrradiancePhoton(indirectMap, nIndirectPaths,
					cur->pos, nLookup, maxDistSquared, cur->n);
		/*
		static bool bFalseZero = false;
		if (cur->Ep.y() != 0) {
			bFalseZero = true;
		}
		if (bFalseZero) {
			printf("Irradiance !\n");
			cur->Ep.Print(stdout);
			printf("\n");
		}
		*/
		cur = cur->next;
	}
}
Spectrum PhotonIntegrator::IrradiancePhoton(KdTree<Photon, PhotonProcess> *map,
		int nPaths, Point p, int nLookup, float maxDistSquared, Vector n)
{
	PhotonProcess proc(nLookup, p);
	proc.photons = (ClosePhoton *) alloca(nLookup * sizeof(ClosePhoton));
	float md2 = maxDistSquared;
	map->Lookup(p, proc, md2);
	float scale = 1.f / (float(nPaths) * md2 * M_PI);
	ClosePhoton *photons = proc.photons;
	int nFound = proc.foundPhotons;
	Spectrum r(0.f);
//	printf("%f %f %f\n", p.x, p.y, p.z);
	if (nFound > 0) {
//		printf("maxDistSquared, md2 %f %f\n", maxDistSquared, md2);
//		printf("Irradiance Storage:\n");
//		printf("%d\n", nFound);
//		printf("%f %f %f\n", p.x, p.y, p.z);
//		printf("%f\n", maxDistSquared);
	}
	for (int i = 0; i < nFound; i++) {
		if (Dot(n, photons[i].photon->wi) > 0.f)
			r += photons[i].photon->alpha;
	}
//	r.Print(stdout);
//	printf("\n");

	return scale * r;
}
#endif
Spectrum PhotonIntegrator::Li(const Scene *scene,
		const RayDifferential &ray, const Sample *sample,
		float *alpha) const {
	// Compute reflected radiance with photon map
	Spectrum L(0.);
	Intersection isect;
	if (scene->Intersect(ray, &isect)) {
#ifdef __CODEDEFT__
		if (isect.dg.shape->IsTranslucent()) {
			// This object is translucent, so compute
			// using the Jensen's technique
			Spectrum M_o(0.f);
			// Find the model
			int key = ((int)isect.dg.shape->GetParent()) % HASH_SIZE;
			ModelInfo* cur = modelHashTable[key];
			while (cur != 0) {
				if (cur->shape == isect.dg.shape->GetParent()) {
					break;
				}
				cur = cur->next;
			}
			if (cur == 0) {
				Assert(0);
			}
			M_o = cur->otroot->ApproximateRadiance(isect.dg.p);
//			M_o.Print(stdout);
//			printf("\n");
			// Assume diffuse radiance for now
			M_o = M_o * INV_PI;
			if (alpha) *alpha = 1.;

			/*
			// In addition, add specular reflection and etc.
			FresnelDielectric fresnel(1.f, 1.3f);
			Spectrum F_t = Spectrum(1.f) - fresnel.Evaluate(Dot(Normalize(isect.dg.nn), Normalize(-ray.d)));
//			M_o.Print(stdout);
//			printf("\n");
			return F_t * M_o / F_dr;
			*/
			// Compute ray differential _rd_ for specular reflection
			Vector wo = -ray.d;
			Vector wi;
			BSDF *bsdf = isect.GetBSDF(ray);
			const Point &p = bsdf->dgShading.p;
			const Normal &n = bsdf->dgShading.nn;
//			Spectrum f = bsdf->Sample_f(wo, &wi,
//				BxDFType(BSDF_REFLECTION | BSDF_DIFFUSE));

//			if (!f.Black()) {
//				printf("I got glossy!\n");
//			}

			M_o += 1.f * UniformSampleAllLights(scene, p, n,
					wo, bsdf, sample, lightSampleOffset, bsdfSampleOffset,
					bsdfComponentOffset);

			/*
			RayDifferential rd(p, wi);
			rd.hasDifferentials = true;
			rd.rx.o = p + isect.dg.dpdx;
			rd.ry.o = p + isect.dg.dpdy;
			// Compute differential reflected directions
			Normal dndx = bsdf->dgShading.dndu * bsdf->dgShading.dudx +
				bsdf->dgShading.dndv * bsdf->dgShading.dvdx;
			Normal dndy = bsdf->dgShading.dndu * bsdf->dgShading.dudy +
				bsdf->dgShading.dndv * bsdf->dgShading.dvdy;
			Vector dwodx = -ray.rx.d - wo, dwody = -ray.ry.d - wo;
			float dDNdx = Dot(dwodx, n) + Dot(wo, dndx);
			float dDNdy = Dot(dwody, n) + Dot(wo, dndy);
			rd.rx.d = wi -
			          dwodx + 2 * Vector(Dot(wo, n) * dndx +
					  dDNdx * n);
			rd.ry.d = wi -
			          dwody + 2 * Vector(Dot(wo, n) * dndy +
					  dDNdy * n);
			M_o += scene->Li(rd, sample) * f * AbsDot(wi, n);
			*/

			return M_o;
		} else {
#endif
		if (alpha) *alpha = 1.;
		Vector wo = -ray.d;
		// Compute emitted light if ray hit an area light source
		L += isect.Le(wo);
		// Evaluate BSDF at hit point
		BSDF *bsdf = isect.GetBSDF(ray);
		const Point &p = bsdf->dgShading.p;
		const Normal &n = bsdf->dgShading.nn;
		// Compute direct lighting for photon map integrator
		if (directWithPhotons)
			L += LPhoton(directMap, nDirectPaths, nLookup,
				bsdf, isect, wo, maxDistSquared);
		else
			L += UniformSampleAllLights(scene, p, n,
				wo, bsdf, sample,
				lightSampleOffset, bsdfSampleOffset,
				bsdfComponentOffset);
		
		// Compute indirect lighting for photon map integrator
		L += LPhoton(causticMap, nCausticPaths, nLookup, bsdf,
			isect, wo, maxDistSquared);
		if (finalGather) {
			// Do one-bounce final gather for photon map
			Spectrum Li(0.);
			for (int i = 0; i < gatherSamples; ++i) {
				// Sample random direction for final gather ray
				Vector wi;
				float u1 = sample->twoD[gatherSampleOffset][2*i];
				float u2 = sample->twoD[gatherSampleOffset][2*i+1];
				float u3 = sample->oneD[gatherComponentOffset][i];
				float pdf;
				Spectrum fr = bsdf->Sample_f(wo, &wi, u1, u2, u3,
					&pdf, BxDFType(BSDF_ALL & (~BSDF_SPECULAR)));
				if (fr.Black() || pdf == 0.f) continue;
				RayDifferential bounceRay(p, wi);
				static StatsCounter gatherRays("Photon Map", // NOBOOK
					"Final gather rays traced"); // NOBOOK
				++gatherRays; // NOBOOK
				Intersection gatherIsect;
				if (scene->Intersect(bounceRay, &gatherIsect)) {
					// Compute exitant radiance at final gather intersection
					BSDF *gatherBSDF = gatherIsect.GetBSDF(bounceRay);
					Vector bounceWo = -bounceRay.d;
					Spectrum Lindir =
						LPhoton(directMap, nDirectPaths, nLookup,
							gatherBSDF, gatherIsect, bounceWo, maxDistSquared) +
						LPhoton(indirectMap, nIndirectPaths, nLookup,
							gatherBSDF, gatherIsect, bounceWo, maxDistSquared) +
						LPhoton(causticMap, nCausticPaths, nLookup,
							gatherBSDF, gatherIsect, bounceWo, maxDistSquared);
					Lindir *= scene->Transmittance(bounceRay);
					Li += fr * Lindir * AbsDot(wi, n) / pdf;
				}
			}
			L += Li / float(gatherSamples);
		}
		else
			L += LPhoton(indirectMap, nIndirectPaths, nLookup,
				bsdf, isect, wo, maxDistSquared);
		if (specularDepth++ < maxSpecularDepth) {
			Vector wi;
			// Trace rays for specular reflection and refraction
			Spectrum f = bsdf->Sample_f(wo, &wi,
				BxDFType(BSDF_REFLECTION | BSDF_SPECULAR));
			if (!f.Black()) {
				// Compute ray differential _rd_ for specular reflection
				RayDifferential rd(p, wi);
				rd.hasDifferentials = true;
				rd.rx.o = p + isect.dg.dpdx;
				rd.ry.o = p + isect.dg.dpdy;
				// Compute differential reflected directions
				Normal dndx = bsdf->dgShading.dndu * bsdf->dgShading.dudx +
					bsdf->dgShading.dndv * bsdf->dgShading.dvdx;
				Normal dndy = bsdf->dgShading.dndu * bsdf->dgShading.dudy +
					bsdf->dgShading.dndv * bsdf->dgShading.dvdy;
				Vector dwodx = -ray.rx.d - wo, dwody = -ray.ry.d - wo;
				float dDNdx = Dot(dwodx, n) + Dot(wo, dndx);
				float dDNdy = Dot(dwody, n) + Dot(wo, dndy);
				rd.rx.d = wi -
				          dwodx + 2 * Vector(Dot(wo, n) * dndx +
						  dDNdx * n);
				rd.ry.d = wi -
				          dwody + 2 * Vector(Dot(wo, n) * dndy +
						  dDNdy * n);
				L += scene->Li(rd, sample) * f * AbsDot(wi, n);
			}
			f = bsdf->Sample_f(wo, &wi,
				BxDFType(BSDF_TRANSMISSION | BSDF_SPECULAR));
			if (!f.Black()) {
				// Compute ray differential _rd_ for specular transmission
				RayDifferential rd(p, wi);
				rd.hasDifferentials = true;
				rd.rx.o = p + isect.dg.dpdx;
				rd.ry.o = p + isect.dg.dpdy;
				
				float eta = bsdf->eta;
				Vector w = -wo;
				if (Dot(wo, n) < 0) eta = 1.f / eta;
				
				Normal dndx = bsdf->dgShading.dndu * bsdf->dgShading.dudx + bsdf->dgShading.dndv * bsdf->dgShading.dvdx;
				Normal dndy = bsdf->dgShading.dndu * bsdf->dgShading.dudy + bsdf->dgShading.dndv * bsdf->dgShading.dvdy;
				
				Vector dwodx = -ray.rx.d - wo, dwody = -ray.ry.d - wo;
				float dDNdx = Dot(dwodx, n) + Dot(wo, dndx);
				float dDNdy = Dot(dwody, n) + Dot(wo, dndy);
				
				float mu = eta * Dot(w, n) - Dot(wi, n);
				float dmudx = (eta - (eta*eta*Dot(w,n))/Dot(wi, n)) * dDNdx;
				float dmudy = (eta - (eta*eta*Dot(w,n))/Dot(wi, n)) * dDNdy;
				
				rd.rx.d = wi + eta * dwodx - Vector(mu * dndx + dmudx * n);
				rd.ry.d = wi + eta * dwody - Vector(mu * dndy + dmudy * n);
				L += scene->Li(rd, sample) * f * AbsDot(wi, n);
			}
		}
		--specularDepth;
#ifdef __CODEDEFT__
		}
#endif
	}
	else {
		// Handle ray with no intersection
		if (alpha) *alpha = 0.;
		for (u_int i = 0; i < scene->lights.size(); ++i)
			L += scene->lights[i]->Le(ray);
		if (alpha && !L.Black()) *alpha = 1.;
		return L;
	}
#ifdef __CODEDEFT__
//	L.Print(stdout);
//	printf("\n");
#endif
	return L;
}
Spectrum PhotonIntegrator::LPhoton(
		KdTree<Photon, PhotonProcess> *map,
		int nPaths, int nLookup, BSDF *bsdf,
		const Intersection &isect, const Vector &wo,
		float maxDistSquared) {
	Spectrum L(0.);
	if (!map) return L;
	BxDFType nonSpecular = BxDFType(BSDF_REFLECTION |
		BSDF_TRANSMISSION | BSDF_DIFFUSE | BSDF_GLOSSY);
	if (bsdf->NumComponents(nonSpecular) == 0)
		return L;
	static StatsCounter lookups("Photon Map", "Total lookups"); // NOBOOK
	// Initialize _PhotonProcess_ object, _proc_, for photon map lookups
	PhotonProcess proc(nLookup, isect.dg.p);
	proc.photons =
		(ClosePhoton *)alloca(nLookup * sizeof(ClosePhoton));
	// Do photon map lookup
	++lookups;  // NOBOOK
	map->Lookup(isect.dg.p, proc, maxDistSquared);
	// Accumulate light from nearby photons
	static StatsRatio foundRate("Photon Map", "Photons found per lookup"); // NOBOOK
	foundRate.Add(proc.foundPhotons, 1); // NOBOOK
	float scale = 1.f / (float(nPaths) * maxDistSquared * M_PI);
	// Estimate reflected light from photons
	ClosePhoton *photons = proc.photons;
	int nFound = proc.foundPhotons;
#ifdef __CODEDEFT__
//	printf("LPhoton : %d %f\n", nFound, maxDistSquared);
#endif
	Normal Nf = Dot(wo, bsdf->dgShading.nn) < 0 ? -bsdf->dgShading.nn :
		bsdf->dgShading.nn;
	if (bsdf->NumComponents(BxDFType(BSDF_REFLECTION |
			BSDF_TRANSMISSION | BSDF_GLOSSY)) > 0) {
		// Compute exitant radiance from photons for glossy surface
		for (int i = 0; i < nFound; ++i) {
			BxDFType flag = Dot(Nf, photons[i].photon->wi) > 0.f ?
				BSDF_ALL_REFLECTION : BSDF_ALL_TRANSMISSION;
			L += bsdf->f(wo, photons[i].photon->wi,	flag) *
				(scale * photons[i].photon->alpha);
		}
	}
	else {
		// Compute exitant radiance from photons for diffuse surface
		Spectrum Lr(0.), Lt(0.);
		for (int i = 0; i < nFound; ++i)
			if (Dot(Nf, photons[i].photon->wi) > 0.f)
				Lr += photons[i].photon->alpha;
			else
				Lt += photons[i].photon->alpha;
		L += (scale * INV_PI) * (Lr * bsdf->rho(wo, BSDF_ALL_REFLECTION) +
			Lt * bsdf->rho(wo, BSDF_ALL_TRANSMISSION));
	}
	return L;
}
PhotonProcess::PhotonProcess(u_int mp, const Point &P)
	: p(P) {
	photons = 0;
	nLookup = mp;
	foundPhotons = 0;
}
void PhotonProcess::operator()(const Photon &photon,
		float distSquared, float &maxDistSquared) const {
	static StatsPercentage discarded("Photon Map", "Discarded photons"); // NOBOOK
	discarded.Add(0, 1); // NOBOOK
	if (foundPhotons < nLookup) {
		// Add photon to unordered array of photons
		photons[foundPhotons++] = ClosePhoton(&photon, distSquared);
		if (foundPhotons == nLookup) {
			std::make_heap(&photons[0], &photons[nLookup]);
			maxDistSquared = photons[0].distanceSquared;
		}
	}
	else {
		// Remove most distant photon from heap and add new photon
		discarded.Add(1, 0); // NOBOOK
		std::pop_heap(&photons[0], &photons[nLookup]);
		photons[nLookup-1] = ClosePhoton(&photon, distSquared);
		std::push_heap(&photons[0], &photons[nLookup]);
		maxDistSquared = photons[0].distanceSquared;
	}
}
extern "C" DLLEXPORT SurfaceIntegrator *CreateSurfaceIntegrator(const ParamSet &params) {
	int nCaustic = params.FindOneInt("causticphotons", 20000);
	int nDirect = params.FindOneInt("directphotons", 100000);
	int nIndirect = params.FindOneInt("indirectphotons", 100000);
	int nUsed = params.FindOneInt("nused", 50);
	int maxDepth = params.FindOneInt("maxdepth", 5);
	bool finalGather = params.FindOneBool("finalgather", true);
	bool directPhotons = params.FindOneBool("directwithphotons", false);
	int gatherSamples = params.FindOneInt("finalgathersamples", 32);
	float maxDist = params.FindOneFloat("maxdist", .1f);
	return new PhotonIntegrator(nCaustic, nDirect, nIndirect,
		nUsed, maxDepth, maxDist, finalGather, gatherSamples,
		directPhotons);
}
